// This software is provided "AS IS" with no warranties of any kind.
// The entire risk arising out of the use or performance of the software
// and source code is with you.

using System;
using System.Data;
using System.Configuration;
using System.Web;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using System.Web.UI.HtmlControls;
using System.Web.Management;
using System.Configuration.Provider;
using System.Collections.Specialized;

using log4net;
using log4net.Config;
using log4net.Core;
using log4net.ObjectRenderer;
using log4net.Util;

namespace HealthMonitoring
{
	/// <summary>
	/// Custom provider that will map Web Events to a log4net log.
	/// </summary>
	/// <remarks>
	/// Using this provider allows any current log4net infrastructure to still be utilized
	/// while adding the ability to capture all the web events that are generated internally in
	/// .NET.
	/// 
	/// Any event that does not implement the ILoggerEvent interface will be logged at the default 
	/// level and to the default log.  The exception to this rule is any web event that extends 
	/// WebErrorEvent.  These events are automatically logged at ERROR level.
	/// 
	/// A possible extension would be to allow rules to be created to define how and where non-ILoggerEvents
	/// are logged to allow more flexibility.
	/// </remarks>
	public class Log4NetBufferedWebEventProvider : BufferedWebEventProvider
	{
		private static Type _thisDeclaringType;

		private string _logName;
		private log4net.Core.Level _logLevel;

		/// <summary>
		/// Initializes the <see cref="Log4NetBufferedWebEventProvider"/> class.
		/// </summary>
		static Log4NetBufferedWebEventProvider()
		{
			_thisDeclaringType = typeof(Log4NetBufferedWebEventProvider);
		}

		/// <summary>
		/// Sets the initial values for this object.
		/// </summary>
		/// <param name="name">The name used in the configuration file to identify this provider.</param>
		/// <param name="config">A <see cref="T:System.Collections.Specialized.NameValueCollection"></see> that specifies the attributes assigned for this provider in the configuration file.</param>
		public override void Initialize(string name, NameValueCollection config)
		{
			if (config == null)
			{
				throw new ArgumentNullException("config");
			}

			// Assign a default name of not specified
			if (string.IsNullOrEmpty(name))
			{
				name = "Log4NetBufferedWebEventProvider";
			}

			// Assign a description if one was not specified
			if (string.IsNullOrEmpty(config["description"]))
			{
				config.Remove("description");
				config.Add("description", "A buffered appender that writes the events to the specified log4net log");
			}

			// Get the default logger name
			_logName = config["loggerName"];
			if (string.IsNullOrEmpty("loggerName"))
			{
				throw new ProviderException("A default logger name must be provided");
			}
			config.Remove("loggerName");

			// Get the default level
			string level = config["level"];
			if (string.IsNullOrEmpty(level))
			{
				level = "INFO";
			}
			else
			{
				level = level.ToUpperInvariant();
			}

			_logLevel = LogManager.GetRepository().LevelMap[level];
			
			if (_logLevel == null)
			{
				throw new ProviderException("An invalid level was specified valid values are DEBUG, INFO, WARN, ERROR, FATAL");
			}
			config.Remove("level");

			base.Initialize(name, config);
		}

		/// <summary>
		/// Processes the event passed to the provider.
		/// </summary>
		/// <param name="eventRaised">The <see cref="T:System.Web.Management.WebBaseEvent"></see> object to process.</param>
		public override void ProcessEvent(WebBaseEvent eventRaised)
		{
			if (UseBuffering)
			{
				base.ProcessEvent(eventRaised);
			}
			else
			{
				LogEvent(eventRaised);
			}
		}

		/// <summary>
		/// Processes the buffered events.
		/// </summary>
		/// <param name="flushInfo">A <see cref="T:System.Web.Management.WebEventBufferFlushInfo"></see> that contains buffering information.</param>
		public override void ProcessEventFlush(WebEventBufferFlushInfo flushInfo)
		{
			foreach (WebBaseEvent e in flushInfo.Events)
			{
				LogEvent(e);
			}
		}

		/// <summary>
		/// Logs the event.
		/// </summary>
		/// <param name="e">The event to log.</param>
		protected virtual void LogEvent(WebBaseEvent e)
		{
			try
			{
				// Default the log name
				string currentLogName = _logName;

				// Default the level
				log4net.Core.Level currentLevel = _logLevel;
				Exception ex = null;

				// Default any unspecified error events to log at error level
				WebErrorEvent errorEvent = e as WebErrorEvent;
				if (errorEvent != null)
				{
					ex = errorEvent.ErrorException;
					currentLevel = log4net.Core.Level.Error;
				}

				// Check if this event contains the extra log4net info and override if appropriate
				ILoggerEvent loggerEvent = e as ILoggerEvent;
				if (loggerEvent != null)
				{
					if (loggerEvent.LogName != null)
					{
						currentLogName = loggerEvent.LogName;
					}

					if (loggerEvent.Level != Level.Default)
					{
						currentLevel = LogManager.GetRepository().LevelMap[loggerEvent.Level.ToString()];
					}
				}

				ILog log = LogManager.GetLogger(currentLogName);
				log.Logger.Log(_thisDeclaringType, currentLevel, e, ex);
			}
			catch(Exception) 
			{ 
				// Do nothing to keep logging from throwing an exception
			}
		}

		/// <summary>
		/// Performs tasks associated with shutting down the provider.
		/// </summary>
		public override void Shutdown()
		{
			try
			{
				Flush();
			}
			catch (Exception) { }

			base.Shutdown();
		}
	}
}
