using System;
using Foulds.Security.Encryption;
using Microsoft.Win32;

namespace Foulds.Security.SectionHandlers.DataProtection
{
	/// <author>Hannes Foulds, 18 August 2005</author>
	/// <summary>
	/// This class is used to encrypt the strings for the configuration file.
	/// </summary>
	public class EncryptionEngine
	{
		#region Declarations
		private string _registryKey;	// the registry key that the master key is stored in
		private string _masterKey;		// the un-encrypted master key
		private string _plainText;		// the string that should be encrypted
		private string _cipherText;		// the encrypted string
		#endregion

		#region Properties
		/// <summary>
		/// The registry key that the master key is stored in.
		/// </summary>
		public string RegistryKey
		{
			get { return this._registryKey; }
			set { this._registryKey = value; }
		}

		/// <summary>
		/// The un-encrypted master key.
		/// </summary>
		public string MasterKey
		{
			get { return this._masterKey; }
			set { this._masterKey = value; }
		}

		/// <summary>
		/// The string that should be encrypted.
		/// </summary>
		public string PlainText
		{
			get { return this._plainText; }
			set { this._plainText = value; }
		}

		/// <summary>
		/// The encrypted string.
		/// </summary>
		public string CipherText
		{
			get { return this._cipherText; }
			set { this._cipherText = value; }
		}
		#endregion

		#region Constructor
		/// <summary>
		/// Initialize a default instance of the class.
		/// </summary>
		public EncryptionEngine()
		{
		}
		#endregion

		#region Load Master Key
		/// <summary>
		/// Load the master key from the registry.
		/// </summary>
		public void LoadMasterKey()
		{
			// make sure that the registry location was provided
			if ( (this.RegistryKey == null) || (this.RegistryKey.Length == 0) )
				throw new ApplicationException("The registry location to load the master key from must be provided.");

			// get the name of the key and the value to read
			string[] registryArray = this.RegistryKey.Split(",".ToCharArray());
			string keyName = registryArray[0];
			string valueName = registryArray[1];

			// get the registry key
			RegistryKey registryKey = Registry.LocalMachine.OpenSubKey(keyName);

			// read the value from the registry
			string cipherText = registryKey.GetValue(valueName) as string;

			// decrypt the master key
			Foulds.Security.Encryption.DataProtection decryptor = new Foulds.Security.Encryption.DataProtection(Foulds.Security.Encryption.DataProtection.Store.USE_MACHINE_STORE);
            this.MasterKey = decryptor.Decrypt(cipherText, null);
		}
		#endregion

		#region Save Master Key
		/// <summary>
		/// Encrypt and save the master key to the registry.
		/// </summary>
		public void SaveMasterKey()
		{
			// make sure that the registry location was provided
			if ( (this.RegistryKey == null) || (this.RegistryKey.Length == 0) )
				throw new ApplicationException("The registry location to load the master key from must be provided.");

			// make sure that the master key was provided
			if ( (this.MasterKey == null) || (this.MasterKey.Length == 0) )
				throw new ApplicationException("The master key to store in the registry must be provided.");

			// get the name of the key and the value to read
			string[] registryArray = this.RegistryKey.Split(",".ToCharArray());
			string keyName = registryArray[0];
			string valueName = registryArray[1];

			// get the registry key
			RegistryKey registryKey = Registry.LocalMachine.CreateSubKey(keyName);

			// get the cipherText for the key
			Foulds.Security.Encryption.DataProtection encryptor = new Foulds.Security.Encryption.DataProtection(Foulds.Security.Encryption.DataProtection.Store.USE_MACHINE_STORE);
			string cipherText = encryptor.Encrypt(this.MasterKey, null);

			// save the key in the registry
			registryKey.SetValue(valueName, cipherText);
		}
		#endregion

		#region Encrypt String
		/// <summary>
		/// Encrypt the plain text string.
		/// </summary>
		public void EncryptString()
		{
			// make sure that the master key was provided
			if ( (this.MasterKey == null) || (this.MasterKey.Length == 0) )
				throw new ApplicationException("The master key used for encryption must be provided.");

			// make sure that the plain text was provided
			if ( (this.PlainText == null) || (this.PlainText.Length == 0) )
				throw new ApplicationException("The plain text to encrypt must be provided.");

			// encrypt the plain text
			TripleDes encryptor = new TripleDes();
			this.CipherText = encryptor.Encrypt(this.PlainText, this.MasterKey);
		}
		#endregion

		#region Decrypt String
		/// <summary>
		/// Decrypt the cipher text.
		/// </summary>
		public void DecryptString()
		{
			// make sure that the master key was provided
			if ( (this.MasterKey == null) || (this.MasterKey.Length == 0) )
				throw new ApplicationException("The master key used for encryption must be provided.");

			// make sure that the cipher text was provided
			if ( (this.CipherText == null) || (this.CipherText.Length == 0) )
				throw new ApplicationException("The cipher text to decrypt must be provided.");

			// encrypt the plain text
			TripleDes decryptor = new TripleDes();
			this.PlainText = decryptor.Decrypt(this.CipherText, this.MasterKey);
		}
		#endregion
	}
}
