using System;
using System.Collections.Specialized;
using System.Xml;
using Microsoft.Win32;

namespace Foulds.Security.SectionHandlers.DataProtection
{
	/// <author>Hannes Foulds, 18 August 2005</author>
	/// <summary>
	/// This class is used to read the encrypted section from the config file.
	/// </summary>
	public class EncryptedSettings : NameValueCollection
	{
		#region Declarations
		/// <summary>The class that will be used for the encryption and decryotion of data.</summary>
		protected EncryptionEngine encryptionEngine;

		private string _registryKey;		// the registry key containing the descryption key
		#endregion

		#region Properties
		/// <summary>
		/// The registry key containing the descryption key.
		/// </summary>
		public string RegistryKey
		{
			get 
			{ 
				return this._registryKey; 
			}
			
			set
			{ 
				this._registryKey = value; 
				this.LoadDecryptionKey();
			}
		}
		#endregion

		#region Constructor
		/// <summary>
		/// Initialize a default instance of the class.
		/// </summary>
		public EncryptedSettings()
		{
			this.Initialize();
		}

		/// <summary>
		/// Initialize a new instance of the class and load the data from the XML node provided.
		/// </summary>
		/// <param name="dataNode">The XML Node from which data should be loaded for the class.</param>
		public EncryptedSettings(XmlNode dataNode)
		{
			this.Load(dataNode);
		}
		#endregion

		#region Initialize
		/// <summary>
		/// Initialize the object.
		/// </summary>
		private void Initialize()
		{
			this.encryptionEngine = new EncryptionEngine();
			this.RegistryKey = string.Empty;
			this.Clear();
		}
		#endregion

		#region Load
		/// <summary>
		/// Load the data for the class from the XML node provided.
		/// </summary>
		/// <param name="dataNode">The XML Node from which data should be loaded.</param>
		public void Load(XmlNode dataNode)
		{
			try
			{
				// initialize the object
				this.Initialize();

				// load the registry key
				this.RegistryKey = dataNode.Attributes["registryKey"].Value;

				// load the value nodes
				XmlNodeList valueNodes = dataNode.SelectNodes("add");
				foreach(XmlNode valueNode in valueNodes)
				{
					this.LoadValueNode(valueNode);
				}
			}
			catch (Exception ex)
			{
				string message = string.Concat(ex.Message, System.Environment.NewLine, ex.StackTrace);
				Console.WriteLine(message);
				throw;
			}
		}
		#endregion

		#region Load Value Node
		/// <summary>
		/// Load a value node for the object.
		/// </summary>
		/// <param name="valueNode"></param>
		private void LoadValueNode(XmlNode valueNode)
		{
			string key = valueNode.Attributes["key"].Value;
			string value = valueNode.Attributes["value"].Value;

			this.Add(key, this.DecryptValue(value));
		}
		#endregion

		#region Load Decryption Key
		/// <summary>
		/// Load the decryption key from the registry.
		/// </summary>
		private void LoadDecryptionKey()
		{
			if (this.RegistryKey.Length > 0)
			{
				this.encryptionEngine.RegistryKey = this.RegistryKey;
				this.encryptionEngine.LoadMasterKey();
			}
		}
		#endregion

		#region Decrypt Value
		/// <summary>
		/// Decrypt the value of a key value pair for storing in the collection.
		/// </summary>
		/// <param name="cipherText">The cipher text of the value to decrypt.</param>
		/// <returns>Returns the decrypted value.</returns>
		private string DecryptValue(string cipherText)
		{
			this.encryptionEngine.CipherText = cipherText;
			this.encryptionEngine.DecryptString();
			return this.encryptionEngine.PlainText;
		}
		#endregion
	}
}
