import java.awt.*;

/**
 * This class represents an object in a 3D world. It is made up by an
 * array of vertices and an array of surfaces built on them. Attributes
 * of the object are position and angle, each of them in 3D. The values
 * of the angle attributes are stored in a special fixed point number
 * format, which is used by and described in FastMath. The coordinates
 * of the vertices reside in "local coordinates", i.e. indepentent from
 * the actual position of the object. The transformation according to
 * the position and angle will be done automatically by a FixedCamera.
 * <P>
 * To make things clear, this is an example code for a tetrahedron:
 * <PRE>
 * int[][] vertices=
 * {
 * 	{0,-1,0},	// vertex 0 x/y/z coordinates
 * 	{-1,1,-1},	// vertex 1 x/y/z coordinates
 * 	{1,1,-1},	// vertex 2 x/y/z coordinates
 * 	{0,0,1}		// vertex 3 x/y/z coordinates
 * };
 * &nbsp;
 * int[][] surfaces=
 * {
 * 	{0,1,2},	// surface 0 is made up by vertices 0,1 and 2
 * 	{0,1,3},	// surface 1 is made up by vertices 0,1 and 3
 * 	{0,2,3},	// surface 2 is made up by vertices 0,2 and 3
 * 	{1,2,3} 	// surface 3 is made up by vertices 1,2 and 3
 * }
 * </PRE>
 * <P>
 * author: Michael Kraus<BR>
 * version: 1.2<BR>
 * date: 1999/5/17<BR>
 * environment: JDK 1.0.2<BR>
 * email: <A href="michael.kraus@informatik.uni-muenchen.de">michael.kraus@informatik.uni-muenchen.de</A><BR>
 * homepage: <A href="www.informatik.uni-muenchen.de/~michael.kraus">www.informatik.uni-muenchen.de/~michael.kraus</A>
 */

public class Wireframe
{
	/**
	 * index of the x coordinate in the local, world and view
	 * coordinate array
	 */

	private static final int X=0;

	/**
	 * index of the y coordinate in the local, world and view
	 * coordinate array
	 */

	private static final int Y=1;

	/**
	 * index of the z coordinate in both the local and world
	 * coordinate array
	 */

	private static final int Z=2;

	/**
	 * x coordinate of the object's position in world coordinates
	 */

	private int xPosition;

	/**
	 * y coordinate of the object's position in world coordinates
	 */

	private int yPosition;

	/**
	 * z coordinate of the object's position in world coordinates
	 */

	private int zPosition;

	/**
	 * x angle of the object's state in special fixed point
	 * number format
	 */

	private int xAngle;

	/**
	 * y angle of the object's state in special fixed point
	 * number format
	 */

	private int yAngle;

	/**
	 * z angle of the object's state in special fixed point
	 * number format
	 */

	private int zAngle;

	/**
	 * array of local coordinates of the object's vertices. Each
	 * vertex is an integer array containing its x, y and z
	 * coordinate.
	 */

	private int local[][];

	/**
	 * array of world coordinates of the object's vertices. Each
	 * vertex is an integer array containing its x, y and z
	 * coordinate.
	 */

	private int world[][];

	/**
	 * array of view coordinates of the object's vertices. Each
	 * vertex is an integer array containing its x and y
	 * coordinate.
	 */

	private int view[][];

	/**
	 * array of surfaces of the object. Each surface is an array
	 * of the indices of the vertices that make up that surface,
	 * as used in the local, world and view arrays.
	 */

	private int surface[][];

	/**
	 * true if the world coords of this object are valid
	 */

	private boolean valid;

	/**
	 * construct a new Wireframe object. Parameter are: local
	 * coordinates, surfaces, position and angle of the object.
	 */

	public Wireframe(int local[][],int surface[][],int xPosition,int yPosition,int zPosition,int xAngle,int yAngle,int zAngle)
	{
		this.local=local;
		this.world=new int[local.length][3];
		this.surface=surface;

		setXPosition(xPosition);
		setYPosition(yPosition);
		setZPosition(zPosition);

		setXAngle(xAngle);
		setYAngle(yAngle);
		setZAngle(zAngle);
	}

	/**
	 * convert the object from local coordinates to world
	 * coordinates, according to its position and angle
	 */

	private void worldcoords()
	{
		int x;
		int y;
		int z;

		int matrix[][]=new int[3][3];

		matrix[0][0]=(FastMath.cos(yAngle-zAngle)+FastMath.cos(yAngle+zAngle))/2;
		matrix[0][1]=((FastMath.cos(xAngle-yAngle-zAngle)+FastMath.cos(xAngle-yAngle+zAngle)-FastMath.cos(xAngle+yAngle-zAngle)-FastMath.cos(xAngle+yAngle+zAngle))/2-FastMath.sin(xAngle-zAngle)+FastMath.sin(xAngle+zAngle))/2;
		matrix[0][2]=((FastMath.sin(xAngle-yAngle+zAngle)+FastMath.sin(xAngle-yAngle-zAngle)-FastMath.sin(xAngle+yAngle-zAngle)-FastMath.sin(xAngle+yAngle+zAngle))/2+FastMath.cos(xAngle-zAngle)-FastMath.cos(xAngle+zAngle))/2;
		matrix[1][0]=(FastMath.sin(yAngle-zAngle)-FastMath.sin(yAngle+zAngle))/2;
		matrix[1][1]=((FastMath.sin(xAngle-yAngle-zAngle)-FastMath.sin(xAngle-yAngle+zAngle)-FastMath.sin(xAngle+yAngle-zAngle)+FastMath.sin(xAngle+yAngle+zAngle))/2+FastMath.cos(xAngle-zAngle)+FastMath.cos(xAngle+zAngle))/2;
		matrix[1][2]=((FastMath.cos(xAngle-yAngle+zAngle)+FastMath.cos(xAngle+yAngle-zAngle)-FastMath.cos(xAngle-yAngle-zAngle)-FastMath.cos(xAngle+yAngle+zAngle))/2+FastMath.sin(xAngle-zAngle)+FastMath.sin(xAngle+zAngle))/2;
		matrix[2][0]=FastMath.sin(yAngle);
		matrix[2][1]=(FastMath.sin(yAngle-xAngle)-FastMath.sin(xAngle+yAngle))/2;
		matrix[2][2]=(FastMath.cos(xAngle-yAngle)+FastMath.cos(xAngle+yAngle))/2;

		for(int n=0;n<local.length;n++)
		{
			x=local[n][X];
			y=local[n][Y];
			z=local[n][Z];

			world[n][X]=xPosition+((matrix[0][0]*x+matrix[0][1]*y+matrix[0][2]*z)>>FastMath.TWOPI_BITS);
			world[n][Y]=yPosition+((matrix[1][0]*x+matrix[1][1]*y+matrix[1][2]*z)>>FastMath.TWOPI_BITS);
			world[n][Z]=zPosition+((matrix[2][0]*x+matrix[2][1]*y+matrix[2][2]*z)>>FastMath.TWOPI_BITS);
		}
	}

	/**
	 * convert the object from world coordinates to view
	 * coordinates, according to the attributes of the
	 * camera argument. This method should never be called
	 * by the user.
	 */

	public void viewcoords(FixedCamera camera)
	{
		validate();

		view=camera.viewcoords(world);
	}

	/**
	 * true if the world coordinates of this Wireframe
	 * object are valid. This method should never be called
	 * by the user.
	 */

	public boolean isValid()
	{
		return valid;
	}

	/**
	 * invalidate the Wireframe object
	 */

	private void invalidate()
	{
		valid=false;
	}

	/**
	 * validate the Wireframe object. This is, calculate the
	 * world coordinates.
	 */

	private void validate()
	{
		if(valid==false)
		{
			worldcoords();

			valid=true;
		}
	}

	/**
	 * draw the object. The local coordinates must have been
	 * be converted to view coordinates before, so this
	 * method should never be called by the user.
	 */

	public void draw(Graphics graphics)
	{
		int n;
		int m;

		for(n=0;n<surface.length;n++)
		{
			Polygon polygon=new Polygon();

			for(m=0;m<surface[n].length;m++)
				polygon.addPoint(view[surface[n][m]][X],view[surface[n][m]][Y]);

			graphics.drawPolygon(polygon);
		}
	}

	/**
	 * get the x position of the Wireframe object
	 */

	public int getXPosition()
	{
		return xPosition;
	}

	/**
	 * set the x position of the Wireframe object
	 */

	public void setXPosition(int xPosition)
	{
		this.xPosition=xPosition;

		invalidate();
	}

	/**
	 * get the y position of the Wireframe object
	 */

	public int getYPosition()
	{
		return yPosition;
	}

	/**
	 * set the y position of the Wireframe object
	 */

	public void setYPosition(int yPosition)
	{
		this.yPosition=yPosition;

		invalidate();
	}

	/**
	 * get the z position of the Wireframe object
	 */

	public int getZPosition()
	{
		return zPosition;
	}

	/**
	 * set the z position of the Wireframe object
	 */

	public void setZPosition(int zPosition)
	{
		this.zPosition=zPosition;

		invalidate();
	}

	/**
	 * move the Wireframe object's position
	 */

	public void move(int dx,int dy,int dz)
	{
		xPosition+=dx;
		yPosition+=dy;
		zPosition+=dz;

		invalidate();
	}

	/**
	 * get the x angle of the Wireframe object
	 */

	public int getXAngle()
	{
		return xAngle;
	}

	/**
	 * set the x angle of the Wireframe object
	 */

	public void setXAngle(int xAngle)
	{
		this.xAngle=xAngle;

		invalidate();
	}

	/**
	 * get the y angle of the Wireframe object
	 */

	public int getYAngle()
	{
		return yAngle;
	}

	/**
	 * set the y angle of the Wireframe object
	 */

	public void setYAngle(int yAngle)
	{
		this.yAngle=yAngle;

		invalidate();
	}

	/**
	 * get the z angle of the Wireframe object
	 */

	public int getZAngle()
	{
		return zAngle;
	}

	/**
	 * set the z angle of the Wireframe object
	 */

	public void setZAngle(int zAngle)
	{
		this.zAngle=zAngle;

		invalidate();
	}

	/**
	 * rotate the Wireframe object's angle
	 */

	public void rotate(int dx,int dy,int dz)
	{
		xAngle+=dx;
		yAngle+=dy;
		zAngle+=dz;

		invalidate();
	}
}

