import java.awt.*;
import java.util.*;

/**
 * This class transforms the 3D coordinates of Wireframe objects into 2D
 * coordinates suitable for screen display. In a sense, this class makes
 * up a 3D world. The name 'Camera' derives from a real camera that puts
 * the real world on a flat film. This camera is 'Fixed' because it always
 * looks in the same direction (it's not really fixed as you can move it
 * around, but its angle is fixed).
 * <P>
 * If you want to set up and display a 3D world, you have to do the following
 * steps:
 * <OL>
 * <LI> create a FixedCamera object
 * <LI> create some Wireframe objects
 * <LI> add the Wireframe objects to the FixedCamera using addWireframe
 * <LI> draw the 3D world within any Graphics context using draw
 * <LI> after that, you can change the camera's attributes using its
 *      setXXX/getXXX/move methods and draw the 3D world again.
 * <LI> you can also change the Wireframe objects' attributes using its
 *      setXXX/getXXX/move/rotate methods and draw the 3D world again.
 * </OL>
 * <P>
 * author: Michael Kraus<BR>
 * version: 1.2<BR>
 * date: 1999/5/17<BR>
 * environment: JDK 1.0.2<BR>
 * email: <A href="michael.kraus@informatik.uni-muenchen.de">michael.kraus@informatik.uni-muenchen.de</A><BR>
 * homepage: <A href="www.informatik.uni-muenchen.de/~michael.kraus">www.informatik.uni-muenchen.de/~michael.kraus</A>
 */

public class FixedCamera
{
	/**
	 * index of the x coordinate in both the world and view
	 * coordinate array
	 */

	private static final int X=0;

	/**
	 * index of the y coordinate in both the world and view
	 * coordinate array
	 */

	private static final int Y=1;

	/**
	 * index of the z coordinate in world coordinate array
	 */

	private static final int Z=2;

	/**
	 * x coordinate of the camera's position in world coordinates
	 */

	private int xposition;

	/**
	 * y coordinate of the camera's position in world coordinates
	 */

	private int yposition;

	/**
	 * z coordinate of the camera's position in world coordinates
	 */

	private int zposition;

	/**
	 * distance of the projection screen of the camera from its
	 * position in world coordinates
	 */

	private int screendistance;

	/**
	 * width of the camera's projection screen in world coordinates.
	 */

	private int screenwidth;

	/**
	 * height of the camera's projection screen in world coordinates
	 */

	private int screenheight;

	/**
	 * list of all Wireframe objects associated with this camera
	 */

	private Vector memberVector;

	/**
	 * true if the view coords of all members are valid
	 */

	private boolean isValid;

	/**
	 * construct a new FixedCamera. Parameters: position of the camera
	 * in world coordinates, distance of the projection screen from
	 * the position in world coordinates, size of the projection
	 * screen in world coordinates.
	 */

	public FixedCamera(int xposition,int yposition,int zposition,int screenwidth,int screenheight,int screendistance)
	{
		memberVector=new Vector();

		setXPosition(xposition);
		setYPosition(yposition);
		setZPosition(zposition);

		setScreenwidth(screenwidth);
		setScreenheight(screenheight);
		setScreendistance(screendistance);
	}

	/**
	 * convert an array of world coordinates into view coordinates.
	 * Both the argument and the result array are two-dimensional.
	 * the first dimension is for the multiple vertices, the second
	 * one for the coordinate. 0=x, 1=y, 2=z coordinate. This method
	 * should not be called by the user.
	 */

	public int[][] viewcoords(int world[][])
	{
		int n;

		int x1;
		int y1;
		int z1;

		int view[][]=new int[world.length][2];

		for(n=0;n<world.length;n++)
		{
			x1=world[n][X]-xposition;
			y1=world[n][Y]-yposition;
			z1=world[n][Z]-zposition;

			view[n][X]=screenwidth/2+screendistance*x1/z1;
			view[n][Y]=screenheight/2+screendistance*y1/z1;
		}

		return view;
	}

	/**
	 * adds a Wireframe object to the member list of the camera
	 */

	public void addWireframe(Wireframe wireframe)
	{
		memberVector.addElement(wireframe);
	}

	/**
	 * removes a Wireframe object from the member list of the camera
	 */

	public void removeWireframe(Wireframe wireframe)
	{
		memberVector.removeElement(wireframe);
	}

	/**
	 * invalidate the camera. This means, that before the members
	 * are drawn, their view coordinates will be recalculated.
	 */

	private void invalidate()
	{
		isValid=false;
	}

	/**
	 * validate the camera. This is, calculate the view
	 * coordinates of all member Wireframe objects.
	 */

	private void validate()
	{
		if(isValid==false)
		{
			for(int n=0;n<memberVector.size();n++)
				((Wireframe)memberVector.elementAt(n)).viewcoords(this);

			isValid=true;
		}
		else
		{
			Wireframe wireframe;

			for(int n=0;n<memberVector.size();n++)
				if(!(wireframe=(Wireframe)memberVector.elementAt(n)).isValid())
					wireframe.viewcoords(this);
		}
	}

	/**
	 * draw all member Wireframe objects from this camera's
	 * point of view
	 */

	public void draw(Graphics graphics)
	{
		validate();

		for(int n=0;n<memberVector.size();n++)
			((Wireframe)memberVector.elementAt(n)).draw(graphics);
	}

	/**
	 * get the x position of the camera
	 */

	public int getXPosition()
	{
		return xposition;
	}

	/**
	 * set the x position of the camera
	 */

	public void setXPosition(int xposition)
	{
		this.xposition=xposition;

		invalidate();
	}

	/**
	 * get the y position of the camera
	 */

	public int getYPosition()
	{
		return yposition;
	}

	/**
	 * set the y position of the camera
	 */

	public void setYPosition(int yposition)
	{
		this.yposition=yposition;

		invalidate();
	}

	/**
	 * get the z position of the camera
	 */

	public int getZPosition()
	{
		return zposition;
	}

	/**
	 * set the z position of the camera
	 */

	public void setZPosition(int zposition)
	{
		this.zposition=zposition;

		invalidate();
	}

	/**
	 * move the camera's position
	 */

	public void move(int dx,int dy,int dz)
	{
		xposition+=dx;
		yposition+=dy;
		zposition+=dz;

		invalidate();
	}

	/**
	 * get the width of the camera's projection screen
	 */

	public int getScreenwidth()
	{
		return screenwidth;
	}

	/**
	 * set the width of the camera's projection screen
	 */

	public void setScreenwidth(int screenwidth)
	{
		this.screenwidth=screenwidth;

		invalidate();
	}

	/**
	 * get the height of the camera's projection screen
	 */

	public int getScreenheight()
	{
		return screenheight;
	}

	/**
	 * set the height of the camera's projection screen
	 */

	public void setScreenheight(int screenheight)
	{
		this.screenheight=screenheight;

		invalidate();
	}

	/**
	 * get the distance of the camera's projection screen
	 * from the position of the camera
	 */

	public int getScreendistance()
	{
		return screendistance;
	}

	/**
	 * set the distance of the camera's projection screen
	 * from the position of the camera
	 */

	public void setScreendistance(int screendistance)
	{
		this.screendistance=screendistance;

		invalidate();
	}
}

