/**
 * This class provides really fast sine and cosine functions. In fact, it
 * uses a precalculated table of sine and cosine values. The number format
 * is NOT floating point, but fixed point. The public attribute TWOPI_BITS
 * indicates the number of bits used to represent the value of 2*pi,
 * i.e. 6.283185... Also, the public attribute TWOPI represents the value
 * of 2*pi in this number format. The methods sin and cos take their
 * arguments in this number format and their return values are also in
 * this special format.
 * <P>
 * author: Michael Kraus<BR>
 * version: 1.2<BR>
 * date: 1999/5/17<BR>
 * environment: JDK 1.0.2<BR>
 * email: <A href="michael.kraus@informatik.uni-muenchen.de">michael.kraus@informatik.uni-muenchen.de</A><BR>
 * homepage: <A href="www.informatik.uni-muenchen.de/~michael.kraus">www.informatik.uni-muenchen.de/~michael.kraus</A>
 */

public class FastMath
{
	/**
	 * number of bits used to represent the value of 2*pi
	 * (6.283185...) in the special fixed point number format
	 * used in this class
	 */

	public static final int TWOPI_BITS=8;

	/**
	 * value of 2*pi (6.283185...) in the special fixed
	 * point number format used in this class
	 */

	public static final int TWOPI=1<<TWOPI_BITS;

	/**
	 * value of pi/2 (1.570796...) in the special fixed
	 * point number format used in this class
	 */

	public static final int PIHALF=TWOPI/4;

	/**
	 * bitmask of 2*pi (6.283185...) in the special fixed
	 * point number format used in this class
	 */

	public static final int TWOPI_MASK=TWOPI-1;

	/**
	 * here the precalculated cosine values are stored
	 */

	private static int costable[];

	/*
	 * precalculate the cosine values in the interval [0;2*pi[
	 */

	static
	{
		int n;

		costable=new int[TWOPI];

		for(n=0;n<TWOPI;n++)
			costable[n]=(int)(Math.cos(2*Math.PI*n/TWOPI)*TWOPI);
	}

	/**
	 * get the sine value of the argument. Both the argument
	 * and the result are in a special fixed point number
 	 * format.
	 */

	public static int sin(int value)
	{
		return costable[Math.abs(PIHALF-value)&TWOPI_MASK];
	}

	/**
	 * get the cosine value of the argument. Both the argument
	 * and the result are in a special fixed point number
 	 * format.
	 */

	public static int cos(int value)
	{
		return costable[Math.abs(value)&TWOPI_MASK];
	}
}

