<?php

/**
* AcronymIT - PHP Class to create acronyms into a text
*
* Tested and working on PHP 4.3.2 and higher
*
* LICENSE: This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License v2 as published by
* the Free Software Foundation.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*
* @author       Giulio Bai <slide.wow@gmail.com>
* @copyright   (C)2007 Giulio Bai - GNU GPL
* @license       http://www.gnu.org/licenses/gpl.html GNU General public License
* @version      1.0
* @link            http://hewle.com
*/


/**
* A class to check the presence of a domain name, having any extension.
*
* Example of use (1):
* The following code will find all the acronyms in the text passed to the method
* and will link to the italian version of Wikipediafor further explanations
* <code>
* include_once('acronymIT.php');
*
* $text = "The PHP version 6 is under development. Read more on the WWW or get info via RSS"; 
* $acronymit = new AcronymIT();
* $text = $acronymit->acronym($text, 1, "it");
*
* echo $text;
* </code>
*
* Example of use (2):
* Listing acronyms stored in the default file, then in another one.
* <code>
* include_once('acronymIT.php');
*
* $acronymit = new AcronymIT();
* echo "File $acronymit->file:<br />"; // Will be "FIle acronyms.dat:"
* $acronymit->list_acronyms(5); // Lists the first 5 acronyms in the default file
*
* $acronymit->set_file("path/to/other_acronyms.txt"); // Changing the file
*
* echo "File $acronymit->file:"; // Will be "File path/to/other_acronyms.txt:"
* $acronymit->list_acronyms() // Lists all the acronyms in the file
*
* echo "Done! ;)";
* </code>
*
* @author      Giulio Bai <slide.wow@gmail.com>
* @copyright  (C)2007 Giulio Bai
* @license      http://www.gnu.org/licenses/gpl.html GNU General Public License
* @version     1.0
* @link           http://hewle.com
*/ 
class AcronymIT
{

	var $file = 'acronyms.dat';
	
	/**
	 * Looks for the known (and most common) acronyms in the text and make them
	 * HTML acronyms (so they'll have the <acronym> tag).
	 *
	 * @param string $text the text to scan for acronyms
	 * @param bool $link if set, prints also the link to Wikipedia
	 * @param string $lang the language of the Wikipedia portal to link
	 * @return string the new text with the acronyms
	 */
	function acronym($text, $link = 0, $lang = "en")
	{		
		$acronyms = parse_ini_file($this->file);

		foreach($acronyms as $acronym => $definition) {
			if ($link)
				$aux = " <a href=\"$lang.wikipedia.org/wiki/$acronym\" title=\"Further info about $acronym\">(?)</a> ";
			else
				$aux = " ";
				
			$text = str_ireplace(" $acronym ", " <span style=\"text-transform: uppercase\"><acronym title=\"$definition\">$acronym</acronym></span>$aux", $text);
		}

		return $text;
	}
	
	
	/**
	 * Lists acronyms stored in the selected file
	 *
	 * By putting a negative value as limit, all the acronyms will be listed
	 *
	 * @param int $limit how many acronyms do you want to print?
	 * @return void
	 */
	function list_acronyms($limit = -1)
	{
		$acronyms = parse_ini_file($this->file);
			
		$count = count($acronyms);
			
		foreach($acronyms as $acronym => $definition) {
			if ($count > count($acronyms) - $limit) {
				echo count($acronyms)-$count . '. ' . $acronym . ': ' . $definition . '<br />';
				
				$count --;
			} else
				return;
		}
	}
	
	
	/**
	 * Sets the file where acronyms are stored in
	 *
	 * @param string $file the file path
	 * @return void
	 */
	function set_file($file)
	{
		$this->file = $file;
	}


	/**
	 * Adds the specified acronym to the file set before through the set_file() method
	 *
	 * @see set_file()
	 * @see rm_acronym()
	 * @param string $acronym the acronym to add	  
	 * @param string $definition what does the acronym mean?	  
	 * @return bool true on success
	 */
	function add_acronym($acronym, $definition)
	{
		$data = "";
		
		for ($i=0; $i<count(file($this->file)); $i++)
			$data .= $this->file[$i];

		$data .= "\n$acronym = \"$definition\"";

		if(file_put_contents($this->file, $data)) return true;
		
		return false;
	}
	
	/**
	 * Deletes the specified acronym from the file set before through the set_file() method
	 *
	 * @see set_file()
	 * @see add_acronym()
	 * @param int $id the id of the acronym to remove	  
	 * @return bool true on success
	 */
	function rm_acronym($id)
	{
		//Loop through reading all lines except the one you want to delete
		for ($i=0; $i<count(file($this->file)); $i++)
			if ($i != $id)
				$data .= $this->file[$i];

		if(file_put_contents($this->file, $data)) return true;
		
		return false;
	}	
}

?>