<?php
/* 
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2005 Tropotek Development
 */

/**
 * 
 *
 * @package Util
 */
class Ext_Db_User extends Dk_Db_Object implements Dk_User_Interface 
{
    
    /**
     * No login required
     */
    const GROUP_PUBLIC = 0;
    /**
     * User permissions required
     */
    const GROUP_USER = 1;
    /**
     * Admin permissions required
     */
    const GROUP_ADMIN = 128;
    
    
    
    /**
     * @var string
     */
    protected $name = '';
    
    /**
     * @var string
     */
    protected $email = '';
    
    /**
     * @var string
     */
    protected $image = '';
    
    /**
     * @var boolean
     */
    protected $active = true;
    
    /**
     * @var string
     */
    protected $username = '';
    
    /**
     * @var string
     */
    protected $password = '';
    
    /**
     * @var integer
     */
    protected $group = self::GROUP_USER;
    
    /**
     * @var string
     */
    protected $hash = '';
    
    /**
     * @var Dk_Util_Date
     */
    protected $modified = null;
    
    /**
     * @var Dk_Util_Date
     */
    protected $created = null;
    

    /**
     * __construct
     *
     */
    function __construct()
    {
        $this->modified = Dk_Util_Date::createDate();
        $this->created = Dk_Util_Date::createDate();
        
    }


    
    
    /**
     * Find a user from the database
     *
     * @param string $username
     * @return Ext_Db_User
     */
    static function findByUsername($username)
    {
        return Ext_Db_UserMapper::findByUsername($username);
    }

    
    /**
     * Required: 
     * Range: A string with 128 characters.
     *
     * @return string
     */
    function getName()
    {
        return $this->name;
    }
    
    /**
     * Required: 
     * Range: A string with 128 characters.
     *
     * @param string $value
     */
    function setName($value)
    {
        $this->name = $value;
    }
    
    /**
     * Required: 
     * Range: A string with 128 characters.
     *
     * @return string
     */
    function getEmail()
    {
        return $this->email;
    }
    
    /**
     * Required: 
     * Range: A string with 128 characters.
     *
     * @param string $value
     */
    function setEmail($value)
    {
        $this->email = $value;
    }
    
    /**
     * Optional: The user avtar image 120x120
     * Range: A string with 255
     *  characters.
     *
     * @return string
     */
    function getImage()
    {
        return $this->image;
    }
    
    /**
     * Optional: The user avtar image 120x120
     * Range: A string with 255
     *  characters.
     *
     * @param string $value
     */
    function setImage($value)
    {
        $this->image = $value;
    }
    
    /**
     * Get the image Url object
     *
     * @return Dk_Util_Url
     */
    function getImageUrl()
    {
        return new Dk_Util_Url($this->getImage());
    }
    
    /**
     * If the user is inactive they cannot login
     * Range: true or false
     *
     * @return boolean
     */
    function getActive()
    {
        return $this->active;
    }
    
    /**
     * If the user is inactive they cannot login
     * Range: true or false
     *
     * @param boolean $b
     */
    function setActive($b)
    {
        $this->active = $b;
    }
    
    /**
     * Required: 
     * Range: A string with 64 characters.
     *
     * @return string
     */
    function getUsername()
    {
        return $this->username;
    }
    
    /**
     * Required: 
     * Range: A string with 64 characters.
     *
     * @param string $value
     */
    function setUsername($value)
    {
        $this->username = $value;
    }
    
    /**
     * Required: 
     * Range: A string with 64 characters.
     *
     * @return string
     */
    function getPassword()
    {
        return $this->password;
    }
    
    /**
     * Required: 
     * Range: A string with 64 characters.
     *
     * @param string $value
     */
    function setPassword($value)
    {
        $this->password = $value;
    }
    
    /**
     * Required: ADMIN = 128, USER = 1
     * Safe Range: A normal-size
     *  integer. The signed range is -2147483648 to 2147483647. The unsigned
     *  range is 0 to 4294967295.
     *
     * @return integer
     */
    function getGroup()
    {
        return $this->group;
    }
    
    /**
     * Required: ADMIN = 128, USER = 1
     * Safe Range: A normal-size
     *  integer. The signed range is -2147483648 to 2147483647. The unsigned
     *  range is 0 to 4294967295.
     *
     * @return integer
     */
    function getGroupId()
    {
        return $this->getGroup();
    }
    
    /**
     * Required: ADMIN = 128, USER = 1
     * Safe Range: A normal-size
     *  integer. The signed range is -2147483648 to 2147483647. The unsigned
     *  range is 0 to 4294967295.
     *
     * @param integer $i
     */
    function setGroup($i)
    {
        $this->group = $i;
    }
    
    /**
     * Used by the user activation system
     * Range: A string with 64
     *  characters.
     *
     * @return string
     */
    function getHash()
    {
        return $this->hash;
    }
    
    /**
     * Used by the user activation system
     * Range: A string with 64
     *  characters.
     *
     * @param string $value
     */
    function setHash($value)
    {
        $this->hash = $value;
    }
    
    /**
     * Create a unique has value for this user
     *
     * @return string
     */
    function createHash()
    {
        return md5($this->username . '' . $this->password . '_DkWiki');
    }
    
    /**
     * Get modified
     * Range: '1000-01-01 00:00:00' to '9999-12-31
     *  23:59:59'. DB values in 'YYYY-MM-DD HH:MM:SS' format.
     *
     * @return Dk_Util_Date
     */
    function getModified()
    {
        return $this->modified;
    }
    
    /**
     * Get created
     * Range: '1000-01-01 00:00:00' to '9999-12-31
     *  23:59:59'. DB values in 'YYYY-MM-DD HH:MM:SS' format.
     *
     * @return Dk_Util_Date
     */
    function getCreated()
    {
        return $this->created;
    }
    
    /**
     * Get the object's data path
     *
     * @return Dk_Util_Path
     */
    function getDataPath()
    {
        return new Dk_Util_Path('/data/user/' . $this->getMapper()->getVolitileId($this));
    }
    

    /**
     * getHomeUrl
     *
     * @return Dk_Util_Url
     */
    function getHomeUrl()
    {
        $url = new Dk_Util_Url('/index.php');
        switch($this->getGroupId()) {
            case self::GROUP_ADMIN:
                $url = new Dk_Util_Url('/index.php');
            case self::GROUP_USER:
                $url = new Dk_Util_Url('/index.php');
        }
        return $url;
    }


}

/**
 * A validator object for the Ext_Db_User object
 *
 * @package Util
 */
class Ext_Db_UserValidator extends Dk_Util_Validator
{

    /**
     * @var Ext_Db_User
     */
    protected $obj = null;

    /**
     * Validates
     *
     */
    function validate()
    {
        
        if (!ereg('^.{1,128}$', $this->obj->getName())) {
            $this->setError('name', 'Invalid Field Value.');
        }
        if (!ereg('^[0-9a-zA-Z]([-_.]*[0-9a-zA-Z])*@[0-9a-zA-Z]([-.]?[0-9a-zA-Z])*$', $this->obj->getEmail())) {
            $this->setError('email', 'Invalid Field Value.');
        }
        $path = Dk_Util_Path::getFileExtension($this->obj->getImage());
        if ($this->obj->getImage() != null && ($path != '' && $path != 'jpeg' && 
            $path != 'jpg' && $path != 'gif' && $path != 'png')) 
        {
            $this->setError('image', 'Invalid file type.');
        }
        if (!ereg('^[a-zA-Z0-9_\.@-]{3,64}$', $this->obj->getUsername())) {
            $this->setError('username', 'Invalid Field Value.');
        }
        if (!ereg('^[a-zA-Z0-9_\. @-]{4,64}$', $this->obj->getPassword())) {
            $this->setError('password', 'Invalid Field Value. (Must be more than 4 characters.)');
        }
        if ($this->obj->getGroup() <= Ext_Db_User::GROUP_PUBLIC || $this->obj->getGroup() > Ext_Db_User::GROUP_ADMIN) {
            $this->setError('group', 'Invalid Field Value.');
            throw new Dk_ExceptionIllegalArgument('Invalid group Assignment.');
        }
        
        //NOTE: You have to manually validate any updates to the username or
        //  email in the component itself, compare the new with the old and check 
        //  no user allready exsist with the new username or password.
        if ($this->obj->getId() == 0) {
            // TODO: Check for duplicate emails
            $chk = Ext_Db_UserMapper::findByEmail($this->obj->getEmail());
            if ($chk) {
                $this->setError('email', 'Email exists for an existing user.');
            }
            // TODO: Check for duplicate usernames
            $chk = null;
            $chk = Ext_Db_UserMapper::findByUsername($this->obj->getUsername());
            if ($chk) {
                $this->setError('username', 'Username exists for an existing user.');
            }
        }
        
    }

}
?>