<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */



/**
 * The base mapper object that controls the mapping of XML columns to objects
 *
 * @package Db
 */
class Dk_Xml_Mapper extends Dk_Object 
{
    
    /**
     * @var array
     */
    static private $mappers = array();
    
    
    /**
     * @var Dk_Mapper_Object
     */
    protected $mapper = null;
    
    
    
    /**
     * __construct
     * 
     * @param Dk_Mapper_Object $mapper
     */
    private function __construct(Dk_Mapper_Object $mapper)
    {
        $this->mapper = $mapper;
    }
    
    /**
     * Looks for a mapper class $class . 'Mapper' or you could use the full mapper classname 'Ext_Util_ObjectMapper'
     * 
     * @param string
     * @return Dk_Xml_Mapper
     */
    static function makeMapper($class = '')
    {
        if (substr($class, -6) == 'Mapper') {
            $class = substr($class, 0, -6);
        }
        if (!array_key_exists($class, self::$mappers)) {
            self::$mappers[$class] = new Dk_Xml_Mapper(Dk_Mapper_ObjectFactory::getMapper($class));
        }
        return self::$mappers[$class];
    }
    
    /**
     * Get the object mapper
     *
     * @return Dk_Mapper_Object
     */
    function getMapper()
    {
        return $this->mapper;
    }
    
    /**
     * The class name this mapper is used for.
     *
     * @return string
     */
    function getClass()
    {
        return $this->getMapper()->getClass();
    }
    
    /**
     * Serialise an object into a REST xml packet
     * 
     * @param Dk_Xml_Serialisable $obj
     * @return string
     */
    function restSerialise($obj)
    {
        $row = Dk_Mapper_ObjectLoader::getInstance()->getObjectValues($obj);
        $propertyList = array();
        if (method_exists($obj, '__xmlSleep')) {
            $propertyList = $obj->__xmlSleep();
        }
        if ($propertyList == null || count($propertyList) == 0) {
            $methods = get_class_methods($obj);
            $regs = array();
            foreach ($methods as $m) {
                if (ereg('^get(.)+', $m, $regs)) {
                    $propertyList[] = lcfirst(substr($regs[0], 3));
                }
            }
        }
        $xml = '';
        /* @var $map Dk_Mapper_ColumnMap */
        foreach ($this->getMapper()->getColumnMaps() as $map) {
            if (in_array($map->getPropertyName(), $propertyList)) {
                $xml .= $this->makeRestNode($map->getColumnName(), $map->getColumnValue($row));
            }
        }
        return "<{$this->getClass()}>\n" . $xml . "</{$this->getClass()}>";
    }
    
    /**
     * Create a rest XMl node
     * 
     * @param string $name
     * @param mixed $value
     * @return string
     */
    private function makeRestNode($name, $value)
    {
        if ($value) {
            return "  <$name>" . htmlentities($value) . "</$name>\n";
        }
        return ''; 
    }
    
    

    /**
     * Unserialise a REST xml packet after it has been converted 
     * to a stdClass object first
     * 
     * @param stdClass $stdClass
     * @return Dk_Xml_Serialisable
     */
    function restUnserialise($stdClass)
    {
        $row = array();
        $stdArray = get_object_vars($stdClass);
        /* @var $map Dk_Mapper_ColumnMap */
        foreach ($this->getMapper()->getColumnMaps() as $map) {
            $property = $map->getPropertyName();
            if(array_key_exists($property, $stdArray)) {
                $row[$property] = $map->getPropertyValue($stdArray);
            }
        }
        $obj = $this->getMapper()->loadArray($row);
        if (method_exists($obj, '__xmlWakeup')) {
            $obj->__xmlWakeup();
        }
        return $obj;
    }
    
    
    
    
}
?>