<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * A PHP5 DOM Template Library (extension for MVC components)
 * 
 * @package Web
 */
class Dk_Web_Template extends Dom_Template 
{
    
   
    
    /**
     * component meata data
     * @var array
     */
    protected $componentList = array();
    
    
    
    /**
     * Make a template from an xml file
     * 
     * @param string $xmlFile
     * @return Dk_Web_Template
     * @throws Exception If there is an error with the XML template
     */
    static function load($xmlFile)
    {
        if (!is_file($xmlFile)) {
            return null;
        }
        $xmlStr = file_get_contents($xmlFile);
        $doc = new DOMDocument();
        if (!$doc->loadXML(self::cleanXml($xmlStr))) {
            throw new Exception('Invalid XML: Cannot create `Dom_Template` error reading xml file: `' . $xmlFile . '`, check document content.');
        }
        return new self($doc);
    }

    /**
     * Make a template from an xml string
     * 
     * @param string $xmlStr
     * @return Dk_Web_Template
     * @throws Exception If there is an error with the XML template
     */
    static function loadXml($xmlStr)
    {
        if ($xmlStr == '') {
            return null;
        }
        $doc = new DOMDocument();
        if (!$doc->loadXML(self::cleanXml($xmlStr))) {
            throw new Exception('Invalid XML: Cannot create `Dom_Template`, check document content.');
        }
        return new self($doc);
    }
    
    
    

    /**
     * This function is used when subclassing the template 
     * implement this function to capture new nodes.
     * For this class we are capturing component meta data
     * 
     * @param DOMElement $node
     * @param string $form The form name if inside a form.
     */
    protected function preInit(DOMElement $node, $form = '') 
    {
        if ($node->hasAttribute('var') && $node->hasAttribute('com-class')) {
            $com = new Dk_Web_ComMetaData($node);
            $this->componentList[] = $com;
            
            $var = $node->getAttribute('var');
            if (!array_key_exists($var, $this->var)) {
                $this->var[$var] = array();
            }
            $this->var[$var][] = $node;
            $node->removeAttribute('var');
            
            if ($com->getInlineDom() != null) {
                $this->removeChildren($node );
            }
            return true;  // Cause the node iteration to skip the rest of this node
        }
    }
    
    
    /**
     * Get this templates widget array of Dk_Web_ComMetaData objects
     *
     * @return array
     */
    function getComponentList() 
    {
        return $this->componentList;
    }
    
    
}

/**
 * A holder for widget node metadata pulled from a template src.
 * 
 * @package Web
 */
class Dk_Web_ComMetaData extends Dk_Object
{
    
    /**
     * @var string
     */
    private $classname = '';
    
    /**
     * @var array
     */
    private $parameters = array();
    
    /**
     * @var string
     */
    private $insertVar = '';
    
    /**
     * @var DOMDocument
     */
    private $inlineDom = null;
    
    /**
     * @var DOMElement
     */
    private $node = null;
    
    
    /**
     * __construct
     *
     * @param DOMElement $node
     */
    function __construct($node) 
    {
    	static $idx = 0;
    	$this->id = $idx++;
        // collect component classname
        $this->classname = $node->getAttribute('com-class');
        $node->removeAttribute('com-class');
        
        $this->insertVar = $node->getAttribute('var');
        $this->node = $node;
        
        //  Collect component parameters
        $this->parameters = array();
        $attList = $node->attributes;
        foreach ($attList as $att) {
            if (substr($att->nodeName, 0, 6) == 'param-') {
                $this->parameters[substr($att->nodeName, 6)] = $att->nodeValue;
            }
        }
        foreach (array_keys($this->parameters) as $name) {
            $node->removeAttribute('param-' . $name);
        }
        
        // NOTE: If the calling component tag has child elements then 
        // it is assumed that we are using an inline template
        if ($this->hasChildElements($node)) {
            $tmpDoc = new DOMDocument();
            $newNode = $tmpDoc->importNode($node, true);
            $tmpDoc->appendChild($newNode);
            $this->inlineDom = $tmpDoc;
        }
    }
    
    
    /**
     * does the node contain child elements
     * 
     * @param DOMElement $node
     * @return boolean
     */
    private function hasChildElements($node)
    {
        $children = $node->childNodes;
        foreach ($children as $child) {
            if ($child->nodeType == XML_ELEMENT_NODE) {
                return true;
            }
        }
        return false;
    }
    
    
    /**
     * Get the objects classname.
     *
     * @return string
     */
    function getClassname() 
    {
        return $this->classname;
    }
    
    /**
     * Get the parameters array.
     *
     * @return array
     */
    function getParameters() 
    {
        return $this->parameters;
    }
    
    /**
     * Get the insert var for this component into its parent 
     *
     * @return string
     */
    function getInsertVar() 
    {
        return $this->insertVar;
    }
    
    /**
     * Get the inline template data
     *
     * @return DOMDocument
     */
    function getInlineDom()
    {
        return $this->inlineDom;
    }
}

?>