<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * A web site front controller to take requests and serve responses.
 * 
 * @package Web
 */
class Dk_Web_SiteFrontController extends Dk_Web_FrontController 
{
    
    /**    
     * @var Dk_Web_ResourceMapper
     */
    protected $resourceMapper = null;
    
    /**    
     * @var Dk_Web_Component
     */
    protected $pageComponent = null;
    
    /**
     * @var Dk_Web_NodeModifier
     */
    protected $nodeModifier = null;
    
    /**
     * @var array
     */
    protected $controllerChain = array();
    
    
    
    /**
     * __construct
     * 
     * @param Dk_Web_ResourceMapper $resourceMapper (optional) Used for locating page templates
     */
    function __construct($resourceMapper = null)
    {
        if ($resourceMapper == null) {
            $resourceMapper = new Dk_Web_ResourceMapper(new Dk_Util_Path(Dk_Config::getInstance()->getHtmlTemplates()));
        }
        $this->resourceMapper = $resourceMapper;
        
        $this->nodeModifier = new Dk_Web_NodeModifier();
    }
    
    
    /**
     * Get the Master page component containing all other components
     *
     * @return Dk_Web_Component
     */
    function getPageComponent()
    {
    	return $this->pageComponent;
    }
    
    /**
     * Add a node modifier object to transpose the document after parsing
     *
     * @param Dk_Web_NodeModifierInterface $mod
     */
    function addNodeModifier(Dk_Web_NodeModifierInterface $mod)
    {
    	$this->nodeModifier->add($mod);
    }
    
    /**
     * Add a controller to be executed when the front controller is executed 
     * 
     *
     * @param Dk_Util_ControllerInterface $controller
     */
    function addController(Dk_Util_ControllerInterface $controller)
    {
    	$this->controllerChain[] = $controller;
    }
    
    /**
     * Helper method to execute the methods 'execute', 'init', 'postInit' methods
     */
    private function executeController($method)
    {
    	if (!ereg('^(execute|init|postInit)$', $method)) {
    		return;
    	}
    	foreach($this->controllerChain as $controller) {
    	   $controller->$method();
    	}
    }
    
    /**
     * pre init the front controller
     *
     */
    function init() 
    { 
        
    	// Get the page template path using the resource mapper
        $templatePath = $this->resourceMapper->getResourcePath(Dk_Request::getInstance()->getRequestUri());
        if ($templatePath == null) {
            $this->set404();
            return;
        }
        // Create page renderer
        $this->pageComponent = Dk_Web_ObjectFactory::getInstance()->createPage($templatePath);
        if ($this->pageComponent == null) {
            $this->set404();
            return;
        }
        // Check secure status if SSL enabled and iss an SSL page then redirect if required
        $isSecure = (Dk_Config::getInstance()->isSslEnabled() && $this->pageComponent->isSecure());
        $this->secureRedirect($isSecure, Dk_Request::getInstance()->getRequestUri());
        
    }
    
    /**
     * Post init the front controller
     * 
     */
    function postInit() 
    {
        if (Dk_Config::getInstance()->isDebugMode()) {
            $title = $this->pageComponent->getTemplate()->getTitleText();
            $this->pageComponent->getTemplate()->setTitleText('Debug: ' . $title);
        }
        $pageDoc = $this->pageComponent->getTemplate()->getDocument();
        
        // Check if data directory writable
        $dataDir = Dk_Config::getInstance()->getDataDir();
        if (!is_dir($dataDir) || !is_writable($dataDir)) {
            $body = $this->pageComponent->getTemplate()->getBodyElement();
            $div = $body->ownerDocument->createElement('div', "Dk-Warning: The directory `$dataDir' is not writable. Please use the command `chmod -R ugo+rw $dataDir`");
            $div->setAttribute('class', '__Dk_Warning');
            $div->setAttribute('style', 'font-size: 10px;border: 1px outset #ccc; background-color: #F99;padding: 2px 4px;font-family: arial,sans-serif;');
            $body->insertBefore($div, $body->firstChild);
        }
        
        // Do any node modifing
        $this->nodeModifier->add(new Dk_Web_NodeModifierPath());
        $this->nodeModifier->setDocument($pageDoc);
        $this->nodeModifier->execute();
    
        // Display any debug/var dumps to the template
        if (Dk_Config::getInstance()->isDebugMode()) {
            Dk_Util_Debugger::appendDump($pageDoc);
        }
    	
    }
    
    
    
    /**
     * Runs the post response
     *
     */
    function doPost()
    {
    	// init
        $this->init();
        $this->executeController('init');
        
        // Execute 
        $this->pageComponent->execute();
        $this->executeController('execute');
        $this->pageComponent->render();
        
        // Post init
        $this->postInit();
        $this->executeController('postInit');
        
        
        // parse XML/HTML
        $pageDoc = $this->pageComponent->getTemplate()->getDocument();
        $pageDoc->formatOutput = true;
        $pageHtml = $pageDoc->saveXml();
        
        // Javascript CDATA hack for php 5.0-5.1
        if (Dk_Config::getInstance()->getCdataFix()) {
            $pageHtml = str_replace(array('<![CDATA[', ']]>'), array('', ''), $pageHtml);
        }
        
        Dk_Response::getInstance()->write($pageHtml);
    }
    
    /**
     * run the GET response
     * In this case all responses are run through POST
     *
     */
    function doGet()
    {
        $this->doPost();
    }
    
    /**
     * Check the page and redirect to secure/unsecure as nessacery
     * 
     * @param boolean $isSecurePage
     * @param string $requestUri
     */
    function secureRedirect($isSecurePage, $requestUri) 
    {
        if ($requestUri->getScheme() == 'https' && !$isSecurePage) {
                $requestUri->setScheme('http');
                $requestUri->redirect();
        } elseif ($requestUri->getScheme() == 'http' && $isSecurePage) {
                $requestUri->setScheme('https');
                $requestUri->redirect();
        }
    }
    
    /**
     * Display a fatal error.
     *
     * @param Exception $e
     */
    function showFatalError(Exception $e)
    {
        Dk_Response::getInstance()->reset();
        $msg = '';
        $config = Dk_Config::getInstance();
        
        if ($config->isDebugMode()) {
            $msg = wordwrap($e->__toString(!$config->isDebugMode()), 180);
        } else {
            $msg = "Page Down.\nThe site administrator has been notified of the problem.\n\nPlease check back soon.";
        }
        
        Dk_Response::getInstance()->sendError(Dk_Response::SC_INTERNAL_SERVER_ERROR, $msg);
        
        // TODO: Send to a webservice or central database....
        $email = $config->getErrorEmail();
        if ($email && ereg(Sdk_Util_Regs::POSIX_EMAIL, $email) && !$config->isDebugMode()) {
            
            $msg = wordwrap($e->__toString(false), 180);
            $status = Dk_Response::SC_INTERNAL_SERVER_ERROR . ' ' . 
                      Dk_Response::$statusText[Dk_Response::SC_INTERNAL_SERVER_ERROR];
            $siteTitle = $config->getSiteTitle();
            $headers = "reply-to: " . $email . "\r\n";
            $headers .= "from: " . $email . "\r\n";
            mail($email, 'Fatal Error: ' . $status . ' - ' . $siteTitle, $msg, $headers);
        }
    }
    
    /**
     * Send a 404 error page.
     */
    protected function set404()
    {
        Dk_Response::getInstance()->sendError(Dk_Response::SC_NOT_FOUND, 
            "The requested URL " . Dk_Request::getInstance()->getRequestUri()->getPath() .
            " was not found on this server.");
    }
    
}
?>