<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * All widgets can be created from this object
 * 
 * @package Web
 */
class Dk_Web_ObjectFactory 
{
    
    /** 
     * @var Dk_Web_ObjectFactory
     */
    static protected $instance = null;
    
    
    /**
     * This is a constructor
     * If no request session or response parameters given the default Sdk objects are used.
     * 
     */
    protected function __construct() { }
    
    /**
     * Get an instance of the object factory
     * 
     * @return Dk_Web_ObjectFactory
     */
    static function getInstance() 
    {
        if (self::$instance == null) {
            self::$instance = new Dk_Web_ObjectFactory();
        }
        return self::$instance;
    }
    
    /**
     * Create a page/owner component
     *
     * @param Dk_Util_Path $path
     * @return Dk_Web_Component
     */
    function createPage(Dk_Util_Path $path) 
    {
        $page = new Dk_Web_Component();
        $page->setTemplate(Dk_Web_Template::load($path->getPath()));
        $this->createComponents($page);
        return $page;
    }
    
    /**
     * Create the page component list
     *
     * @param Dk_Web_Component $parent
     */
    private function createComponents(Dk_Web_Component $parent)
    {
        $componentList = $parent->getTemplate()->getComponentList();
        /* @var $metaData Dk_Web_ComMetaData */
        foreach ($componentList as $metaData) {
            $com = null;
            $method = 'create' . $metaData->getClassname();
            if (method_exists($this, $method)) {
                $com = $this->$method($metaData);    
            } else {
                $com = $this->createDefaultComponent($metaData);    
            }
            if ($com == null) {
                continue;
            }
            $template = $this->getDefaultTemplate($metaData);
            if ($template) {
                $com->setTemplate($template);
            }
            $parent->addChild($com, $metaData->getInsertVar());
            if ($com->getTemplate()) {
                $this->createComponents($com);
            }
            // TODO: Test this
            foreach ($metaData->getParameters() as $k => $v) {
                //$method = 'set' . strtolower($k);
                $method = 'set' . ucfirst($k);
                if (method_exists($com, $method)) {
                    $com->$method($v);
                }
            }
        }
    }


    /**
     * The default component factory method.
     *
     * This method can be overriden in sub classes, or a custom factory method
     * for a specific component can be added. The custom factory method must
     * accept the same parameters as the default method and the name must
     * follow the format of the classname prefixed with the string 'create',
     * for example:
     *
     * createDk_Modules_ImageGallery_Manager(Dk_Web_ComMetaData $metaData)
     *
     * @param Dk_Web_ComMetaData $metaData
     * @return Dk_Web_Component
     */
    private function createDefaultComponent(Dk_Web_ComMetaData $metaData)
    {
        $class = $metaData->getClassname();
        if (!class_exists($class)) {
            return;
        }
        return new $class();
    }
    
    /**
     * Create a default component template
     *
     * @param string $classname
     * @return Dk_Web_Template  Returns null if no tempalte exists
     */
    private function getDefaultTemplate(Dk_Web_ComMetaData $metaData)
    {
        $template = null;
        // Check for an inline template
        if ($metaData->getInlineDom()) {
            return new Dk_Web_Template($metaData->getInlineDom());
        }
        
        // Check for an external template
        $arr = explode('_', $metaData->getClassname());
        $n = count($arr);
        $src = $arr[$n-1];
        
        $language = 'en';
        if (Dk_Config::getInstance()->getLanguage() && Dk_Config::getInstance()->getLanguage() != 'en') {
              $language = Dk_Config::getInstance()->getLanguage();
        }
        array_pop($arr);
        $templateFile = implode('/', $arr);
        
        // Places to search for templates
        $tplFiles = array();
        $tplFiles[] = $templateFile . '/' . $src . '.xml';
        $tplFiles[] = $templateFile . '/templates/' . $src . '.xml';
        $tplFiles[] = $templateFile . '/templates/language/' . $language . '/' . $src . '.xml';
        
        foreach ($tplFiles as $templateFile) {
            $file = Dk_Config::getinstance()->getFileRoot().'/lib/'.$templateFile;
            if (is_file($file)) {
                $template = Dk_Web_Template::load($file);
                break;
            }
        }
        return $template;
    }
    
}
?>