<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * A base component object. 
 * 
 * 
 * @package Web
 */
class Dk_Web_Component extends Dk_Web_Renderer implements Dk_Util_ControllerInterface
{
    
    /**
     * @var array
     */
    private $events = array();
    
    /**
     * @var boolean
     */
    private $secure = false;
    
    
    /**
     * @var string
     */
    private $insertVar = '';
    
    /**
     * @var array
     */
    private $children = array();

    /**
     * @var Dk_Web_Component
     */
    private $parent = null;

    /**
     * @var Dk_Web_Component
     */
    private $page = null;
    
    /**
     * @var Dk_Form_Object
     */
    private $form = null;
    
    /**
     * If this is false the execute/render methods will exit and not run
     * @var boolean
     */
    private $enabled = true;
    
    
    
    
    
    /**
     * __construct
     * 
     * 
     */
    function __construct()
    {
        $this->id = self::createId();
        $this->page = $this;
        $this->insertVar = get_class($this);
    }
    
    /**
     * If enabled is set to false then the widget does not execute/render
     * 
     * @param boolean $b
     */
    function setEnabled($b)
    {
        $this->enabled = ($b==true);
    }
    
    /**
     * Get the enabled status of this component
     * 
     * @return boolean
     */
    function isEnabled()
    {
        return $this->enabled;
    }
    
    /**
     * An alias for $this->setEnabled(false); to exit a widget
     * 
     * <code>
     *   function init()
     *   {
     *     if (true) {
     *       return $this->disable();
     *     }
     *     ...
     *   }
     * 
     * </code>
     * 
     * @return boolean
     */
    function disable()
    {
        $this->setEnabled(false);
        return false;
    }
    
    
    /**
     * The Component Event Engine Lies HERE!
     * Execute this component and its children
     * Only call this on the parent/page component, usualy in
     * a front controller
     *
     * @return boolean
     */
    function execute()
    {
        if (!$this->isEnabled()) {
            return false;
        }
        $this->getCrumbStack()->init($this);
        $this->init();
        $eventExecuted = false;
        foreach ($this->events as $event => $method) {
            if ($this->getRequest()->exists($event)) {
                $this->executeEvent($method);
                $eventExecuted = true;
                break;
            }
        }
        if (!$eventExecuted) {
            $this->executeEvent('doDefault');
        }
        foreach ($this->children as $child) {
            $child->execute();
        }
        $this->postInit();
        return true;
    }
        
    /**
     * The Component Render Engine Lies HERE!
     * This fucntion calls all show methods and renders the widgets
     * Only call this on the parent/page component, usualy in
     * a front controller
     * 
     * @return boolean
     */
    function render()
    {
        if (!$this->isEnabled()) {
            return false;
        }
        foreach ($this->children as $child) {
            $child->render();
        }
        if ($this->form != null) {
            $formRenderer = new Dk_Form_Renderer($this->form);
            $formRenderer->show($this->getTemplate());
        }
        if ($this->parent) {
            $this->parent->insertRenderer($this, $this->getInsertVar());
        }
        return true;
    }
    
    
    
    
    /**
     * The default show method.
     * @param Dom_Template $template
     */
    function show($template = null) { }
    

    /**
     * The default init method
     *
     */
    function init() { }
    
    /**
     * Post init 
     * 
     */
    function postInit() { }
    
    
    /**
     * The default event handler.
     * 
     */
    function doDefault() { }
    
    /**
     * Execute this objects event and all sub object events.
     * 
     * @param string $method
     * @return mixed
     */
    private function executeEvent($method) 
    {
        if (method_exists($this, $method)) {
            return $this->$method();
        }
    }
    
    /**
     * Adds an event.
     *
     * Where $event is a parameter for the request. Events trigger the 
     * call to $method. For example, if $event = 'submit' and the $method = 'doSubmit' the
     * doSubmit() method will be called if submit is found in the request.
     *
     * When executing returns at the first event found in the request. If
     * there are no events or no events found in the request the the
     * doDefault() method is called.
     *
     * @param string $event The request parameter key/name
     * @param string $method The method to execute
     */
    function addEvent($event, $method)
    {
        $this->events[$event] = $method;
    }
    
    /**
     * Return true if this page is SSL enabled.
     * 
     * @return boolean
     */
    function isSecure()
    {
        foreach ($this->children as $child) {
            if ($child->isSecure()) {
                return true;
            }
        }
        return $this->secure;
    }
    
    /**
     * Set the SSL status of the page.
     * NOTE: It only takes one component to be secure 
     * then the page will be redirected to the https://... url if not so already
     * 
     * NOTE: For this to work you must have the SSL certificate installed 
     * to the same directory as main website. We hope to have Shared SSL 
     * in the future...
     * 
     * @see Dk_Web_SiteFrontController
     * @param boolean $b
     */
    function setSecure($b)
    {
        $this->secure = $b;
    }
    
    
    
    /**
     * Insert a Renderer template into the component
     *
     * @param Dom_RendererInterface $renderer
     */
    function insertRenderer(Dom_RendererInterface $renderer, $insertVar = '')
    {
        if ($renderer->getTemplate() == null) {
            $renderer->setTemplate($this->getTemplate());
        }
        $renderer->show($renderer->getTemplate());
        if ($renderer->getTemplate() != $this->getTemplate()) {
            if ($this->getTemplate()->keyExists('var', $insertVar)) {
                $this->getTemplate()->insertTemplate($insertVar, $renderer->getTemplate());
            }
            // TODO: If no var would it be good to append after the last child element
            
        }
    }
    
    /**
     * Add a child componet to this component
     * 
     *
     * @param Dk_Web_Component $component
     * @param string $var The template var where the child dom will be inserted.
     */
    function addChild(Dk_Web_Component $component, $insertVar = '')
    {
        $component->setParent($this);
        $component->setPage($this->page);
        if ($insertVar) {
            $component->insertVar = $insertVar;
        }
        $this->children[] = $component;
    }
    
    /**
     * Set the parent component
     *
     * @param Dk_Web_Component $component
     */
    protected function setParent(Dk_Web_Component $component)
    {
        $this->parent = $component;
    }
    
    /**
     * Get the parent component
     *
     * @return Dk_Web_Component
     */
    function getParent()
    {
        return $this->parent;
    }
    
    /**
     * Set the top most page component.
     *
     * @param Dk_Web_Component $component
     */
    protected function setPage(Dk_Web_Component $component)
    {
        $this->page = $component;
    }
    
    /**
     * Get the top most page component.
     * This is the base XHTML Template component. Use this component to 
     * set any elements on the page template. 
     * The code would look like:
     * <code>
     *   $component->getPage()->getTemplate()->replaceText('var', 'text');
     * </code>
     *
     * @return Dk_Web_Component
     */
    function getPage()
    {
        return $this->page;
    }
    
    /**
     * Is this the owner/page component.
     *
     * @return boolean
     */
    function isPage()
    {
        return ($this->page == null);
    }
    
    /**
     * Get the insert var name for the template
     * we will be inserting this component into
     *
     * @return string
     */
    function getInsertVar()
    {
        return $this->insertVar;
        
    }

    /**
     * Set a form for this component. if set then the form is rendered automatically
     *
     * @param Dk_Form_Object $form
     */
    function setForm(Dk_Form_Object $form)
    {
        $this->form = $form;
    }
    
    /**
     * Return the component form object.
     * 
     * @return Dk_Form_Object Returns null if not set
     */
    function getForm()
    {
        return $this->form;
    }
    

    /**
     * Get the widget class session context. This is a map that contains session data
     * for a widget class, not an instance so all class instances have access to this data.
     *
     * @TODO: have context types eg: TYPE_PAGE, TYPE_CLASS, TYPE_INSTANCE to allow for different session scopes
     * @return ArrayObject
     */
    function getSessionContext()
    {
        $context = new ArrayObject();
        if ($this->getSession()->exists('context_'.get_class($this)) && 
            $this->getSession()->getParameter('context_'.get_class($this)) instanceof ArrayObject) 
        {
            $context = $this->getSession()->getParameter('context_'.get_class($this));
        } else {
            $this->getSession()->setParameter('context_'.get_class($this), $context);
        }
        return $context;
    }
    

    /**
     * Get the request key for an event. This will include the component id
     * This can be nessasery to avoid event collisions when using multiple 
     * instances of a component.
     *
     * @param string $event
     * @return string
     */
    function getEventKey($event)
    {
        return $event . '_' . $this->getId();
    }
    

    /**
     * Get the user base directory for creating urls.
     *
     * @return Dk_User_Interface
     */
    function getUser()
    {
        return Dk_User_Auth::getInstance()->getUser();
    }

    /**
     * Get the user base directory for creating urls.
     * 
     * @return string
     */
    function getUserDir()
    {
        if ($this->getUser()) {
            return dirname($this->getUser()->getHomeUrl()->toString());
        } else {
            return '/';
        }
    }

    /**
     * Add a top crumb for this component
     * 
     * @param Dk_Util_Url $url
     * @param string $name
     */
    function addCrumbUrl(Dk_Util_Url $url, $name = '')
    {
        if ($this->getParent() != $this->getPage()) {
            return;
        }
        if ($this->getRequest()->getRequestUri()->getBasename() != $url->getBasename()) {
            if ($name == 'Home') {
                $this->getCrumbStack()->reset();
            }
            $this->getCrumbStack()->putUrl($url, $name);
        }
        
    }
    
    /**
     * Get the topmost crumb url
     * 
     * @return Dk_Util_Url
     */
    function getCrumbUrl()
    {
        $url = $this->getCrumbStack()->getCurrent();
        if ($url == null) {
            $url = new Dk_Util_Url('index.html');
        }
        return $url;
    }
    
    
    /**
     * Get the sites config object
     *
     * @return Dk_Config
     */
    function getConfig()
    {
        return Dk_Config::getInstance();
    }
    
    /**
     * Get the request object. 
     *
     * @return Dk_Request
     */
    function getRequest()
    {
        return Dk_Request::getInstance();
    }
    
    /**
     * Get the current session object
     *
     * @return Dk_Session
     */
    function getSession()
    {
        return Dk_Session::getInstance();
    }
    
    /**
     * Get the crumbs object
     *
     * @return Dk_Util_CrumbStack
     */
    function getCrumbStack()
    {
        return Dk_Util_CrumbStack::getInstance();
    }
    
    

    /**
     * A factory method to generate new component id
     * This will allow for the id to be created in the constructor
     * 
     * @param boolean $incrementIdx Set this to false to give child components the same ID as it parent
     * @return integer
     */
    static function createId()
    {
        static $idx = 0;
        return $idx++;
    }
}
?>
