<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @author Darryl Ross <darryl.ross@aot.com.au>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * A URL class.
 * 
 * <b>[[&lt;scheme&gt;://][[&lt;user&gt;[:&lt;password&gt;]@]&lt;host&gt;[:&lt;port&gt;]]][/[&lt;path&gt;][?&lt;query&gt;][#&lt;fragment&gt;]]</b>
 * 
 * where:<br/>
 *  o scheme defaults to http <br/>
 *  o host defaults to the current host.<br/>
 *  o port defaults to 80<br/>
 * 
 * The spec must be for an absolute Url (with a scheme), or start with 'www'
 * or '/'.
 * 
 * If the spec starts with '/' then the base Url defaults to 
 * $_SERVER['HOST_NAME'] . Url::$pathPrefix
 * 
 * Note: There is no support for relative paths.
 *
 * @package Util
 */
class Dk_Util_Url extends Dk_Object
{
    /**
     * A prefix to append to path.
     *
     * Useful for when working in a dev enviroment, where the application is
     * not installed in the docroot of the domain.
     * @var string
     */
    static $pathPrefix = '';
    /**
     * For saving the path prefix for sessions
     */
    private $sesPrefix = '';
    
    
    
    /**
     * This is the supplied full url
     * @var string
     */
    private $spec = '';
    
    
    /**
     * @var boolean
     */
    private $relative = false;
    /**
     * @var string
     */
    private $fragment = '';
    /**
     * @var string
     */
    private $host = '';
    /**
     * @var string
     */
    private $password = '';
    /**
     * @var string
     */
    private $path = '';
    /**
     * @var string
     */
    private $port = '80';
    /**
     * @var aray
     */
    private $query = array();
    /**
     * @var string
     */
    private $scheme = 'http';
    /**
     * @var string
     */
    private $user = '';
    
    
    /**
     * 
     * 
     * @param string $spec The String to parse as a URL
     */
    function __construct($spec)
    {
        $this->spec = $spec;
        $this->init();
    }
    
    /**
     * Initalise the url object
     */
    private function init()
    {
        $spec = $this->spec;
        $host = 'localhost';
        if ($spec == null || $spec == 'http://') {
            $spec = 'http://localhost/';
        }
        if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on') {
            $this->scheme = 'https';
        }
        if (isset($_SERVER['HTTP_X_FORWARDED_HOST'])) {
            $host = $_SERVER['HTTP_X_FORWARDED_HOST'];
        } else if (isset($_SERVER['HTTP_HOST'])) {
            $host = $_SERVER['HTTP_HOST'];
        }
        if (!eregi("^(http|https|ftp|file|gopher|news)", $spec) ) {
            if (strtolower(substr($spec, 0, 3)) == 'www') {
                $spec = $this->scheme . '://' . $spec;
            } else {
                if ($spec != '' && $spec{0} != '/') {
                    $spec = '/' . $spec;
                }
                $spec = $this->scheme . '://' . $host . $spec;
                $this->relative = true;
            }
        }

        $components = parse_url($spec);
        if ($components) {
            if (array_key_exists('scheme', $components)) {
                $this->scheme = $components['scheme'];
            }
            if (array_key_exists('host', $components)) {
                $this->host = $components['host'];
            }
            if (array_key_exists('port', $components)) {
                $this->port = $components['port'];
                if ($this->relative) {
                    $this->port = $_SERVER['SERVER_PORT'];
                }
            }
            if (array_key_exists('user', $components)) {
                $this->user = $components['user'];
            }
            if (array_key_exists('pass', $components)) {
                $this->password = $components['pass'];
            }
            if (array_key_exists('path', $components)) {
                $this->path = $components['path'];
                if ($this->host == '' && substr($this->path, 0, 3) == 'www') {
                    $this->host = $this->path;
                    $this->path = '';
                }
            }
            if (array_key_exists('query', $components)) {
                parse_str($components['query'], $this->query);
            }
            
            if (array_key_exists('fragment', $components)) {
                $this->fragment = urldecode($components['fragment']);
            }
    
            if ($this->path != '' && $this->path{0} == '.') {
                $this->path = substr($this->path, 1);
            }
            if ($this->path != '' && $this->path{0} != '/') {
                $this->path = '/' . $this->path;
            }
        }

        if (self::$pathPrefix != '' && $this->relative) {
            $len = strlen(self::$pathPrefix);
            if (substr($this->path, 0, $len) != self::$pathPrefix) {
                $this->path = self::$pathPrefix . $this->path;
            }
        }
        $this->sesPrefix = self::$pathPrefix;

        if ($this->host == '') {
            $this->host = $host;
        }
    }

    /**
     * on serialise 
     * 
     * @return array
     */
    public function __sleep()
    {
        $this->init();
        return array('spec', 'sesPrefix');
    }
    
    /**
     * on unserialise
     * 
     */
    public function __wakeup()
    {
        self::$pathPrefix = $this->sesPrefix;
        $this->init();
    }
    
    /**
     * Normalize the url, this is as if the url had been reloaded
     * Useful for when the object is serialised/unserialised
     */
    function normlize()
    {
        $this->spec = $this->toString();
        $this->init();
    }
    
    
    /**
     *  
     * @return string
     */
    function getFragment()
    {
        return $this->fragment;
    }
    
    /**
     * 
     * @param string $str
     */
    function setFragment($str)
    {
        $this->fragment = $str;
    }
    
    /**
     *  
     * @return string
     */
    function getHost()
    {
        return $this->host;
    }

    /**
     *  
     * @return string
     */
    function getPassword()
    {
        return $this->password;
    }

    /**
     *  
     * @return string
     */
    function getPath()
    {
        return $this->path;
    }

    /**
     *  
     * @return string
     */
    function getPort()
    {
        return $this->port;
    }

    /**
     *  
     * @return string
     */
    function getQuery()
    {
        $query = '';
        foreach ($this->query as $field => $value) {
            if (is_array($value)) {
                foreach ($value as $v) {
                    $query .= urlencode($field) . '[]=' . urlencode($v) . '&';
                }
            } else {
                $query .= urlencode($field) . '=' . urlencode($value) . '&';
            }
        }
        $query = substr($query, 0, -1);
        return $query;
    }

    /**
     *  
     * @param string $scheme
     */
    function setScheme($scheme)
    {
        $this->scheme = $scheme;
    }

    /**
     *  
     * @return string
     */
    function getScheme()
    {
        return $this->scheme;
    }

    /**
     *  
     * @return string
     */
    function getUser()
    {
        return $this->user;
    }
    
    
    
    
    /**
     * Add a field to the query string
     *
     * @param string $field
     * @param string $value
     */
    function setQueryField($field, $value)
    {
        $this->query[$field] = $value;
    }

    /**
     * Remove a field in the querystring
     *
     * @param string $field
     */
    function removeQueryField($field)
    {
        if (array_key_exists($field, $this->query)) {
            unset($this->query[$field]);
        }
    }
    
    /**
     * clear and reset the querystring
     * 
     */
    function resetQueryFields()
    {
       $this->query = array(); 
    }
    
    /**
     * Get the array of queryfields in a map
     * 
     * @return array 
     */
    function getQueryFields()
    {
        return $this->query;
    }
    
    /**
     * Get the array of queryfields in a map
     * 
     * @return array
     */
    function setQueryFields($map)
    {
        if ($map != null) {
            $this->query = $map;
        }
    }
    
    /**
     * Get a value from teh query string.
     *
     * @param string $field
     * @return string
     */
    function getQueryFieldValue($field)
    {
        if (array_key_exists($field, $this->query)) {
            return $this->query[$field];
        } else {
            return null;
        }
    }
    
    /**
     * Returns file extension for this pathname. 
     *
     * A the last period ('.') in the pathname is used to delimit the file
     * extension .If the pathname does not have a file extension null is 
     * returned.
     * 
     * @return string
     */
    function getExtension()
    {
        $pos = strrpos(basename($this->getPath()), '.');
        if ($pos) {
            return substr(basename($this->getPath()), $pos + 1);
        }
        return '';
    }
    
    /**
     * Get the basename of this url.
     *
     * @return string
     */
    function getBasename()
    {
        return basename($this->getPath());
    }
    
    /**
     * redirect
     * 
     * Codes:
     * 
     *  301: Moved Permanently
     *
     *    - The requested resource has been assigned a new permanent URI and any 
     *      future references to this resource SHOULD use one of the returned URIs. 
     *      Clients with link editing capabilities ought to automatically re-link 
     *      references to the Request-URI to one or more of the new references 
     *      returned by the server, where possible. This response is cacheable 
     *      unless indicated otherwise.
     * 
     *  302: Found
     * 
     *    - The requested resource resides temporarily under a different URI. Since 
     *      the redirection might be altered on occasion, the client SHOULD continue to 
     *      use the Request-URI for future requests. This response is only cacheable 
     *      if indicated by a Cache-Control or Expires header field.
     * 
     *  303: See Other
     * 
     *    - The response to the request can be found under a different URI and SHOULD 
     *      be retrieved using a GET method on that resource. This method exists primarily 
     *      to allow the output of a POST-activated script to redirect the user agent 
     *      to a selected resource. The new URI is not a substitute reference for 
     *      the originally requested resource. The 303 response MUST NOT be cached, 
     *      but the response to the second (redirected) request might be cacheable.
     * 
     *  304: Not Modified
     * 
     *    - If the client has performed a conditional GET request and access is allowed, 
     *      but the document has not been modified, the server SHOULD respond with this 
     *      status code. The 304 response MUST NOT contain a message-body, and thus is 
     *      always terminated by the first empty line after the header fields.
     * 
     *  305: Use Proxy
     * 
     *    - The requested resource MUST be accessed through the proxy given by the Location 
     *      field. The Location field gives the URI of the proxy. The recipient is expected 
     *      to repeat this single request via the proxy. 305 responses MUST only be 
     *      generated by origin servers.
     * 
     *  306: (Unused)
     * 
     *    - The 306 status code was used in a previous version of the specification, is
     *      no longer used, and the code is reserved.
     * 
     *  307: Temporary Redirect
     * 
     *    - The requested resource resides temporarily under a different URI. Since the 
     *      redirection MAY be altered on occasion, the client SHOULD continue to use the 
     *      Request-URI for future requests. This response is only cacheable if indicated 
     *      by a Cache-Control or Expires header field.
     * 
     * 
     *  
     * 
     * func: redirect($to,$code=307)
     * @link http://www.w3.org/Protocols/rfc2616/rfc2616-sec10.html
     * @link http://edoceo.com/creo/php-redirect.php
     */
    function redirect($code = 301)
    {
        $response = Dk_Response::getInstance();
        $response->reset();
        $location = $this->toString();
        $hs = headers_sent();
        if ($hs == false) {
            switch ($code) {
                case 301:
                case 302:
                case 303:
                case 304:
                case 307:
                    $response->addHeader('Status', $code);
                    break;
                default:
                    throw new Dk_ExceptionIllegalArgument("Unhandled redirect() HTTP Code: $code", E_USER_ERROR);
                    break;
            }
            $response->addHeader('Location', $location);
            $response->addHeader('Cache-Control', 'no-store, no-cache, must-revalidate, post-check=0, pre-check=0');
            $response->flushBuffer();
        } elseif (($hs == true) || ($code == 302) || ($code == 303)) {
            throw new Dk_ExceptionIllegalArgument("Headers Allready Sent.", E_USER_ERROR);
        }
        exit();
    }
    
    
    /**
     * Return a string representation of this object
     *
     * @return string
     */
    function toString()
    {
        if (eregi("^(javascript|mailto)", $this->spec)) {
            return $this->spec;
        }
        $url = '';
        if ($this->scheme != '') {
            $url .= $this->scheme . '://';
        }
        if ($this->user != '' || $this->password != '') {
            $url .= $this->user . ':' . $this->password . '@';
        }
        if ($this->host != '') {
            $url .= $this->host;
            if ($this->port != 80) {
                $url .= ':' . $this->port;
            }
        }
        if ($this->path != '') {
            $url .= $this->path;
        }
        $query = $this->getQuery();
        if ($query != '') {
            $url .= '?' . $query;
        }
        if ($this->fragment != '') {
            $url .= '#' . $this->fragment;
        }
        return $url;
    }
    
    /**
     * Return a string representation of this object without the host portion
     *
     * @return string
     */
    function toUriString()
    {
        $url = '';
        if ($this->path != '') {
            $url .= $this->path;
            if (Dk_Config::getInstance()->isDebugMode()) {
                $url = str_replace(self::$pathPrefix, '', $url);
            }
        }
        
        $query = '';
        foreach ($this->query as $field => $value) {
            if ($value) {
                $query .= $field . '=' . urlencode($value) . '&';
            }
        }
        $query = substr($query, 0, -1);
        if ($query != '') {
            $url .= '?' . $query;
        }
        
        if ($this->fragment != '') {
            $url .= '#' . $this->fragment;
        }
        return $url;
    }
}
?>