<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * A registry object.
 * 
 * @package Util
 */
class Dk_Util_Registry extends Dk_Object
{
    
    /**
     * @var array
     */
    private $cache = array();
    
    
    
    /**
     * Parse a config file either XML or INI
     *
     * @param string $file
     * @throws Dk_ExceptionRuntime
     */
    function parseConfigFile($file) 
    {
        if ($file == '' || !is_file($file)) {
            return;
        }
        $arr = explode('.', $file);
        $ext = $arr[count($arr)-1];
        if ($ext == 'ini') {
            $array = $this->parseIniFile($file);
        } elseif ($ext == 'cfg' || $ext == 'xml') {
            $array = $this->parseXmlFile($file);
        } else {
            throw new Dk_ExceptionRuntime('Invalid config file: ' . $file);
        }
        $this->load($array);
    }
    
    /**
     * Load the values using the setter methods if available
     * NOTE: The array must be a one dimensional array
     *
     * @param array $array The array of parameters to find setters for
     */
    function load($array) 
    {
        if (!count($array)) {
            return;
        }
        foreach ($array as $k => $v) {
            $method = 'set'.ucfirst($k);
            if (method_exists($this, $method)) {
                $this->$method($v);
            }else{
                $this->setEntry($k, $v);
            }
        }
    }
    
    /**
     * Read and apply the ini file to the registry.
     *
     * @param string $iniFile
     */
    private function parseIniFile($iniFile) 
    {
        $array = parse_ini_file($iniFile, false);
        return $array;
    }
    
    /**
     * Parse a config xml file.
     * Example Config File:
     *  <code>
     *    <config>
     *     <name1>value1</name1>
     *     <name2>value2</name2>
     *    </config>
     * </code>
     * 
     * @param string $xmlFile
     */
    private function parseXmlFile($xmlFile) 
    {
        $doc = DOMDocument::load($xmlFile);
        $firstChild = $doc->documentElement;
        $array = array();
        foreach($firstChild->childNodes as $node) {
            if ($node->nodeType != XML_ELEMENT_NODE) {
                continue;
            }
            $k = $node->nodeName;
            $v = $node->nodeValue;
            $type = 'string';
            if ($node->hasAttribute('type')) {
                $type = $node->getAttribute('type');
            }
            if ($type == 'boolean') {
                $v = (strtolower($v) == 'true');
            }
            $array[$k] = $v;
        }
        return $array;
    }
    
    /**
     * Set an entry into the registry cache
     *
     * @param string $key
     * @param object $item
     */
    function setEntry($key, $item) 
    {
        $key = ucfirst($key);
        $this->cache[$key] = $item;
    }
    
    /**
     * Return an entry from the registry cache
     *
     * @param string $key
     * @return object
     */
    function getEntry($key) 
    {
        $key = ucfirst($key);
        if (array_key_exists($key, $this->cache)) {
            return $this->cache[$key];
        }
    }
    
    /**
     * Test if an entry exists and is not null
     *
     * @param string $key
     * @return boolean
     */
    function isEntry($key) 
    {
        $key = ucfirst($key);
        return ($this->getEntry($key) !== null);
    }
    
    /**
     * Return a string representation of the registry object
     * 
     * @return string
     */
    function toString()
    {
        $str = "";
        foreach ($this->cache as $k => $v) {
            if (is_object($v)) {
                $str .= "[$k] => {" . get_class($v) . "}\n";
            } elseif (is_array($v)) {
                $str .= "[$k] =>  {Array(" . count($v) . ")}\n";
            } else {
                $str .= "[$k] => $v\n";
            }
        }
        return $str;
    }
}
?>