<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * An abstract representation of file and directory pathnames.
 *
 * @package Util
 */
class Dk_Util_Path extends Dk_Object
{
    /**
     * A prefix to append to path.
     * 
     * Useful for when working in a dev enviroment, where the application is
     * not installed in the docroot of the domain. 
     * This should be set to the $config->getSiteFileRoot() path if used.
     * 
     * @var string
     */
    static $pathPrefix = '';
    
    /**
     * @var string
     */
    private $pathname = '';
    
    
    /**
     * Create a Path object 
     * 
     * @param string $pathname
     */
    function __construct($pathname)
    {
        if (substr($pathname, -1) == '/') {
            $pathname = substr($pathname, 0, -1); 
        }
        $pathname = str_replace('//', '/', $pathname);
        $this->pathname = $pathname;
    }
    
    /**
     * Return a path object with the full path of a relative 
     * file/directory from the site root.
     * 
     * return Dk_Util_Path
     */
    static function createFromRalative($pathname)
    {
        return new Dk_Util_Path(self::$pathPrefix . $pathname);
    }
    
    
    /**
     * Checks whether the file or directory denoted by this pathname exists. 
     * @return boolean
     */
    function exists()
    {
        return file_exists($this->pathname);
    }

    /**
     * Returns file extension for this pathname. 
     *
     * @return string
     */
    function getExtension()
    {
        return self::getFileExtension($this->pathname);
    }

    /**
     * Returns file extension for this pathname. 
     *
     * A the last period ('.') in the pathname is used to delimit the file
     * extension. If the pathname does not have a file extension an empty string is returned.
     * 
     * @return string
     */
    static function getFileExtension($file)
    {
        $pos = strrpos($file, '.');
        if ($pos) {
            return strtolower(substr($file, $pos + 1));
        }
        return '';
    }

    /**
     * Returns the pathname. 
     *
     * @return string
     */
    function getPath()
    {
        return $this->pathname;
    }
    
    /**
     * This function returns a path with the self::$pathPrefix prepended
     * Should only be callued when using paths that are relitive from 
     * the site root.
     * 
     * @return Dk_Util_Path
     */
    function getFullPath()
    {
        return new self(self::$pathPrefix . $this->pathname);
    }
    
    /**
     * Get the pathname without the $pathPrefix prepended.
     * If $pathPrefix is null then the entire path is returned.
     * This is the path relative to the site root.
     * 
     * @return string
     */
    function getSiteRootPath()
    {
        if (strpos($this->pathname, self::$pathPrefix) !== false) {
            return str_replace(self::$pathPrefix, '', $this->pathname);
        }
        return $this->pathname;
    }

    /**
     * Returns the size of the file in bytes.
     *
     * If pathname does not exist or is not a file, 0 is returned.
     *
     * @return integer
     */
    function getSize()
    {
        if ($this->isFile()) {
            return filesize($this->pathname);
        }
        return 0;
    }

    /**
     * Checks whether this pathname is a directory. 
     * 
     * @return boolean
     */
    function isDir()
    {
        return is_dir($this->pathname);
    }

    /**
     * Checks whether this pathname is a regular file. 
     * 
     * @return boolean
     */
    function isFile()
    {
        return is_file($this->pathname);
    }

    
    /**
     * return the dirname of the path
     * 
     * @return Dk_Util_Path
     */
    function getDirname()
    {
        return new Dk_Util_Path(dirname($this->pathname));
    }
    
    /**
     * Return the base name of the path
     * 
     * @return Dk_Util_Path
     */
    function getBasename()
    {
        return new Dk_Util_Path(basename($this->pathname));
    }
    
    
    /**
     * Clean a filename 
     * 
     * @param string $file
     * @return string
     */
    static function cleanFilename($file)
    {
        $file = str_replace(' ', '_', $file);
        return preg_replace("/[^a-z0-9_\\-\\.\\/]/i", '_', $file);
    }
    
    
    /**
     * Get the bytes from a string like 40M, 10T, 100K
     *
     * @param string $str
     * @return integer
     */
    static function string2Bytes($str)
    {
        $sUnit = substr($str, -1);
        $iSize = (int) substr($str, 0, -1);
        switch (strtoupper($sUnit))
        {
            case 'Y' : $iSize *= 1024; // Yotta
            case 'Z' : $iSize *= 1024; // Zetta
            case 'E' : $iSize *= 1024; // Exa
            case 'P' : $iSize *= 1024; // Peta
            case 'T' : $iSize *= 1024; // Tera
            case 'G' : $iSize *= 1024; // Giga
            case 'M' : $iSize *= 1024; // Mega
            case 'K' : $iSize *= 1024; // kilo
        };
        return $iSize;
    }
    
    /**
     * Convert a value from bytes to a human readable value
     * 
     * @param integer $bytes
     * @return string
     * @author http://php-pdb.sourceforge.net/samples/viewSource.php?file=twister.php
     */
    static function bytes2String($bytes)
    {
        $tags = array('b', 'kB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB');
        $index = 0;
        while ($bytes > 999 && isset($tags[$index + 1])) {
            $bytes /= 1024;
            $index ++;
        }
        $rounder = 1;
        if ($bytes < 10) {
            $rounder *= 10;
        }
        if ($bytes < 100) {
            $rounder *= 10;
        }
        $bytes *= $rounder;
        settype($bytes, 'integer');
        $bytes /= $rounder;
        
        return $bytes . ' ' . $tags[$index];
    }
    
    /**
     * The trouble is the sum of the byte sizes of the files in your directories 
     * is not equal to the amount of disk space consumed, as andudi points out. 
     * A 1-byte file occupies 4096 bytes of disk space if the block size is 4096. 
     * Couldn't understand why andudi did $s["blksize"]*$s["blocks"]/8. 
     * Could only be because $s["blocks"] counts the number of 512-byte disk 
     * blocks not the number of $s["blksize"] blocks, so it may as well 
     * just be $s["blocks"]*512. Furthermore none of the dirspace suggestions allow 
     * for the fact that directories are also files and that they also consume disk 
     * space. The following code dskspace addresses all these issues and can also 
     * be used to return the disk space consumed by a single non-directory file. 
     * It will return much larger numbers than you would have been seeing with 
     * any of the other suggestions but I think they are much more realistic
     *
     * @param string $dir
     * @return integer
     */
    static function diskSpace($dir)
    {
        if (is_dir($dir)) {
            $s = stat($dir);
        }
        //$space = $s["blocks"] * 512;  // Does not work value $s["blocks"] = -1 allways
        if (!isset($s['size'])) {
        	 return 0;
        }
        $space = $s["size"];
        if (is_dir($dir)) {
            $dh = opendir($dir);
            while (($file = readdir($dh)) !== false) {
                if ($file != "." and $file != "..") {
                    $space += self::diskSpace($dir."/".$file);
                }
            }
            closedir($dh);
        }
        return $space;
    }
    

    /**
     * Get a string representation of this object
     *
     * @return string
     */
    function toString()
    {
        return $this->pathname;
    }
    
}
?>
