<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @author Darryl Ross <darryl.ross@aot.com.au>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * The Money object.
 * 
 * @package Util
 */
class Dk_Util_Money extends Dk_Object
{
    
    /**
     * Default Currency code
     * @var string
     */
    public static $defaultCurrencyCode = 'AUD';
    
    /**
     * @var integer
     */
    private $amount = 0;
    
    /**
     * @var string
     */
    private $currencyCode = 'AUD';
    
    /**
     * @var Dk_Util_Currency
     */
    private $currency = null;
    
    
    /**
     * 
     * 
     * @param integer $amount The amount in cents.
     * @param Dk_Util_Currency $currency The currency, Default 'AUD'.
     */
    function __construct($amount, $currency = null) 
    {
        $this->amount = intval($amount);
        if ($currency == null) {
            $this->currency = Dk_Util_Currency::getInstance(self::$defaultCurrencyCode);
        }else{
            $this->currency = $currency;
        }
    }
    
    /**
     * Create a money object from a string
     *
     * @param string $amount An amount string: '200.00', '$200.00'
     * @return Dk_Util_Money Returns null on invalid format
     */
    static function parseFromString($amount, $currency = null) 
    {
        if (!($currency instanceof Dk_Util_Currency)) {
            $currency = Dk_Util_Currency::getInstance(self::$defaultCurrencyCode);
        }else{
            $currency = $currency;
        }
        $digits = $currency->getDefaultFractionDigits();
        if (!ereg("^(\$)?(\-)?[0-9]+((\.)[0-9]{1,{$digits}})?$",$amount)) {
            return null;
        }
        $amount = str_replace(array(',', '$'), array('',''), $amount);
        $amount = floatval($amount);
        return new Dk_Util_Money($amount * 100, $currency);
    }
    
    /**
     * Serialise Read
     * 
     */
    function __wakeup() 
    {
        $this->currency = Dk_Util_Currency::getInstance($this->currencyCode);
    }
    
    /**
     * Serialise Write.
     * 
     * @return array
     */
    function __sleep()
    {
        $this->currencyCode = $this->currency->getCurrencyCode();
        $class = "\0".__CLASS__."\0";
        return array($class."amount", $class."currencyCode");
    }
    
    /**
     * Returns the amount in cents.
     * 
     * @return integer
     */
    function getAmount()
    {
        return $this->amount;
    }
    
    /**
     * Returns the currency.
     * 
     * @return Dk_Util_Currency
     */
    function getCurrency()
    {
        return $this->currency;
    }
    
    /**
     * Adds the value of another instance of money and returns a new instance.
     * 
     * @param Dk_Util_Money $other
     * @return Dk_Util_Money
     */
    function add(Dk_Util_Money $other)
    {
        $this->assertSameCurrencyAs($other);
        return new Dk_Util_Money($this->amount + $other->amount);
    }
    
    /**
     * Subtracts the value of another instance of money and returns a new instance.
     * 
     * @param Dk_Util_Money $other
     * @return Dk_Util_Money
     */
    function subtract(Dk_Util_Money $other)
    {
        $this->assertSameCurrencyAs($other);
        return new Dk_Util_Money($this->amount - $other->amount);
    }
    
    /**
     * Divide the amount by the denominator.
     *
     * @param float $denominator
     * @throws Dk_IllegalArgumentException
     */
    function divideBy($denominator) 
    {
        if ($denominator === 0){
            throw new Dk_IllegalArgumentException('Divide by zero exception.');
        }
        return new Dk_Util_Money($this->amount / $denominator);
    }
    
    /**
     * Compares the value to another instance of money.
     * 
     * @param Dk_Util_Money $other
     * @return integer Returns the difference, 0 = equal.
     */
    function compareTo(Dk_Util_Money $other)
    {
        $this->assertSameCurrencyAs($other);
        return $this->getAmount() - $other->getAmount();
    }
    
    /**
     * Multiplies the value of the money by an amount and returns a new instance.
     *
     * @param double $multiplyer
     * @return Dk_Util_Money
     */
    function multiply($multiplyer)
    {
        return new Dk_Util_Money(intval($this->amount * $multiplyer), $this->currency);
    }
    
    /**
     * Checks if the money value is greater than the value of another instance of money.
     * 
     * @param Dk_Util_Money $other
     * @return boolean
     */
    function greaterThan(Dk_Util_Money $other)
    {
        return $this->compareTo($other) > 0;
    }
    
    /**
     * Checks if the money value is greater than or equal the value of another instance of money.
     * 
     * @param Dk_Util_Money $other
     * @return boolean
     */
    function greaterThanEqual(Dk_Util_Money $other)
    {
        return ($this->compareTo($other) > 0) || ($other->amount == $this->amount);
    }
    
    /**
     * Checks if the money value is less than the value of another instance of money.
     * 
     * @param Dk_Util_Money $other
     * @return boolean
     */
    function lessThan(Dk_Util_Money $other)
    {
        return $this->compareTo($other) < 0;
    }
    
    /**
     * Checks if the money value is less than or equal the value of another instance of money.
     * 
     * @param Dk_Util_Money $other
     * @return boolean
     */
    function lessThanEqual(Dk_Util_Money $other)
    {
        return ($this->compareTo($other) < 0) || ($other->amount == $this->amount);
    }
    
    /**
     * Checks if the money value is equal to the value of another instance of money.
     * 
     * @param Dk_Util_Money $other
     * @return boolean
     */
    function equals(Dk_Util_Money $other)
    {
      return ($this->compareTo($other) == 0);
    }
    
    /**
     * Return a formatted string to the nearest dollar representing the currency
     * 
     * @return string
     */
    function toNearestDollarString()
    {
        $amount = round(($this->getAmount() / 100) + .205);
        return $this->currency->getSymbol() . $amount;
    }
    
    /**
     * Return a string amount as a 2 point presision float. Eg: '200.00'
     *
     * @return string
     */
    function toFloatString() 
    {
        return sprintf("%.02f", ($this->getAmount()/100));
    }
    
    /**
     * Return a formatted string representing the currency
     * 
     * @return string
     */
    function toString($decSep = '.', $thousandthSep = ',') 
    {
        $strValue = $this->currency->getSymbol($this->getCurrency()->getCurrencyCode()) . number_format(($this->getAmount()/100), $this->currency->getDefaultFractionDigits(), $decSep, $thousandthSep);
        return $strValue;
    }
    
    /**
     * Test for the same currency instance
     * 
     * @param Dk_Util_Money $arg 
     * @throws Dk_Exception
     */
    private function assertSameCurrencyAs(Dk_Util_Money $arg)
    {
        if ($this->currency !== $arg->currency) {
            throw new Dk_Exception('Money math currency instance mismatch.');
        }
    }
    
}
?>
