<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @author Darryl Ross <darryl.ross@aot.com.au>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * Dk_Util_Image
 * 
 * @package Util
 */
class Dk_Util_Image extends Dk_Util_Path 
{
    
    
    /**
     * Return a GD image from a source file
     * 
     * @param Dk_Util_Path $file
     * @return resource A GD image resource
     * @throws Dk_ExceptionRuntime On unsuported file type
     * @throws Dk_ExceptionIllegalArgument
     */
    static function createImagefromFile(Dk_Util_Path $file)
    {
        if (!$file->exists()) {
            throw new Dk_ExceptionRuntime("Source image does not exsit: `{$file->getPath()}`");
        }
    
        $image = null;
        switch (strtolower($file->getExtension())) {
            case 'jpeg':
            case 'jpg':
                $image = imagecreatefromjpeg($file->getPath());
                break;
            case 'gif':
                $image = imagecreatefromgif($file->getPath());
                break;
            case 'png':
                $image = imagecreatefrompng($file->getPath());
                break;
            default:
                throw new Dk_ExceptionIllegalArgument("File type unsupported: `{$file->getExtension()}`");
        }
        return $image;
    }
    
    
    /**
     * Save an image to a destination
     * 
     * @param resource $image An GD image resource
     * @param Dk_Util_Path $dest The destination file location
     * @return boolean Returns false on failure
     */
    static function saveImage($image, Dk_Util_Path $dest)
    {
        switch ($dest->getExtension()) {
            case 'jpeg':
            case 'jpg':
                if (!imagejpeg($image, $dest->getPath())) {
                    return false;
                }
                break;
            case 'gif':
                if (!imagegif($image, $dest->getPath())) {
                    return false;
                }
                break;
            case 'png':
                if (!imagepng($image, $dest->getPath())) {
                    return false;
                }
                break;
        }
        return true;
    }
    
    
    
    /**
     * Place a colored border on an image
     * 
     * @param Dk_Util_Path $file 
     * @param integer $pixels Default 3
     * @param Dk_Util_Color $color If null Default will be White ('FFFFFF')
     * @throws Dk_ExceptionIllegalArgument On unsuported file type
     */
    static function addBorder(Dk_Util_Path $file, $pixels = 3, Dk_Util_Color $color = null)
    {
        if ($color == null) {
            $color = new Dk_Util_Color(255, 255, 255);
        }
        
        $image = self::createImagefromFile($file);
        
        $width = imagesx($image)-1;
        $height = imagesy($image)-1;
        $lineColor = imagecolorallocate($image, $color->getRed(), $color->getGreen(), $color->getBlue());
        for ($i = 0; $i < $pixels; $i++) {
            imageline($image, 0, 0+$i, $width, 0+$i, $lineColor);     // Top
            imageline($image, 0, $height-$i, $width, $height-$i, $lineColor);     // Bottom
            imageline($image, 0+$i, 0, 0+$i, $height, $lineColor);     // left
            imageline($image, $width-$i, 0, $width-$i, $height, $lineColor);     // right
        }
        
        self::saveImage($image, $file);
        imagedestroy($image);
        return true;
        
    }
    
    
    /**
     * copy and Resize an image to the destination
     *
     * @param string $src The source image path
     * @param string $dest The destination image path
     * @param integer $width The new image width
     * @param integer $height The new image height
     * @param string $bgColor Hex color. eg: '000000', 'FFFFFF'
     * @param boolean $padImg Pad the image out to size using the bgHexColor
     * @throws Dk_ExceptionRuntime
     */
    static function resize($src, $dest, $width, $height, $bgHexColor = 'FFFFFF', $padImg = false)
    {
        $width = intval($width);
        $height = intval($height);
    
        if (is_dir($dest)) {
            throw new Dk_ExceptionRuntime('Destination file is a directory.');
        }
        if (!is_writable(dirname($dest))) {
            $dir = dirname($dest);
            throw new Dk_ExceptionRuntime("Destination directory not writable: `$dir`");
        }
        if ($width < 1 || $height < 1) {
            throw new Dk_ExceptionRuntime('Dimensions have invalid values');
        }
        
        $srcImg = self::createImagefromFile(new Dk_Util_Path($src));
        
        // resize
        $px = $py = 0;
        $sWidth = imagesx($srcImg);
        $sHeight = imagesy($srcImg);
        
        
        $xscale=$sWidth/$width;
        $yscale=$sHeight/$height;
        if ($yscale>$xscale){
            $destWidth = round($sWidth * (1/$yscale));
            $destHeight = round($sHeight * (1/$yscale));
            $px = ($width-$destWidth) / 2;
        } else {
            $destWidth = round($sWidth * (1/$xscale));
            $destHeight = round($sHeight * (1/$xscale));
            $py = ($height-$destHeight) / 2;
        }
        
        $c = Dk_Util_Color::parseHex($bgHexColor);
        if ($padImg) {
            $thumb = imagecreatetruecolor($width, $height);
            $bgColor = imagecolorallocate($thumb, $c->getRed(), $c->getGreen(), $c->getBlue());
            imagefill($thumb, 0, 0 ,$bgColor);
        } else {
            $thumb = imagecreatetruecolor($destWidth, $destHeight);
            $px = $py = 0;
        }
        
        // Resize
        if (!imagecopyresampled($thumb, $srcImg, $px, $py, 0, 0, $destWidth, $destHeight, $sWidth, $sHeight)) {
            return false;
        }
        
        self::saveImage($thumb, new Dk_Util_Path($dest));
        
        imagedestroy($thumb);
        imagedestroy($srcImg);
        return true;
    }
    
    /**
     * This function will only resize a file if it exists and exceeds
     *   the width and height specified. Otherwise the file will remain un-modified
     * 
     * @param string $src
     * @param string $dest
     * @param integer $width
     * @param integer $height
     * @return boolean Return false on error
     */
    static function conditionalResize($src, $dest, $width, $height, $bgHexColor = 'FFFFFF', $padImg = false)
    {
        if (!is_file($src)) {
            return false;
        }
        $imageInfo = getimagesize($src);
        if (!$imageInfo) {
            return false;
        }
        if ($imageInfo[0] > $width || $imageInfo[1] > $height) {
            return self::resize($src, $dest, $width, $height, $bgHexColor, $padImg);
        }
        return true;
    }
    
    
    /**
     * copy and Resize an image to the destination
     *
     * @param string $src The source image path
     * @param string $dest The destination image path
     * @param integer $width The new image width
     * @param integer $height The new image height
     * @param string $bgColor Hex color. eg: '000000', 'FFFFFF'
     * @param boolean $padImg Pad the image out to size using the bgHexColor
     * @throws Dk_ExceptionRuntime
     */
    static function staticResize($src, $dest, $width, $height)
    {
        $width = intval($width);
        $height = intval($height);
    
        if (is_dir($dest)) {
            throw new Dk_ExceptionRuntime('Destination file is a directory.');
        }
        if (!is_writable(dirname($dest))) {
            $dir = dirname($dest);
            throw new Dk_ExceptionRuntime("Destination directory not writable: `$dir`");
        }
        if ($width < 1 || $height < 1) {
            throw new Dk_ExceptionRuntime('Dimensions have invalid values');
        }
        
        $srcImg = self::createImagefromFile(new Dk_Util_Path($src));
        
        // resize
        $sWidth = imagesx($srcImg);
        $sHeight = imagesy($srcImg);
        if ($sWidth == $width && $sHeight == $height) {
            return;
        }

        $thumb = imagecreatetruecolor($width, $height);
        
        // Resize
        if (!imagecopyresampled($thumb, $srcImg, 0, 0, 0, 0, $width, $height, $sWidth, $sHeight)) {
            return false;
        }
        
        self::saveImage($thumb, new Dk_Util_Path($dest));
        
        imagedestroy($thumb);
        imagedestroy($srcImg);
        return true;
    }
    
    /**
     * Is the supplied image extension valid for the SDK
     *
     * @param string $ext
     * @return boolean
     */
    static function isValidImage($ext)
    {   
        switch ($ext) {
            case 'jpeg':
            case 'jpg':
            case 'gif':
            case 'png':
                return true;
        }
        return false;
    }
    
}
?>