<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @author Darryl Ross <darryl.ross@aot.com.au>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */



/**
 * This debugger is used to help debug the application in question.
 * Use 'Dk_Util_Debugger::enableBacktrace()' to get a trace in the vd() call.
 * 
 * @package Util
 */
class Dk_Util_Debugger extends Dk_Object
{
  
    const NONE_MODE = 0;
    const HTML_MODE = 1;
    const CLI_MODE = 2;
    
    private static $backtrace = true;
    
    private $mode = Dk_Util_Debugger::HTML_MODE;
    private $htmlDump = '';
    
    /**
     * Define an assoc array of error string
     * in reality the only entries we should
     * consider are E_WARNING, E_NOTICE, E_USER_ERROR,
     * E_USER_WARNING and E_USER_NOTICE
     */
   private static $errortype = array (
               'E_ERROR'               => 'Error',
               'E_WARNING'             => 'Warning',
               'E_PARSE'               => 'Parsing Error',
               'E_NOTICE'              => 'Notice',
               'E_CORE_ERROR'          => 'Core Error',
               'E_CORE_WARNING'        => 'Core Warning',
               'E_COMPILE_ERROR'       => 'Compile Error',
               'E_COMPILE_WARNING'     => 'Compile Warning',
               'E_USER_ERROR'          => 'User Error',
               'E_USER_WARNING'        => 'User Warning',
               'E_USER_NOTICE'         => 'User Notice',
               'E_STRICT'              => 'Runtime Notice',
               'E_RECOVERABLE_ERROR'   => 'Catchable Fatal Error'
               );

    /**
     * Get the instance of the debugger
     *
     * @param Dk_Config $config 
     * @return Dk_Util_Debugger
     */
    static private function getInstance() 
    {
        static $instance = null;
        if ($instance == null) {
            $instance = new Dk_Util_Debugger();
        }
        return $instance;
    }
    
    
    /**
     * Set the mode of the debugger
     * 
     * @param boolean $mode
     */
    static function setMode($mode) 
    {
        Dk_Util_Debugger::getInstance()->doSetMode($mode);
    }
    
    /**
     * Enable the backtrace 
     * 
     */
    static function disableBacktrace() 
    {
        Dk_Util_Debugger::$backtrace = false;
    }
    
    /**
     * Dump the objects out to the output stream
     * 
     * @param array $objects
     */
    static function dump($objects) 
    {
        Dk_Util_Debugger::getInstance()->doDump($objects);
    }
    
    /**
     * Append the dump to a template.
     * 
     * @param DOMDocument $doc
     */
    static function appendDump(DOMDocument $doc) 
    {
        Dk_Util_Debugger::getInstance()->doAppendDump($doc);
    }
    
    /**
     * Set the mode of the debugger.
     * Avalible options:
     *   o Dk_Util_Debugger::HTML_MODE
     *   o Dk_Util_Debugger::CLI_MODE
     *
     * @param integer $mode
     */
    private function doSetMode($mode) 
    {
        $this->mode = $mode;
    }
    
    /**
     * Append the dump to a DomDocument
     *
     * @param DOMDocument $doc
     */
    private function doAppendDump(DOMDocument $doc)
    {
        if ($this->mode & self::HTML_MODE) {
            $nodes = $doc->getElementsByTagName('body');
            if ($nodes->length > 0) {
                $body = $nodes->item(0);
                $html = '<div style="color: #000;background-color: #FFF;text-align: left;" class="Dk_vd">'.
                $this->htmlDump ."</div>\n";
                $doc2 = new DOMDocument();
                $doc2->loadHTML($html);
                $nodes = $doc2->getElementsByTagName('div');
                if ($nodes->length > 0) {
                    $dumpNode = $nodes->item(0);
                    $dumpNode = $doc->importNode($dumpNode, true);
                    $node = $dumpNode->cloneNode(true);
                    $body->appendChild($node);
                }
            }
        }
        
    }
    
    /**
     * The debugger instance of dump used in vd()
     *
     * @param array $objects
     */
    private function doDump($objects)
    {
        $backtrace = debug_backtrace();
        $trace = $backtrace[2]; // get the line where vd was called..
        
        // CLI dump
        $cliStr = "\nDebug Msg: ".$trace['file']." (".$trace['line'].")\n--\n";
        // HTML dump
        $htmlStr = "\n".'<div style="color: #000;background-color: #FFF;text-align: left;" class="Dk_vd"><pre><i>vd() '.
                    $trace['file']." (".$trace['line'].")</i><br/><div style=\"padding-left: 30px;\">\n";
        
        foreach ($objects as $object) {
            if ($object === null) {
                $objStr = '{NULL}';
            } elseif (is_object($object) && method_exists($object, '__toString')) 
            {
                $objStr = $object->__toString();
            } else {
                $objStr = str_replace("\0", '|', print_r($object, true));
            }
            $htmlStr .= htmlentities($objStr)."<br/>\n";
            $cliStr .= $objStr."\n";
        }
        if (Dk_Util_Debugger::$backtrace) {
            $htmlStr .= "<br/>".$this->getBacktrace(4)."<br/>\n";
            $cliStr .= "\n".$this->getBacktrace()."\n";
        }
        $htmlStr .= "</div></pre></div>\n\n";
        
        if ($this->mode & Dk_Util_Debugger::HTML_MODE) {
            $this->htmlDump .= $htmlStr;
        } elseif ($this->mode & Dk_Util_Debugger::CLI_MODE) {
            echo $cliStr;
        }
        error_log($cliStr."\n");
    }
    
    /**
     * Get a string representation of the backtrace
     *
     * @param integer $remove The number of entries to skip
     * @return string
     */
    static function getBacktrace($remove = 4) {
        $str = '';
        
        $trace = debug_backtrace();
        for($i = 0; $i < $remove; $i ++) {
            unset($trace[$i]);
        }
        $trace = array_merge($trace, array());
        $n = count($trace);
        for($i = 0; $i < $n; $i++) {
            $type = '';
            if (isset($trace[$i]['type'])) {
                $type = $trace[$i]['type'];
            }
            $class = '';
            if (isset($trace[$i]['class'])) {
                $class = $trace[$i]['class'];
            }
            $file = '';
            if (isset($trace[$i]['file'])) {
                $file = $trace[$i]['file'];
            }
            $line = '';
            if (isset($trace[$i]['line'])) {
                $line = $trace[$i]['line'];
            }
            $function = '';
            if (isset($trace[$i]['function'])) {
                $function = $trace[$i]['function'];
            }
            $str .= sprintf("[%s] %s(%s): %s%s%s \n", 
                $i, $file, $line, $class, $type, $function);
        }
        return $str;
    }
    
    /**
     * Handeler for standard PHP errors.
     * Only throw exceptions on ERRORS and log all warnings and notices
     *
     * @param integer $code
     * @param string $string
     * @param string $file
     * @param integer $line
     * @param string $context
     */
    static function errorHandlerCallback($code, $string, $file, $line, $context) 
    {
        $errorDesc = '';
        $ignore = array();
        
        foreach ($ignore as $msg) {
            if (strpos($string, $msg) !== false) {
                return;
            }
        }
        
        if (array_key_exists($code, self::$errortype)) {
            $errorDesc = self::$errortype[$code];
        }
        if ($code == E_ERROR || $code == E_USER_ERROR) {
            $e = new Dk_ExceptionRuntime($errorDesc . '.' . $context . ': ' . $string, $code);
            if (Dk_Config::getInstance()->isDebugMode()) {
                error_log($e->__toString());
            }
            throw $e;
        } else {
            if (Dk_Config::getInstance()->isDebugMode()) {
                error_log("\n    " . $errorDesc . '.' . $context . ': ' . 
                    $string . "\n    " . "$file($line) \n\n" . self::getBacktrace(0) . "\n");
            }
        }
    }
}

set_error_handler(array("Dk_Util_Debugger", "errorHandlerCallback"), E_ALL);



/**
 * Output a visual dump of an object.
 * 
 * EG:<br/>
 * <code>
 * <?php
 *   // var dump usage
 *   vd($arg1, $arg2, $arg3, ...);
 * ?>
 * </code>
 * 
 * @param mixed $args Multiple vars retrived using func_get_args()
 */
function vd() 
{
    $objects = func_get_args();
    Dk_Util_Debugger::dump($objects);
}

?>