<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */


/**
 * A date set is a set of dates that represent a period from and to 
 * Assumtions: DateFrom is allways less than DateTo
 * 
 * 
 * @package Util
 */
class Dk_Util_DateSet extends Dk_Object 
{
    
    /**
     * @var Dk_Util_Date
     */
    private $dateFrom = null;
    
    /**
     * @var Dk_Util_Date
     */
    private $dateTo = null;
    
    
    /**
     * __construct
     * 
     * @param Dk_Util_Date $dateFrom The date from.
     * @param integer $numberOfDays The number of days.
     */
    function __construct(Dk_Util_Date $dateFrom, $numberOfDays = 1)
    {
        $this->dateFrom = $dateFrom;
        $this->dateTo = $dateFrom->addDays($numberOfDays);
    }
    
    /**
     * Create a Default dateset for 1 day
     *
     * @return Dk_Util_Dateset
     */
    static function createDefaultDateSet() 
    {
        $fromDate = Dk_Util_Date::createDate();
        return new Dk_Util_DateSet($fromDate, 1);
    }
    
    /**
     * Create a dateset from given dates
     *
     * @param Dk_Util_Date $dateFrom
     * @param Dk_Util_Date $checkOut
     * @return Dk_Util_DateSet
     */
    static function createFromDates(Dk_Util_Date $dateFrom, Dk_Util_Date $dateTo) 
    {
        $dateSet = new Dk_Util_DateSet($dateFrom);
        $dateSet->setDateTo($dateTo);
        return $dateSet;
    }
    
    
    
    /**
     * Sets the date from which a service is required.
     * 
     * @param Dk_Util_Date $dateFrom
     */
    function setDateFrom(Dk_Util_Date $dateFrom)
    {
        if ($this->dateTo != null) {
            if ($this->dateTo->lessThan($dateFrom)) {
                $this->dateTo = $dateFrom->addDays(1);
            }
        }
        $this->dateFrom = $dateFrom;
    }
    
    /**
     * Gets the date from which a service is required.
     * 
     * @return Dk_Util_Date
     */
    function getDateFrom()
    {
        return $this->dateFrom;
    }
    
    /**
     * Sets the last day the service is charged for.
     *
     * The dateTo day is always 24 hours prior to checkOut.
     * @param Dk_Util_Date $dateTo
     */
    function setDateTo(Dk_Util_Date $dateTo)
    {
        if ($this->dateFrom != null) {
            if ($this->dateFrom->greaterThan($dateTo)) {
                $this->dateFrom = $dateTo->addDays(-1);
            }
        }
        $this->dateTo = $dateTo;
    }
    
    /**
     * Gets the last day the service is charged for.
     * @return Dk_Util_Date
     */
    function getDateTo()
    {
        return $this->dateTo;
    }
    
    /**
     * Sets the date the service is returned or vacated.
     * 
     * The checkOut day is always 24 hours from the dateTo.
     * 
     * @param Dk_Util_Date $checkOut
     */
    function setCheckOut(Dk_Util_Date $checkOut)
    {
        $this->dateTo = $checkOut->addDays(-1);
        if ($this->dateFrom != null) {
            $this->numberOfDays = $this->dateTo->dayDifference($this->dateFrom) + 1;
        }
        $this->checkOut = $checkOut;
    }
    
    /**
     * Check if a dateSet overlapps this date.
     * eg:
     * 
     * <code>
     * x = dateFrom
     * y = dateTo
     * 
     * =>     x1 |-----------------| y1     <=
     *   x2 |--------| y2   x2 |--------| y2
     *           x2 |-----------| y2
     *       x2 |-------------------| y2
     * 
     * if ((x2 >= x1 && x2 <= y1) || (y2 >= x1 && y2 <=y1) || (x2 < x1 && y2 > y1))
     *   then true;
     * 
     * </code>
     * 
     * @param Dk_Util_DateSet $compare
     * @return boolean
     */
    function isDateOverlapp(Dk_Util_DateSet $compare) 
    {
        $x1 = $this->getDateFrom();
        $y1 = $this->getDateTo();
        $x2 = $compare->getDateFrom();
        $y2 = $compare->getDateTo();
        
        if (($x2->greaterThanEqual($x1) && $x2->lessThanEqual($y1)) ||
            ($y2->greaterThanEqual($x1) && $y2->lessThanEqual($y1)) ||
            ($x2->lessThan($x1) && $y2->greaterThan($y1))) {
                return true;
        }
        return false;
    }
    
    /**
     * Check if a given date resides within this dateset.
     *
     * @param Dk_Util_Date $date
     * @return boolean
     */
    function contains(Dk_Util_Date $date) 
    {
        $x = $this->getDateFrom();
        $y = $this->getDateTo();
        
        if ($date->greaterThanEqual($x) && $date->lessThanEqual($y)) {
            return true;
        }
        return false;
    }
    
    /**
     * Gets the date the service is returned or vacated.
     *   Equivelent to dateTo + 1 day.
     * 
     * @return Dk_Util_Date
     */
    function getCheckOut()
    {
        return $this->dateTo->addDays(1);
    }
    
    /**
     * Returns the number of days from dateFrom to dateTo.
     *
     * @return integer
     */
    function getNumberOfDays()
    {
        if ($this->dateFrom == null || $this->dateTo == null) {
            return 1;
        }
        return $this->dateTo->dayDifference($this->dateFrom) + 1;
    }
    
    /**
     * Returns the number of days from dateFrom to dateTo.
     *
     * @return integer
     * @todo Check this is accurate
     */
    function getNumberOfNights()
    {
        if ($this->dateFrom == null || $this->dateTo == null) {
            return 0;
        }
        return $this->dateTo->dayDifference($this->dateFrom);
    }
    
    /**
     * Calculate the number of months to a float number.
     * For example for a date set of 16/8/08 - 16/10/08
     *   we need to get the remainder for the 8th month and the remainder for the 10 month and the 11th is added as 1.
     *   therefore we would end up with the eqn: 0.5 + 1 + 0.5 = 2
     * 
     * This number can then be used to caculate timed products so in this case we would bill for 2 months.
     *
     * @return float
     */
    function getMonthCount()
    {
        $total =  0.0;
        $dateFrom = $this->getDateFrom()->floor();
        $dateTo = $this->getDateTo()->floor();
        $numDays = $dateTo->dayDifference($dateFrom)+1;
        if ($dateFrom->getMonth() == $dateTo->getMonth() && $dateFrom->getYear() == $dateTo->getYear()) {
            $total = round(($numDays/$dateFrom->getLastDay()->getDate()), 2);
        } else {
            // we have different months.
            for ($i = $dateFrom->getFirstDay(); $i->lessThanEqual($dateTo->getFirstDay()); $i = $i->addMonths(1)) {
                if ($dateFrom->getDate() > 1 && $dateFrom->getMonth() == $i->getMonth() && $dateFrom->getYear() == $i->getYear()) {
                    $j =round((($dateFrom->getLastDay()->dayDifference($dateFrom))/$dateFrom->getLastDay()->getDate()), 2);
                    //vd('From: ' . $j);
                    $total += $j;
                } else if ($dateTo->getDate() < $dateTo->getLastDay()->getDate() && $dateTo->getMonth() == $i->getMonth() && $dateTo->getYear() == $i->getYear()) {
                    $j = round((($dateTo->dayDifference($dateTo->getFirstDay())+1)/$dateTo->getLastDay()->getDate()), 2);
                    //vd('To: ' . $j);
                    $total += $j;
                } else {
                    $total++;
                    //vd('Month: ' . 1);
                }
            }
        }
        //vd($dateFrom, $dateTo, $total);
        return $total;
    }
    
    /**
     * Get a date string in the format of
     * `13 Feb 2008 - 14 Mar 2009` 
     *
     * @return string
     */
    function getMediumDate()
    {
        return $this->dateFrom->getMediumDate() . ' - ' . $this->dateTo->getMediumDate();
    }
    
    /**
     *  Rreturn a string representation of this object
     * 
     * @return string
     */
    function toString() {
       $str = '';
       if ($this->dateFrom != null) {
           $str .= 'DateFrom: '.$this->dateFrom->toString()."\n";
       }
       if ($this->dateTo != null) {
           $str .= 'DateTo: '.$this->dateTo->toString()."\n";
       }
       return $str;
    }
    
}
?>