<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @author Darryl Ross <darryl.ross@aot.com.au>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * The Date object to handle Date functions.
 *
 * @package Util
 */
class Dk_Util_Date extends Dk_Object
{

    /**
     * @var integer
     */
    private $timestamp = 0;

    /**
     * Month abbreviations.
     * @var array
     */
    static private $monthAbbrev = array(
        '1' =>  'Jan',
        '2' =>  'Feb',
        '3' =>  'Mar',
        '4' =>  'Apr',
        '5' =>  'May',
        '6' =>  'Jun',
        '7' =>  'Jul',
        '8' =>  'Aug',
        '9' =>  'Sep',
        '10' => 'Oct',
        '11' => 'Nov',
        '12' => 'Dec'
    );

    /**
     * Month end days.
     * @var array
     */
    static private $monthEnd = array(
        '1' => '31',
        '2' => '28',
        '3' => '31',
        '4' => '30',
        '5' => '31',
        '6' => '30',
        '7' => '31',
        '8' => '31',
        '9' => '30',
        '10' => '31',
        '11' => '30',
        '12' => '31'
    );




    /**
     *
     *
     * @param integer $timestamp A unix timestamp.
     */
    function __construct($timestamp = null)
    {
        if ($timestamp === null) {
            $timestamp = time();
        }
        $this->timestamp = $timestamp;
    }

    /**
     * Create an Dk_Util_Date object with a time value set to 00:00:00.
     *
     * Cuent date time is geneated if no timestamp given
     *
     * @param integer $timestamp
     * @return Dk_Util_Date
     */
    static function createDate($timestamp = null)
    {
        if ($timestamp === null) {
            $timestamp = time();
        }

        return new Dk_Util_Date($timestamp);
    }

    /**
     * Create an Dk_Util_Date object with a default date of 01-01-1970 (Unix Epoch, January 1 1970 00:00:00 GMT)
     *
     * Cuent date time is geneated if no timestamp given
     *
     * @param integer $timestamp
     * @return Dk_Util_Date
     */
    static function createTime($timestamp = null)
    {
        if ($timestamp === null) {
            $timestamp = time();
        }
        // Set date to 01-01-1970
        $strTime = date('H:i:s', $timestamp);
        list($h, $m, $s) = explode(':', $strTime, 3);
        $timestamp = mktime($h, $m, $s);

        return new Dk_Util_Date($timestamp);
    }



    /**
     * Create an Dk_Util_Date object from ISO date time.
     * ISO Format is:
     *        yyyy-mm-dd hh:mm:ss, where
     *  o yyyy is a four digit numeral that represents the year.
     *  o the remaining '-'s are separators between parts of the date portion;
     *  o the first mm is a two-digit numeral that represents the month;
     *  o dd is a two-digit numeral that represents the day;
     *  o hh is a two-digit numeral that represents the hour.
     *  o ':' is a separator between parts of the time-of-day portion;
     *  o the second mm is a two-digit numeral that represents the minute;
     *  o ss is a two-integer-digit numeral that represents the whole seconds;
     *
     * @param string $iso
     * @return Dk_Util_Date Returns null if $iso_date is not a valid ISO date.
     */
    static function parseIso($isoDate)
    {
        $regs = null;
        if ($isoDate == null || $isoDate == '0000-00-00 00:00:00') {
            return null;
        }
        if (!ereg('^([0-9]{4})-([0-9]{1,2})-([0-9]{1,2})[T ]?(([0-9]{2}):([0-9]{2})(:([0-9]{2}))?)?$',
            $isoDate, $regs))
        {
            return null;
        }
        if (!checkdate($regs[2], $regs[3], $regs[1])) {
            return null;
        }
        $timestamp = mktime(0, 0, 0, $regs[2], $regs[3], $regs[1]);

        // check time
        if (isset($regs[5]) && $regs[5] != '') {
            if ($timestamp === false) {
                $regs[1] = 1970;
                $regs[2] = 1;
                $regs[3] = 1;
            }
            if (isset($regs[8]) && $regs[8] != '') {
                $timestamp = mktime($regs[5], $regs[6], $regs[8], $regs[2], $regs[3], $regs[1]);
            }else{
                $timestamp = mktime($regs[5], $regs[6], 0, $regs[2], $regs[3], $regs[1]);
            }
        }
        // If all else fails return false
        if ($timestamp === null) {
            return null;
        }
        
        return new Dk_Util_Date($timestamp);
    }

    /**
     * Create a date object from the date in the format of dd/mm/yyyy
     * @param string $str
     * @return Dk_Util_Date
     */
    static function parseFromString($str) 
    {
        if (ereg('^[0-9]{2}/[0-9]{2}/([0-9]{2})?[0-9]{4}$', $str)) {
            $arr = explode('/', $str);
            $timestamp = mktime(0, 0, 0, $arr[1], $arr[0], $arr[2]);
            return new Dk_Util_Date($timestamp);
        } else {
            return null;
        }
    }
    
    /**
     * This function converts an integer to an array of hours and minutes.
     *
     * Eg: 140 = array('2', '20')
     *
     * @param integer $minutes
     */
    static function convertMinutes($minutes)
    {
        $hrs = floor($minutes / 60);
        $min = $minutes % 60;
        return array($hrs, $min);
    }


    /**
     * Return this dates timestamp
     *
     * @return integer
     */
    function getTimestamp()
    {
        return $this->timestamp;
    }

    /**
     * Set the time of a date object to 23:59:59
     *
     * @return Dk_Util_date
     */
    function ceil()
    {
        $ts = mktime(23, 59, 59, $this->getMonth(), $this->getDate(), $this->getYear());
        return new Dk_Util_Date($ts);
    }

    /**
     * Set the time of a date object to 00:00:00
     *
     * @return Dk_Util_date
     */
    function floor()
    {
        $ts = mktime(0, 0, 0, $this->getMonth(), $this->getDate(), $this->getYear());
        return new Dk_Util_Date($ts);
    }

    /**
     * Adds days to date and returns a new instance.
     * NOTE: Days are calculated as ($days * 86400)
     *
     * To subtract days, use a negative number of days.
     * @param integer $days
     * @return Dk_Util_Date
     */
    function addDays($days)
    {
        return new Dk_Util_Date($this->getTimestamp() + ($days * 86400));
    }

    /**
     * Add seconds to a date.
     *
     * @param integer $sec
     * @return Dk_Util_Date
     */
    function addSeconds($sec)
    {
        return new Dk_Util_Date($this->getTimestamp() + $sec);
    }

    
    /**
     * Add actual months to a date
     *
     * @param integer $months
     * @return Dk_Util_Date
     */
    function addMonths($months)
    {
        $ts = mktime($this->getHour(), $this->getMinute(), $this->getSecond(), $this->getMonth()+$months, 1, $this->getYear());
        $tmpDate = self::createDateTime($ts);
        if ($this->getDate() == self::getMonthEnd($this->getMonth(), $this->getYear()) || $this->getDate() > $tmpDate->getLastDay()->getDate()) {
            $ts = mktime($this->getHour(), $this->getMinute(), $this->getSecond(), $this->getMonth()+$months, $tmpDate->getLastDay()->getDate(), $this->getYear());
        } else {
            $ts = mktime($this->getHour(), $this->getMinute(), $this->getSecond(), $this->getMonth()+$months, $this->getDate(), $this->getYear());
        }
        $date = self::createDateTime($ts);
        return $date;
    }

    /**
     * Add actual years to a date
     *
     * @param integer $years
     * @return Dk_Util_Date
     */
    function addYears($years)
    {
        $ts = mktime($this->getHour(), $this->getMinute(), $this->getSecond(), $this->getMonth(),
            $this->getDate(), $this->getYear()+$years);
        return self::createDate($ts);
    }

    /**
     * Returns the difference between this date and other in days.
     *
     * @param Dk_Util_Date $other
     * @return integer
     */
    function dayDifference(Dk_Util_Date $other)
    {
        return ceil(($this->getTimestamp() - $other->getTimestamp()) / 86400);
    }

    /**
     * Return the diffrence between this date and other in hours.
     *
     * @param Dk_Util_Date $other
     * @return integer
     */
    function hourDiffrence(Dk_Util_Date $other)
    {
        return ceil(($this->getTimestamp() - $other->getTimestamp()) / 3600);
    }

    /**
     * Compares the value to another instance of date.
     *
     * @param Dk_Util_Date $other
     * @return integer Returns -1 if less than , 0 if equal to, 1 if greater than.
     */
    function compareTo(Dk_Util_Date $other)
    {
        $retVal = 1;
        if ($this->getTimestamp() < $other->getTimestamp()) {
            $retVal = -1;
        } elseif ($this->getTimestamp() == $other->getTimestamp()) {
            $retVal = 0;
        }

        return $retVal;
    }

    /**
     * Checks if the date value is greater than the value of another instance of date.
     *
     * @param Dk_Util_Date
     * @return boolean
     */
    function greaterThan(Dk_Util_Date $other)
    {
        return ($this->compareTo($other) > 0);
    }
    /**
     * Checks if the date value is greater than or equal the value of another instance of date.
     *
     * @param Dk_Util_Date
     * @return boolean
     */
    function greaterThanEqual(Dk_Util_Date $other)
    {
        return ($this->compareTo($other) >= 0);
    }

    /**
     * Checks if the date value is less than the value of another instance of date.
     *
     * @param Dk_Util_Date
     * @return boolean
     */
    function lessThan(Dk_Util_Date $other)
    {
        return ($this->compareTo($other) < 0);
    }

    /**
     * Checks if the date value is less than or equal the value of another instance of date.
     *
     * @param Dk_Util_Date
     * @return boolean
     */
    function lessThanEqual(Dk_Util_Date $other)
    {
        return ($this->compareTo($other) <= 0);
    }

    /**
     * Checks if the date is equal to the value of another instance of date.
     *
     * @param Dk_Util_Date
     * @return boolean
     */
    function equals(Dk_Util_Date $other)
    {
        return ($this->compareTo($other) == 0);
    }



    /**
     * Gets an ISO (yyyy-mm-dd hh:mm:ss) formated string representation of the date.
     * Usful for DB entries.
     *
     * @param boolean $time Return with the timestamp
     * @return string
     */
    function getIsoDate($time = true)
    {
        if ($time) {
            return $this->toString('Y-m-d H:i:s');
        } else {
            return $this->toString('Y-m-d');
        }
    }

    /**
     * Retreve the time with optional sections.
     *
     * @param boolean $seconds Return a value with seconds
     * @return string A time in ISO format.
     */
    function getIsoTime($seconds = false)
    {
        if ($seconds) {
            return $this->toString('H:i:s');
        }else{
            return $this->toString('H:i');
        }
    }

    /**
     * Get a short date in the format of 'dd/mm/yy'
     *
     * @param boolean $showTime
     * @return string
     */
    function getShortDate($shwTime = false)
    {
        if ($shwTime) {
            return $this->toString('d/m/y h:i A');
        } else {
            return $this->toString('d/m/y');
        }
    }

    /**
     * Get a short date in the format of '23 Apr 2003'
     *
     * @param boolean $withYear
     * @return string
     */
    function getMediumDate($withYear = true)
    {
        list($y, $m, $d) = explode('-', $this->getIsoDate(), 3);

        $m = intval($m);
        if ($withYear) {
            //$y = $y - 2000;
            $shortDate = sprintf('%02d %s %d', $d, self::$monthAbbrev[$m], $y);
        }else{
            $shortDate = sprintf('%02d %s', $d, self::$monthAbbrev[$m]);
        }

        return $shortDate;
    }

    /**
     * Get a long date in the format of 'Tuesday, 23 Apr 2003'
     *
     * @param boolean $showTime
     * @return string
     */
    function getLongDate($shwTime = false)
    {
        if ($shwTime) {
            return $this->toString('l, j M Y h:i A');
        } else {
            return $this->toString('l, j M Y');
        }
    }


    /**
     * Get the time in the format of 10:30 AM
     *
     * @param boolean $showSeconds
     * @return string
     */
    function getTimeString($showSeconds = false)
    {
        if ($showSeconds) {
            return $this->toString('h:i:s A');
        } else {
            return $this->toString('h:i A');
        }
    }

    /**
     * Return the months abbreviatted string eg: 1 = 'Jan', 12 = 'Dec'
     *
     * @param integer $m
     * @return string
     */
    static function getMonthAbbrev($m)
    {
        return self::$monthAbbrev[$m];
    }

    /**
     * Get the months ending date 1 = 31-Jan, 12 = 31-Dec
     *
     * @param integer $m
     * @param integer $y
     * @return integer
     */
    static function getMonthEnd($m, $y = '')
    {
        if ($m == 2) {  // feb test for leap year
            if (self::isLeapYear($y)) {
                return self::$monthEnd[$m]+1;
            }
        }
        return self::$monthEnd[$m];
    }

    /**
     * Get the first day of this dates month
     *
     *
     * @return Dk_Util_Date
     */
    function getMonthFirstDay()
    {
        $ts = mktime(23, 59, 59, $this->getMonth(), 1, $this->getYear());
        return self::createDate($ts);
    }

    /**
     * Get the last day of this dates month
     *
     *
     * @return Dk_Util_Date
     */
    function getMonthLastDay()
    {
        $lastDay = self::getMonthEnd($this->getMonth(), $this->getYear());
        $ts = mktime(23, 59, 59, $this->getMonth(), $lastDay, $this->getYear());
        return self::createDate($ts);
    }

    /**
     * Is the supplied year a leap year
     *
     * @param integer $y
     * @param boolean
     */
    static function isLeapYear($y)
    {
        if ($y % 4 != 0) {
            return false;  // use 28 for days in February
        } else if  ($y % 400 == 0) {
            return true; // use 29 for days in February
        } else if ($y % 100 == 0) {
            return false; // use 28 for days in February
        } else {
            return true;  // use 29 for days in February
        }
    }

    /**
     * Get the integer value for the hour
     *
     * @return integer
     */
    function getHour()
    {
        return intval($this->toString('H'), 10);
    }

    /**
     * Get the integer value of teh minute
     *
     * @return integer
     */
    function getMinute()
    {
        return intval($this->toString('i'), 10);
    }

    /**
     * Get the seconds integer value
     *
     * @return integer
     */
    function getSecond()
    {
        return intval($this->toString('s'), 10);
    }

    /**
     * Get the integer value of the day date.
     *
     * @return integer
     */
    function getDate()
    {
        return intval($this->toString('j'), 10);
    }

    /**
     * Get the integer value of the month
     *
     * @return integer
     */
    function getMonth()
    {
        return intval($this->toString('n'), 10);
    }

    /**
     * Get the 4 digit integer value of the year
     *
     * @return integer
     */
    function getYear()
    {
        return intval($this->toString('Y'), 10);
    }

    /**
     * Return a string representation of this object
     *
     * @param string $format Optional date format string
     * @return string
     * @see http://au.php.net/manual/en/function.date.php date() function in the php manual.
     */
    function toString($format = '')
    {
        if ($format == '') {
            return $this->getIsoDate() . ' (' . $this->getTimestamp() . ')';
        }else{
            return date($format, $this->getTimestamp());
        }
    }
}
?>
