<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @author Darryl Ross <darryl.ross@aot.com.au>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * Represents a currency. 
 * Currencies are identified by their currency codes.
 *  o AUSTRALIA    Australian Dollar   AUD  036
 *  o NEW ZEALAND  New Zealand Dollar  NZD  554
 *   
 * The class is designed so that there's never more than one Currency instance 
 * for any given currency. Therefore, there's no public constructor. You obtain 
 * a Currency instance using the getInstance methods.
 * 
 * @link http://www.iso.org/iso/en/prods-services/popstds/currencycodeslist.html
 * @package Util
 */
class Dk_Util_Currency extends Dk_Object 
{
    
    /**
     * @var string
     */
    private $currencyCode = '';
    
    /**
     * @var array
     */
    static private $objects = array();
    
    /**
     * @var array
     */
    static $currencyList = array(
        'AUD' => array( 'name' => 'Australian Dollar',
                        'locale' => 'AUSTRALIA',
                        'symbol' => '$',
                        'altSymbol' => 'AUD$',
                        'digits' => 2),
        'NZD' => array( 'name' => 'New Zealand Dollar',
                        'locale' => 'NEW ZEALAND',
                        'symbol' => '$',
                        'altSymbol' => 'NZD$',
                        'digits' => 2),
        'USD' => array( 'name' => 'American Dollar',
                        'locale' => 'United Stated Of America',
                        'symbol' => '$',
                        'altSymbol' => 'USD$',
                        'digits' => 2),
        'THB' => array( 'name' => 'Thai baht',
                        'locale' => 'Thailand',
                        'symbol' => '฿',
                        'altSymbol' => 'THB$',
                        'digits' => 2));

    
    
    
    /**
     * __construct 
     * 
     * @param string $currencyCode (optional) Default value is 'AUD'
     */
    private function __construct($currencyCode = 'AUD') 
    {
        $this->currencyCode = $currencyCode;
    }
    
    /**
     * Returns the Currency instance for the given currency code.
     *
     * @return Dk_Util_Currency
     * @throws Dk_ExceptionIllegalArgument
     */
    static function getInstance($currencyCode = 'AUD')
    {
        if (!array_key_exists($currencyCode, Dk_Util_Currency::$currencyList)) {
            throw new Dk_ExceptionIllegalArgument('Invalid Currency code: ' . $currencyCode);
        }
        if (!isset(Dk_Util_Currency::$objects[$currencyCode])) {
            Dk_Util_Currency::$objects[$currencyCode] = new Dk_Util_Currency($currencyCode);
        }
        return Dk_Util_Currency::$objects[$currencyCode];
    }
    
    /**
     * Gets the ISO 4217 currency code of this currency.
     * 
     * @return string
     */
    function getCurrencyCode()
    {
        return $this->currencyCode;
    }
    
    /**
     * Gets the symbol of this currency for the specified locale. 
     *
     * For example, for the US Dollar, the symbol is "$" if the specified 
     * locale is the US, while for other locales it may be "US$". If no 
     * symbol can be determined, the ISO 4217 currency code is returned.
     *
     * If locale is null, then the default locale is used.
     *
     * @param string $locale (optional) The locale, default 'AUSTRALIA'.
     * @return string The symbol of this currency for the specified locale.
     */
    function getSymbol($code = 'AUD')
    {
        if ($code == $this->currencyCode) {
            return Dk_Util_Currency::$currencyList[$this->currencyCode]['symbol'];
        }else{
            return Dk_Util_Currency::$currencyList[$this->currencyCode]['altSymbol'];
        }
    }
    
    /**
     * Gets the default number of fraction digits used with this currency. 
     *
     * For example, the default number of fraction digits for the Euro is 2, 
     * while for the Japanese Yen it's 0. In the case of pseudo-currencies, 
     * such as IMF Special Drawing Rights, -1 is returned.
     *
     * @return integer
     */
    function getDefaultFractionDigits()
    {
        return Dk_Util_Currency::$currencyList[$this->currencyCode]['digits'];
    }
}
?>
