<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @author Darryl Ross <darryl.ross@aot.com.au>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * This class is used to manage an RGB color type
 * 
 * @package Util
 */
class Dk_Util_Color extends Dk_Object
{
    /**
     * @var integer
     */
    private $red = 0;
    /**
     * @var integer
     */
    private $green = 0;
    /**
     * @var integer
     */
    private $blue = 0;
    
    
    /**
     *
     * @param integer $red
     * @param integer $green
     * @param integer $blue
     */
    function __construct($red = 0, $green = 0, $blue = 0)
    {
        $this->setRed($red);
        $this->setGreen($green);
        $this->setBlue($blue);
    }
    
    /**
     * Create a color from a hex string either '#FF7FH8' or 'FF7FH8'
     *
     * @param string $hex
     * @return Dk_Util_Color
     */
    static function parseHex($hex)
    {
        $regs = null;
        if (!ereg('^(\#)?([A-F0-9]{6})$', $hex, $regs)) {
            return;
        }
        $hex = $regs[2];
        $color = new Dk_Util_Color(
          intval(substr($hex, 0, 2), 16), 
          intval(substr($hex, 2, 2), 16),
          intval(substr($hex, 4, 2), 16));
        return $color;
    }
    
    
    
    /**
     * Create a primary color from a name
     *
     * @param string $hex
     * @return Dk_Util_Color
     */
    static function parseString($str)
    {
        switch(strtolower($str))
        {
          case 'white':
              return Dk_Util_Color::parseHex('000000');
          case 'black':
              return Dk_Util_Color::parseHex('FFFFFF');
          case 'red':
              return Dk_Util_Color::parseHex('FF0000');
          case 'green':
              return Dk_Util_Color::parseHex('00FF00');
          case 'blue':
              return Dk_Util_Color::parseHex('0000FF');
          case 'yellow':
              return Dk_Util_Color::parseHex('FFFF00');
          case 'purple':
          case 'magenta':
              return Dk_Util_Color::parseHex('FF00FF');
          case 'cyan':
              return Dk_Util_Color::parseHex('00FFFF');
          default:
              return Dk_Util_Color::parseHex('000000');
        }
    }
    
    /**
     * setRed
     *
     * @param integer $i
     */
    function setRed($i)
    {
        $this->red = $i%256;
    }
    /**
     * getRed
     *
     * @return integer
     */
    function getRed()
    {
        return $this->red;
    }
    
    /**
     * setGreen
     *
     * @param integer $i
     */
    function setGreen($i)
    {
        $this->green = $i%256;
    }
    /**
     * getGreen
     *
     * @return integer
     */
    function getGreen()
    {
        return $this->green;
    }
    
    /**
     * setBlue
     *
     * @param integer $i
     */
    function setBlue($i)
    {
        $this->blue = $i%256;
    }
    /**
     * getBlue
     *
     * @return integer
     */
    function getBlue()
    {
        return $this->blue;
    }
    
    /**
     * Get the inverse color to this color.
     *
     * @return Dk_Util_Color
     */
    function getInverse()
    {
        return new Dk_Util_Color(255-$this->red, 255-$this->green, 255-$this->blue);
    }
    
    /**
     * Return a value of this color as a HEX value
     *
     * @return string
     */
    
    function toString()
    {
        return sprintf('%02X%02X%02X', $this->red, $this->green, $this->blue);
    }
}
?>