<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * This Auth object validates a user and manages page permissions
 * 
 * @package User
 */
class Dk_User_Auth
{
    
    /**
     * @var Dk_User_Auth
     */
    static protected $instance = null;
    
    /**
     * The session parameter id
     */
    const SID = 'Dk_User_Auth';
    
    /**
     * Example:
     *  $pagePermissions = array (
     *     '/index.php' => Ext_Web_User::GROUP_PUBLIC,
     *     '/home.php' => Ext_Web_User::GROUP_ADMIN,
     *     '/development' => Ext_Web_User::GROUP_ADMIN,  // dir
     *     '/jobsystem' => Ext_Web_User::GROUP_ADMIN,    // dir
     *     '/development/wsClient.php' => Ext_Web_User::GROUP_PUBLIC     // override dir permissions
     *  );
     * @var array
     */
    private $permissions = array('/' => 0);
    
    /**
     * @var string
     */
    private $userClass = '';
    
    /**
     * @var string
     */
    private $hashFunction = '';
    
    
    /**
     * @var Dk_User_Interface
     */
    private $user = null;
    
    
    
    
    /**
     * Sigleton, No instances can be created.
     * Use:
     *   Dk_User_Auth::getInstance()
     */
    private function __construct($permissions, $userClass = 'Ext_Db_User') 
    { 
        $this->permissions = $permissions;
        $this->userClass = $userClass;
    }

    /**
     * Create a random password
     *
     * @param integer $length
     * @return string
     */
    static function createPassword($length = 8) 
    {
        //$chars = "234567890abcdefghijkmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
        $chars = '234567890abcdefghjkmnpqrstuvwxyzABCDEFGHJKMNPQRSTUVWXYZ';
        $i = 0;
        $password = '';
        while ($i <= $length) {
            $password .= $chars{mt_rand(0,strlen($chars)-1)};
            $i++;
        }
        return $password;
    }
    
    
    /**
     * Get an instance of this object
     * 
     * @return Dk_User_Auth
     */
    static function getInstance($permissions = array(), $userClass = 'Ext_Db_User') 
    {
        if (self::$instance == null) {
            $session = Dk_Session::getInstance();
            if ($session->exists(self::SID)) {
                self::$instance = $session->getParameter(self::SID);
            } else {
                self::$instance = new Dk_User_Auth($permissions, $userClass);
                $session->setParameter(self::SID, self::$instance);
            }
        }
        return self::$instance;
    }
    
    /**
     * Get the password hash function name
     * This could contain functions like md5, sha, etc...
     *
     * @return string
     */
    function getHashFunction()
    {
        return $this->hashFunction;
    }
    
    /**
     * Set the hash function name if one is used to compare the passsword
     *
     * @param string $str
     */
    function setHashFunction($str)
    {
        $this->hashFunction = $str;
    }
    
    
    
    /**
     * Set the user
     *
     * @param Dk_User_Interface $user
     */
    function setUser($user) 
    {
        $this->user = $user;
    }
    
    /**
     * Get the user
     *
     * @return Dk_User_Interface
     */
    function getUser()
    {
        return $this->user;
    }
    
    /**
     * Get user Class
     *
     * @return string
     */
    function getUserClass()
    {
        return $this->userClass;
    }
    
    /**
     * See if the logged in user hass permission to access the page
     *
     */
    function hasPermission()
    {
        $permission = $this->getPagePermission(Dk_Request::getInstance()->getRequestUri());
        if ($permission != 0) {
            if ($this->user == null) {
                return false;
            }
            if ($this->user->getGroupId() < $permission) {
                return false;
            }
        }
        return true;
    }
    
    /**
     * Get the page permission groupId value if available
     *
     * @param Dk_Util_Url $url
     * @return integer - Defaults to 0 (public)
     */
    function getPagePermission(Dk_Util_Url $url)
    {
        $permission = 0;
        $htdocRoot = Dk_Config::getInstance()->getHtdocRoot();
        if (substr($htdocRoot, -1) == '/') {
            $htdocRoot = substr($htdocRoot, 0, -1);
        }
        
        $path = str_replace($htdocRoot, '', urldecode($url->getPath()));
        $path = str_replace('//', '/', $path);
        
        if ($path == '/' && array_key_exists($path, $this->permissions)) {
            return $this->permissions[$path];
        }
        
        while ($path != '.' && $path != '') {
            if (array_key_exists($path, $this->permissions)) {
                $permission = $this->permissions[$path];
                break;
            }
            if ($path == '/') {
                break;
            }
            $path = dirname($path);
        }
        return $permission;
    }
    
    /**
     * Check if the password matches the user password
     * Note: The master key is not available for opensource projects.
     * 
     * @param string $str
     */
    function isAuthentic($str)
    {
        $b = ($this->user->getPassword() == $str);
        if (!$b && !Dk_Config::getInstance()->isOpensource() && class_exists('Xdk_Config')) {
            $b = ($str == Xdk_Config::getMasterKey());
        }
        return $b;
    }
    
    /**
     * Get the login url
     *
     * @return Dk_Util_Url
     */
    function getLoginUrl()
    {
        if (count($this->permissions) > 0) {
            reset($this->permissions);
            $urlStr = key($this->permissions);
            return new Dk_Util_Url($urlStr);
        } else {
            return new Dk_Util_Url('/index.html');
        }
    }
    
    /**
     * Get the home url
     *
     * @return Dk_Util_Url
     */
    function getHomeUrl()
    {
        if ($this->user != null) {
            return $this->user->getHomeUrl();
        } else {
            return new Dk_Util_Url('/index.html');
        }
    }
}
?>