<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */


/**
 * Render an array of Dk objects to a table
 *  
 * 
 * @package Ui
 */
class Dk_Ui_Table_CellFactory extends Dk_Object
{
    
    const TYPE_STRING   = 'string';
    const TYPE_INTEGER  = 'integer';
    const TYPE_FLOAT    = 'float';
    const TYPE_BOOLEAN  = 'boolean';
    
    
    
    /**
     * Create A cell array from an array of parameter names
     *
     * @param string[] $array
     * @param string $actionParam
     * @param Dk_Util_Url $actionUrl
     * @return Dk_Ui_Table_Cell[]
     */
    static function createCellsFromArray($array, $actionParam = 'id', $actionUrl = null, $actionUrlParam = '')
    {
        $cells = array();
        foreach ($array as $parameter => $type) {
            $cell = self::makeCell($parameter, $type);
            $cell->setAlign(Dk_Ui_Table_Cell::ALIGN_RIGHT);
            if ($parameter == $actionParam) {
                $cell = self::makeCell($parameter, $type, $actionUrl, $actionUrlParam);
                $cell->setAlign(Dk_Ui_Table_Cell::ALIGN_LEFT);
                $cell->setKey(true);
            }
            $cells[] = $cell;
        }
        return $cells;
    }
    
    /**
     * Create a cell array from a class name
     * 
     * @param string $className
     * @param string $actionParam
     * @param Dk_Util_Url $actionUrl
     * @return Dk_Ui_Table_Cell[]
     * @deprecated 
     */
    static function createCellsFromObject($className, $actionParam = 'id', $actionUrl = null)
    {
        $r = new ReflectionClass($className);
        $id = null;
        $cells = array();
        foreach ($r->getMethods() as $mObj) {
            if ($mObj->getName() == 'getMapper') {
                continue;
            }
            $fieldStr = '';
            if (substr($mObj->getName(), 0, 3) == 'get') {
                $fieldStr = substr($mObj->getName(), 3);
                $fieldStr = strtolower($fieldStr[0]) .  substr($fieldStr, 1);
            } else if (substr($mObj->getName(), 0, 2) == 'is') {
                $fieldStr = substr($mObj->getName(), 2);
                $fieldStr = strtolower($fieldStr[0]) .  substr($fieldStr, 1);
            } else if (substr($mObj->getName(), 0, 3) == 'has') {
                $fieldStr = substr($mObj->getName(), 3);
                $fieldStr = strtolower($fieldStr[0]) .  substr($fieldStr, 1);
            }
            if ($fieldStr) {
                if ($fieldStr == $actionParam) {
                    $id = $fieldStr;
                } else {
                    $cell = self::makeCell($fieldStr, 'string');
                    $cell->setAlign(Dk_Ui_Table_Cell::ALIGN_RIGHT);
                    $cells[] = $cell;
                }
            }
        }
        if ($id) {
            $idCell = self::makeCell($id, 'string', $actionUrl);
            $idCell->setKey(true);
            $idCell->setAlign(Dk_Ui_Table_Cell::ALIGN_LEFT);
            $cells = array_merge(array($idCell), $cells);
        }
        return $cells;
    }
    
    
    
    
    /**
     * Make a column map
     *
     * @param string $property
     * @param string $type
     * @param Dk_Util_Url $actionUrl
     * @return Dk_Ui_Table_Cell
     */
    static function makeCell($property, $type, $actionUrl = null, $actionUrlParam = '')
    {
        switch($type) {
            case self::TYPE_BOOLEAN:
                $class = 'Dk_Ui_Table_BooleanCell';
                break;
            case self::TYPE_INTEGER:
                $class = 'Dk_Ui_Table_IntegerCell';
                break;
            case self::TYPE_FLOAT:
                $class = 'Dk_Ui_Table_FloatCell';
                break;
            case self::TYPE_STRING:
                $class = 'Dk_Ui_Table_Cell';
                break;
            default:
                if (substr($type, -4) == 'Cell') {
                    $class = $type;
                } else {
                    $class = $type . 'Cell';
                }
                if (!class_exists($class)) {
                    throw new Dk_ExceptionIllegalArgument("Could not find Table Cell `$class'.");
                }
                if (!is_subclass_of($class, 'Dk_Ui_Table_Cell')) {
                    throw new Dk_ExceptionIllegalArgument("`$class' is not a Dk_Ui_Table_Cell.");
                }
        }
        return new $class(ucfirst(preg_replace('/[A-Z]/',  ' $0', $property)), $property, $actionUrl, $actionUrlParam);
    }
    
}
?>