<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */


/**
 * Render an array of Dk objects to a table
 *  
 * 
 * @package Ui
 */
class Dk_Ui_Table_Cell extends Dk_Ui_Table_ExtraInterface 
{
    const ORDER_NONE  = '';
    const ORDER_ASC  = 'ASC';
    const ORDER_DESC = 'DESC';
    
    const ALIGN_LEFT = 'left';
    const ALIGN_RIGHT = 'right';
    const ALIGN_CENTER = 'center';
    
    
    
    /**
     * @var string
     */
    protected $name = '';
    
    /**
     * @var string
     */
    protected $property = '';
    
    /**
     * @var boolean
     */
    protected $key = false;
    
    /**
     * @var string
     */
    protected $align = self::ALIGN_LEFT;
    
    /**
     * @var Dk_Util_Url
     */
    protected $actionUrl = null;
    
    /**
     * @var string
     */
    protected $actionUrlParam = '';
    
    /**
     * @var array
     */
    protected $rowClass = array();
    
    
    
    /**
     * Create the object instance
     *  
     * @param string $name
     * @param string $property
     * @param Dk_Util_Url $actionUrl
     * @param string $actionUrlParam The param that holds the object id
     */
    function __construct($name, $property = '', $actionUrl = null, $actionUrlParam = '') 
    {
        $this->name = $name;
        $this->property = $property;
        $this->actionUrl = $actionUrl;
        $this->actionUrlParam = $actionUrlParam;
    }
    
    /**
     * Process any events or actions on execution
     * 
     */
    function doProcess() { }
    
    /**
     * get the collumn name
     *
     * @return string
     */
    function getName()
    {
        return $this->name;
    }
    
    /**
     * get the object property
     *
     * @return string
     */
    function getProperty()
    {
        return $this->property;
    }
    
    /**
     * Get the access method if it exists.
     *
     * @param Dk_Object $obj
     * @return string Returns an empty string on fail.
     */
    function getMethod($obj)
    {
        if (!$this->property) {
            return '';
        }
        $method = 'get' . ucfirst($this->property);
        if (!method_exists($obj, $method)) {
            $method = 'is' . ucfirst($this->property);
        }
        if (!method_exists($obj, $method)) {
            $method = 'has' . ucfirst($this->property);
        }
        if (!method_exists($obj, $method)) {
            return '';
        }
        return $method;
    }
    
    
    
    /**
     * get the parameter data from an object if available
     *
     * @param Dk_Object $obj
     * @return string
     */
    function getPropertyData($obj)
    {
        $str = '';
        $method = $this->getMethod($obj);
        if ($method) {
            $str = $obj->$method();
        }
        return htmlentities($str);
    }
    
    /**
     * get the table data from an object if available
     *
     * @param Dk_Object $obj
     * @return string
     */
    function getTableData($obj)
    {
        $this->rowClass = array(); // reset row class list
        $str = '';
        if ($this->getActionUrl()) {
            if ($this->actionUrlParam === '') {
                $pos = strrpos(get_class($obj), '_');
                $this->actionUrlParam = strtolower(substr(get_class($obj), $pos+1)).'Id';
            }
            $url = $this->getActionUrl();
            $url->setQueryField($this->actionUrlParam, $obj->getId());
            $str = '<a href="' . htmlentities($url->toString()) . '">' . $this->getPropertyData($obj) . '</a>';
        } else {
            $str = $this->getPropertyData($obj);
        }
        return $str . '';
    }
    
    /**
     * get the table data from an object if available
     *
     * @param Dk_Object $obj
     * @return string
     */
    function getTableHeader()
    {
        $url = $this->getOrderByUrl();
        if ($url) {
            $str = '<a href="' . htmlentities($url->toString()) . '">' . $this->getName() . '</a>';
        } else {
            $str = '<a href="javascript:;">' . $this->getName() . '</a>';
        }
        return $str;
    }
    
    /**
     * Get the action url object.
     *
     * @return Dk_Util_Url
     */
    function getActionUrl()
    {
        if ($this->actionUrl) {
            return clone $this->actionUrl;
        }
    }
    
    /**
     * Set cell key status.
     * If true then this cell's width should be at maximum width
     *
     * @param boolean $b
     */
    function setKey($b)
    {
        $this->key = $b;
    }
    
    /**
     * Is this cell a key cell
     *
     * @return boolean
     */
    function isKey() 
    {
        return $this->key;
    }
    
    /**
     * Set the cell alignment
     * 
     * Valid parameters are:
     * o Dk_Ui_Table_Cell::ALIGN_CENTER
     * o Dk_Ui_Table_Cell::ALIGN_LEFT
     * o Dk_Ui_Table_Cell::ALIGN_RIGHT
     * 
     * @param string $str
     */
    function setAlign($str)
    {
        if ($str != self::ALIGN_CENTER && $str != self::ALIGN_LEFT && $str != self::ALIGN_RIGHT) {
            throw new Dk_ExceptionIllegalArgument('Invalid alignment value');
        }
        $this->align = $str;
    }
    
    /**
     * get the alignment value of this cell
     * 
     * Valid return values are:
     * o Dk_Ui_Table_Cell::ALIGN_CENTER
     * o Dk_Ui_Table_Cell::ALIGN_LEFT
     * o Dk_Ui_Table_Cell::ALIGN_RIGHT
     * 
     * @return string
     */
    function getAlign()
    {
        return $this->align;
    }
    
    /**
     * Add a row class for rendering
     *
     * @param string $class
     */
    function addRowClass($class)
    {
        $this->rowClass[$class] = $class;
    }
    
    /**
     * Get all the additional classes to attach to a row
     *
     * @return array
     */
    function getRowClassList()
    {
        return $this->rowClass;
    }
    
    /**
     * Get the orderBy status of this cell
     *
     * @return string
     */
    function getOrderBy()
    {
        $request = Dk_Request::getInstance();
        $eventKey = $this->getEventKey('orderBy');
        $pre = '`' . $this->property . '` ';
        if ($request->getParameter($eventKey) == $pre . self::ORDER_ASC) {
            return self::ORDER_ASC;
        } else if ($request->getParameter($eventKey) == $pre . self::ORDER_DESC) {
            return self::ORDER_DESC;
        } else {
            return self::ORDER_NONE;
        }
    }
    
    /**
     * getOrderByUrl
     *
     * @param string $eventKey
     * @return Dk_Util_Url
     */
    function getOrderByUrl()
    {
        $request = Dk_Request::getInstance();
        $eventKey = $this->getEventKey('orderBy');
        $url = $request->getRequestUri();
        $url->removeQueryField($eventKey);
        $pre = '`' . $this->property . '` ';
        if ($request->getParameter($eventKey) == $pre . self::ORDER_ASC) {
            $url->setQueryField($eventKey, $pre . self::ORDER_DESC);
        } else if ($request->getParameter($eventKey) == $pre . self::ORDER_DESC) {
            $url->removeQueryField($eventKey);
        } else {
            $url->setQueryField($eventKey,  $pre . self::ORDER_ASC);
        }
        return $url;
    }
}
?>