<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */
 

/**
 * Render an array of Dk objects to a table
 *  
 * 
 * @package Ui
 */
class Dk_Ui_Table_Base extends Dk_Web_Component
{
    
    /**
     * @var Dk_Db_Array
     */
    protected $list = null;
    
    /**
     * @var Dk_Ui_Table_Cell[]
     */
    protected $cells = array();
    
    /**
     * @var Dk_Ui_Table_Action[]
     */
    protected $actions = array();
    
    /**
     * @var Dk_Ui_Table_ActionInterface[]
     */
    protected $filters = array();
    
    
    
    
    
    /**
     * Create the object instance
     * 
     * @param Dk_Db_Array $list
     */
    function __construct(Dk_Db_Array $list) 
    {
        $this->list = $list;
        parent::__construct();
    }
    
    
    /**
     * Execute this component
     * Execute any process code in the cells, actions or filters.
     * This can be used to execute events in these objects 
     * 
     * @return mixed
     */
    function execute()
    {
        parent::execute();
        //  
        foreach($this->cells as $cell) {
            $cell->doProcess();
        }
        foreach($this->actions as $action) {
            $action->doProcess();
        }
        foreach($this->filters as $action) {
            $action->doProcess();
        }
    }
    
    /**
     * Return the class name of the items we are dealing with in the list.
     * Returns null if not defined
     *
     * @return string
     */
    function getListClass()
    {
        if ($this->list->count()) {
            $obj = $this->list->getFirst();
            return get_class($obj);
        }
        return '';
    }
    
    /**
     * Set this ID same as the parent component ID
     *
     * @param Dk_Web_Component $component
     */
    protected function setParent(Dk_Web_Component $component)
    {
        $this->id = $component->getId();
        parent::setParent($component);
    }
    
    /**
     * makeTemplate
     *
     * @return string
     */
    protected function __makeTemplate()
    {
        $xmlStr = sprintf('<?xml version="1.0"?>
<div class="Dk_Ui_Table_Base" choice="table">
  <form id="%s" method="post" var="formId">
    <div class="actions" choice="actions">
      <div class="actionRow">
        <div class="left" choice="action">
          <span var="actionCell" repeat="actionCell"></span>
          %s
        </div>
        <div class="right" choice="filter">
          <a class="toggle" href="javascript:;" title="Action Link" onclick="$(\'#filterRow\').toggle(\'slow\');">Hide Filter</a>
        </div>
      </div>
      <div class="filterRow" id="filterRow" choice="filter">
        <div var="filterCell" repeat="filterCell"></div>
        %s
      </div>
      <div class="clear"></div>
    </div>
    
    <div class="pager">
      <table border="0" cellpadding="0" cellspacing="0"><tr>
        <td>
          <div var="Dk_Ui_Results" />
        </td>
        <td>
          <div var="Dk_Ui_Pager" />
        </td>
        <td>
          <div var="Dk_Ui_Limit" />
        </td>
      </tr></table>
    </div>
    <table border="0" cellpadding="0" cellspacing="0" class="manager">
      <thead>
        <tr>
          <th var="th" repeat="th"></th>
        </tr>
      </thead>
      <tbody>
        <tr var="tr" repeat="tr">
          <td var="td" repeat="td"></td>
        </tr>
      </tbody>
    </table>
    <div class="pager">
      <table border="0" cellpadding="0" cellspacing="0"><tr>
        <td class="r">
          <div var="Dk_Ui_Results" />
        </td>
        <td class="p">
          <div var="Dk_Ui_Pager" />
        </td>
        <td class="l">
          <div var="Dk_Ui_Limit" />
        </td>
      </tr></table>
    </div>
    
  </form>
  <div class="clear" />
</div>
', $this->getFormId(), $this->getActionHtml(), $this->getFilterHtml());
        
        $template = Dk_Web_Template::loadXml($xmlStr);
        
        // Setup form
        $form = new Dk_Form_Object($this->getFormId());
        $this->setForm($form);
        
        /* @var $action Dk_Ui_Table_Action */
        foreach ($this->actions as $action) {
            $action->setFormFields($form);
        }
        /* @var $action Dk_Ui_Table_Action */
        foreach ($this->filters as $action) {
            $action->setFormFields($form);
        }
        
        return $template;
    }
    
    /**
     * Init the component
     *
     */
    function init()
    {
        $this->setTemplate($this->__makeTemplate());
        $this->getForm()->loadFromRequest($this->getRequest());
        
        $cLimit = Dk_Ui_Limit::makeFromList($this->list);
        $this->addChild($cLimit);
        
        $cPager = Dk_Ui_Pager::makeFromList($this->list);
        $this->addChild($cPager);
        
        $cResults = Dk_Ui_Results::makeFromList($this->list);
        $this->addChild($cResults);
    }
    
    
    /**
     * Render the widget.
     * 
     */
    function show($template = null) 
    {
        if ($this->list == null) {
            return;
        }
        $template->setChoice('table');
        $this->showTh($template);
        $this->showTd($template);
        
        if (count($this->actions) > 0 || count($this->filters) > 0) {
            $template->setChoice('actions');
        }
        
        if (count($this->actions) > 0) {
            $template->setChoice('action');
        }
        if (count($this->filters) > 0) {
            $template->setChoice('filter');
        }
        
    }
    
    /**
     * show the action icons
     *
     * @param Dom_Template $template
     */
    function getActionHtml()
    {
        $html = '';
        /* @var $action Dk_Ui_Table_Action */
        foreach ($this->actions as $action) {
            
            $html .= $action->getHtml() . "\n";
        }
        return $html;
    }
    
    
    /**
     * show the action icons
     *
     */
    function getFilterHtml()
    {
        $html = '';
        /* @var $action Dk_Ui_Table_Action */
        foreach ($this->filters as $action) {
            $html .= $action->getHtml() . "\n";
        }
        return $html;
    }
    
    
    
    /**
     * Render the table headers
     *
     * @param Dom_Template $template
     */
    function showTh(Dom_Template $template)
    {
        /* @var $cell Dk_Ui_Table_Cell */
        foreach ($this->cells as $cell) {
            $repeat = $template->getRepeat('th');
            $class = $repeat->getAttr('th', 'class');
            $repeat->replaceHtml('th', $cell->getTableHeader());
            
            if ($cell->getOrderBy() == Dk_Ui_Table_Cell::ORDER_ASC) {
                $class .= ' orderAsc';
            } else if ($cell->getOrderBy() == Dk_Ui_Table_Cell::ORDER_DESC) {
                $class .= ' orderDesc';
            }
            if ($cell->isKey()) {
                $class .= ' key';
            }
            $repeat->replaceAttr('th', 'class', $class);
            $repeat->append();
        }
    }
    
    /**
     * Render the table data rows
     *
     * @param Dom_Template $template
     */
    function showTd(Dom_Template $template)
    {
        $idx = 0;
        /* @var $obj Dk_Object */
        foreach ($this->list as $obj) {
            $repeatRow = $template->getRepeat('tr');
            $rowClassArr = array();
            /* @var $cell Dk_Ui_Table_Cell */
            foreach ($this->cells as $k => $cell) {
                $repeat = $repeatRow->getRepeat('td');
                
                $class = $repeat->getAttr('td', 'class');
                
                $data = $cell->getTableData($obj);
                if ($data === null) {
                    $data = '&nbsp;';
                }
                $repeat->replaceHTML('td', $data);
                
                
                if ($cell->getProperty()) {
                    $class .= ' m' . ucfirst($cell->getProperty());
                }
                $class .= ' ' . $cell->getAlign();
                $repeat->replaceAttr('td', 'class', $class);
                
                $rowClassArr = array_merge($rowClassArr, $cell->getRowClassList());
                $repeat->append();
            }
            
            $rowClass = $repeatRow->getAttr('tr', 'class') . ' ';
            if (count($rowClassArr) > 0) {
                $rowClass .= implode(' ', $rowClassArr);
            }
            
            if ($idx%2) {
                $repeatRow->replaceAttr('tr', 'class', 'odd ' . $rowClass);
            } else {
                $repeatRow->replaceAttr('tr', 'class', 'even ' . $rowClass);
            }
            $idx++;
            $repeatRow->append();
        }
    }
    

    
    /**
     * Add an action to this table
     *
     * @param Dk_Ui_Table_ActionInterface $action
     */
    function addAction(Dk_Ui_Table_ActionInterface $action)
    {
        $action->setTable($this);
        $this->actions[] = $action; 
    }
    
    /**
     * Add an action to this table
     *
     * @param Dk_Ui_Table_ActionInterface $action
     */
    function addFilter(Dk_Ui_Table_ActionInterface $action)
    {
        $action->setTable($this);
        $this->filters[] = $action; 
    }
    
    /**
     * Add a cell to this table
     *
     * @param Dk_Ui_Table_Cell $cell
     */
    function addCell($cell)
    {
        $cell->setTable($this);
        $this->cells[] = $cell;
    }
    
    /**
     * Set the cells, init with the table
     *
     * @param Dk_Ui_Table_Cell[] $array
     */
    function setCells($array)
    {
        foreach ($array as $cell) {
            $cell->setTable($this);
        }
        $this->cells = $array;
    }
    
    /**
     * Get a cell from the array by its property name
     *
     * @param string $property
     */
    function getCell($property)
    {
        if (array_key_exists($property, $this->cells)) {
            return $this->cells[$property];
        }
    }
    
    /**
     * Get the orderBy String
     *
     * @return string
     */
    function getOrderBy()
    {
        if (Dk_Request::getInstance()->exists($this->getEventKey('orderBy'))) {
            return Dk_Request::getInstance()->getParameter($this->getEventKey('orderBy'));
        }
        return '';
    }
    
    /**
     * Get the list of items given to this table
     *
     * @return Dk_Db_Array
     */
    function getList()
    {
        return $this->list;
    }
    
    /**
     * Get the form id after rendering
     * Useful for calling javascript function that do operations on this form.
     *
     * @return string
     */
    function getFormId()
    {
        return 'Table_' . $this->getId();
    }
}
?>
