<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */


/**
 * A pager component that pagenates table data.
 * 
 * 
 * @package Ui
 */
class Dk_Ui_Pager extends Dk_Web_Component
{
    
    /**
     * @var integer
     */
    private $size = 0;
    
    /**
     * @var integer
     */
    private $limit = 0;
    
    /**
     * @var integer
     */
    private $offset = 0;
    
    /**
     * @var integer
     */
    private $maxPages = 10;
    
    
    
    /**
     * Create the pagenator class
     *
     * @param integer $size
     * @param integer $limit
     * @param integer $offset
     */
    function __construct($size = 0, $limit = 25, $offset = 0) 
    {
        $this->size   = intval($size);
        $this->limit  = intval($limit);
        $this->offset = intval($offset);
        parent::__construct();
    }
    
    /**
     * Make a pager from a db list
     * 
     * @return Dk_Ui_Pager
     */
    static function makeFromList(Dk_Db_Array $list)
    {
        if (!$list instanceof Dk_Db_Array) {
            throw new Dk_ExceptionIllegalArgument('Invalid list object.');
        }
        return new self($list->getTotal(), $list->getLoo()->getLimit(), $list->getLoo()->getOffset());
    }
    
    /**
     * Set this ID same as the parent component ID
     *
     * @param Dk_Web_Component $component
     */
    protected function setParent(Dk_Web_Component $component)
    {
        $this->id = $component->getId();
        parent::setParent($component);
    }
    
    /**
     * makeTemplate
     *
     * @return string
     */
    protected function __makeTemplate()
    {
        $xmlStr = '<?xml version="1.0"?>
<ul class="Dk_Ui_Pager" choice="Dk_Ui_Pager">
  <li class="start" var="start"><a href="javascript:;" var="startUrl" rel="nofollow">Start</a></li>
  <li class="back" var="back"><a href="javascript:;" var="backUrl">&lt;&lt;</a></li>
  
  <li repeat="page" var="page"><a href="javascript:;" var="pageUrl" rel="nofollow"></a></li>
  
  <li class="next" var="next"><a href="javascript:;" var="nextUrl">&gt;&gt;</a></li>
  <li class="end" var="end"><a href="javascript:;" var="endUrl" rel="nofollow">End</a></li>
</ul>';
        $template = Dk_Web_Template::loadXml($xmlStr);
        return $template;
    }
    
    /**
     * Render the widget.
     * 
     */
    function show($template = null) 
    {
        $currentPage = 0;
        $numPages = 1;
        
        if ($this->limit > -1 && $this->limit < $this->size) {
            $numPages = 0;
            $currentPage = 0;
            if ($this->limit > 0) {
                $numPages = ceil($this->size / $this->limit);
                $currentPage = ceil($this->offset / $this->limit);
            }
            
            $startPage = 0;
            $endPage = $this->maxPages;
            $center = floor($this->maxPages/2);
            if ($currentPage > $center) {
                $startPage = $currentPage - $center;
                $endPage = $startPage + $this->maxPages;
            }
            if ($startPage > $numPages - $this->maxPages) {
                $startPage = $numPages - $this->maxPages;
                $endPage = $numPages;
            }
            
            if ($startPage < 0) {
                $startPage = 0;
            }
            if ($endPage >= $numPages) {
                $endPage = $numPages; 
            }
            
            if ($numPages > 0) {
                $template->setChoice('Dk_Ui_Pager');
            }
            
            $pageUrl = Dk_Request::getInstance()->getRequestUri();
            $pageUrl->removeQueryField($this->getEventKey('offset'));
            
            for ($i = $startPage; $i < $endPage; $i++) {
                $repeat = $template->getRepeat('page');
                $repeat->replaceText('pageUrl', $i+1);
                $repeat->replaceAttr('pageUrl', 'title', 'Page ' . ($i+1));
                $pageUrl->setQueryField($this->getEventKey('offset'), $i*$this->limit);
                $repeat->replaceAttr('pageUrl', 'href', $pageUrl->toString());
                if ($i == $currentPage) {
                    $repeat->replaceAttr('page', 'class', 'selected');
                    $repeat->replaceAttr('pageUrl', 'title', 'Current Page ' . ($i+1));
                }
                $repeat->append();
            }
            
            if ($this->offset >= $this->limit) {
                $pageUrl->setQueryField($this->getEventKey('offset'), $this->offset - $this->limit);
                $template->replaceAttr('backUrl', 'href', $pageUrl->toString());
                $template->replaceAttr('backUrl', 'title', 'Previous Page');
                $pageUrl->setQueryField($this->getEventKey('offset'), 0);
                $template->replaceAttr('startUrl', 'href', $pageUrl->toString());
                $template->replaceAttr('startUrl', 'title', 'Start Page');
            } else {
                $template->replaceAttr('start', 'class', 'off');
                $template->replaceAttr('back', 'class', 'off');
            }
            
            if ($this->offset < ($this->size - $this->limit)) {
                $pageUrl->setQueryField($this->getEventKey('offset'), $this->offset + $this->limit);
                $template->replaceAttr('nextUrl', 'href', $pageUrl->toString());
                $template->replaceAttr('nextUrl', 'title', 'Next Page');
                $pageUrl->setQueryField($this->getEventKey('offset'), ($numPages-1) * $this->limit);
                $template->replaceAttr('endUrl', 'href', $pageUrl->toString());
                $template->replaceAttr('endUrl', 'title', 'Last Page');
            } else {   
                $template->replaceAttr('end', 'class', 'off');
                $template->replaceAttr('next', 'class', 'off');          
            }
        }
    }
    
    /**
     * Set the maximum number of page values to display
     * Default: 10 page numbers
     * 
     * @param integer $i
     */
    function setMaxPages($i)
    {
        $this->maxPages = $i;
    }
    
}
?>