<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * An OO Wrapper around a HTTP response.
 * 
 * @package Dk
 */
class Dk_Response extends Dk_Object
{
    
    /**
     * Status code 404 indicating that the requested resource is not 
     *  available.
     * @var integer
     */
    const SC_NOT_FOUND = 404;
    /**
     * Status code 200 indicating all is OK
     * @var integer
     */
    const SC_OK = 200;

    /**
     * Status code (500) indicating an error inside the HTTP server which 
     * prevented it from fulfilling the request.
     * @var integer
     */
    const SC_INTERNAL_SERVER_ERROR = 500;
    
    /**
     * Redirect 301 Moved Permanently. 
     * Convert to GETConfirm re-POST
     * @var integer
     */
    const SC_REDIRECT_MOVED_PERMANENTLY = 301;
    
    /**
     * Redirect 302 Found. 
     * Confirm re-POST
     * @var integer
     */
    const SC_REDIRECT_FOUND = 302;
    
    /**
     * Redirect 303 See Other. 
     * dont cache, always use GET
     * @var integer
     */
    const SC_REDIRECT_SEE_OTHER = 303;
    
    /**
     * Redirect 304 Not Modified. 
     * use cache
     * @var integer
     */
    const SC_REDIRECT_NOT_MODIFIED = 304;
    
    /**
     * Redirect 305 Use Proxy. 
     * @var integer
     */
    const SC_REDIRECT_USE_PROXY = 305;
    
    /**
     * Redirect 307 Temorary Redirect. 
     * @var integer
     */
    const SC_REDIRECT_TEMPORARY_REDIRECT = 307;
    
    
    /**
     * @var array
     */
    static $statusCodes = array(
        Dk_Response::SC_OK,
        Dk_Response::SC_NOT_FOUND,
        Dk_Response::SC_INTERNAL_SERVER_ERROR,
        Dk_Response::SC_REDIRECT_MOVED_PERMANENTLY,
        Dk_Response::SC_REDIRECT_FOUND,
        Dk_Response::SC_REDIRECT_SEE_OTHER,
        Dk_Response::SC_REDIRECT_NOT_MODIFIED,
        Dk_Response::SC_REDIRECT_USE_PROXY,
        Dk_Response::SC_REDIRECT_TEMPORARY_REDIRECT
        );

    /**
     * @var array
     */
    static $statusText = array(
        Dk_Response::SC_OK => 'OK',
        Dk_Response::SC_NOT_FOUND => 'Resource Not Found',
        Dk_Response::SC_INTERNAL_SERVER_ERROR => 'Internal Server Error',
        Dk_Response::SC_REDIRECT_MOVED_PERMANENTLY => 'Moved Permanently',
        Dk_Response::SC_REDIRECT_FOUND => 'Found',
        Dk_Response::SC_REDIRECT_SEE_OTHER => 'See Other',
        Dk_Response::SC_REDIRECT_NOT_MODIFIED => 'Not Modified',
        Dk_Response::SC_REDIRECT_USE_PROXY => 'Use Proxy',
        Dk_Response::SC_REDIRECT_TEMPORARY_REDIRECT => 'Temporary Redirect'
        );
    
    /**
     * Valid 404 error page names
     * @var array
     */
    static $errorPgs = array('error.html', 'error.htm', 'error.php', '404.html', '404.htm', '404.php');
    
    
    /**
     * @var string
     */
    static $errorTemplate = '<html>
  <head>
    <title>%s %s</title>
    <style type="text/css">
      body {
        margin: 0;
        padding: 0;
      }
      h1 {
       margin: 0px;
       padding: 4px 10px;
       background: #369;
       color: #FFF;
      }
      pre {
        color: #000;
        padding: 10px;
        font-size: 12px;
      }
    </style>
  </head>
  <body>
    <h1>%s</h1>
    <pre class="msg">%s</pre>
  </body>
</html>';

    /**
     * @var string
     */
    private $buffer = '';

    /**
     * @var boolean
     */
    private $committed = false;
    
    /**
     * @var array
     */
      private $headers = array();
      
    /**
     * @var array
     */
      private $cookies = array();
      
      
      
      
    /**
     * @var Dk_Response
     */
    static protected $instance = null;
    
    
    
    
    /**
     * Sigleton, No instances can be created.
     * Use:
     *   Dk_Response::getInstance()
     */
    private function __construct() { }
    
    /**
     * Get an instance of this object
     * 
     * @return Dk_Response
     */
    static function getInstance() 
    {
        if (self::$instance == null) {
            self::$instance = new Dk_Response();
        }
        return self::$instance;
    }
    
    
    
    /**
     * Add a header value to send with the response.
     * 
     * @param string $name
     * @param string $value
     */
    function addHeader($name, $value = '')
    {
        $this->headers[$name] = $value;
    }
  
    /**
     * Add a cookie value to send with the response.
     * 
     * 
     * @param string $name
     * @param string $value
     * @param Dk_Util_Date $expire
     * @param string $path
     * @param string $domain
     * @param boolean $secure
     * @param boolean $httponly
     */
    function addCookie($name, $value = '', $expire = null, $path = '', $domain = '', 
        $secure = false, $httponly = false)
    {
        $this->cookies[$name] = array();
        $this->cookies[$name]['value'] = $value;
        if ($expire instanceof Dk_Util_Date) {
            $this->cookies[$name]['expire'] = $expire->getTimestamp();
        } else {
            $this->cookies[$name]['expire'] = 0;
        }
        $this->cookies[$name]['path'] = $path;
        $this->cookies[$name]['domain'] = $domain;
        $this->cookies[$name]['secure'] = $secure;
        $this->cookies[$name]['httponly'] = $httponly;
    }
    
    /**
     * Forces any content in the buffer to be written to the client. 
     *
     * A call to this method automatically commits the response, meaning the 
     * status code and headers will be written.
     * 
     */
    function flushBuffer()
    {
        $this->committed = true;
        $this->flushHeaders();
        $this->flushCookies();
        $this->buffer = str_replace('<?xml version="1.0"?>'."\n", "", $this->buffer);
        echo $this->buffer;
    }
    
    /**
     * Write any cookies to the buffer
     *
     */
    private function flushCookies()
    {
        foreach ($this->cookies as $name => $cookie) {
            setcookie(
                $name,
                $this->cookies[$name]['value'],
                $this->cookies[$name]['expire'],
                $this->cookies[$name]['path'],
                $this->cookies[$name]['domain'],
                $this->cookies[$name]['secure'],
                $this->cookies[$name]['httponly']
            );
        }
        $this->cookies = array();
    }
    
    /**
     * Write any headers to the buffer
     *
     */
    private function flushHeaders()
    {
        if (in_array(Dk_Request::getInstance()->getRequestUri()->getBasename(), self::$errorPgs)) {
            $this->addHeader('Status', 404);
        } 

        // Date in the past
        header("Expires: Mon, 26 Jul 1997 05:00:00 GMT");
        header("Last-Modified: " . gmdate("D, d M Y H:i:s") . " GMT");
        header("Cache-Control: no-store, no-cache, must-revalidate");
        header("Cache-Control: post-check=0, pre-check=0", false);
        header("Pragma: no-cache"); 

        foreach ($this->headers as $name => $value) {
            if (strtolower($name) == 'status') {
                header($_SERVER['SERVER_PROTOCOL'] . ' ' . $value . self::$statusText[$value]);
            } else {
                header("$name: $value");
            }
        }
        $this->headers = array();
    }

    /**
     * Returns a boolean indicating if the response has been committed. 
     *
     * A committed response has already had its status code and headers written.
     * @return boolean
     */
    function isCommitted()
    {
        return $this->committed;
    }

    /**
     * Clears any data that exists in the buffer.
     * 
     * @throws Dk_Exception
     */
    function reset()
    {
        if ($this->committed) {
            throw new Dk_Exception('1000: The response has already been committed.');
        }
        $this->buffer = '';
    }

    /**
     * Sends an error response to the client using the specified status. 
     *
     * The response will look like an HTML-formatted server error page 
     * containing the specified message, The the content type will be set to 
     * "text/html", and cookies and other headers will be left unmodified.
     *
     * If the response has already been committed, this method throws an 
     * IllegalStateException. After using this method, the response should be 
     * considered to be committed and should not be written to.
     *
     * @param const $statusCode The error status code
     * @param string $msg An optional descriptive message
     * @param Dk_Web_Dom_Template $template An optional template to use
     * @throws Dk_Exception
     *
     */
    function sendError($statusCode, $msg = '')
    {
        if (!in_array($statusCode, self::$statusCodes)) {
            throw new Dk_Exception('Invalid status code');
        }
        if ($this->committed) {
            throw new Dk_Exception('1001: The response has already been committed.');
        }
        
        $this->reset();
        @header(self::$statusText[$statusCode], true, $statusCode);
        
        $config = Dk_Config::getInstance();
        $templateFile = $config->getFileRoot() . '/html/error.html';
        // TODO: look for language error pages
        if (!$config->isDebugMode() && is_file($templateFile) && Dk_Request::getInstance()->getRequestUri()->getBasename() != 'error.html') {
            $url = new Dk_Util_Url('/error.html');
            $url->redirect();
        }
        
        $this->write(sprintf(self::$errorTemplate, 
            $statusCode, self::$statusText[$statusCode],
            $statusCode . ' ' . self::$statusText[$statusCode], $msg));
        
        $this->committed = true;
        $this->flushBuffer();
        exit;
    }
    

    /**
     * Writes to the response buffer.
     *
     * @param string
     * @throws Dk_Exception
     */
    function write($data)
    {
        if ($this->committed) {
            throw new Dk_Exception('1002: The response has already been committed.');
        }
        $this->buffer .= $data;
    }    

    /**
     * Returns a textual representation of the object.
     *
     * @return string
     */
    function __toString()
    {
        return $this->buffer;
    }    
}
?>