<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * This object handles the all HttpRequests.
 * 
 * @package Dk
 */
class Dk_Request extends Dk_Object
{
    /**
     * @var Dk_Request
     */
    static protected $instance = null;
    
    
    
    /**
     * Sigleton, No instances can be created.
     * Use:
     *   Dk_Request::getInstance()
     */
    private function __construct() { }
    
    /**
     * Get an instance of this object
     * 
     * @return Dk_Request
     */
    static function getInstance() 
    {
        if (self::$instance == null) {
            self::$instance = new Dk_Request();
        }
        return self::$instance;
    }
    
    
    
    /**
     * Returns the referering Url.
     *
     * @return Dk_Util_Url Returns null if there was no referer.
     */
    function getReferer()
    {
        $referer = $this->getServerParameter('HTTP_REFERER');
        if ($referer != null) {
            return new Dk_Util_Url($referer);
        } else {
            return null;
        }
    }

    /**
     * Returns the URI which was given in order to access the page.
     * 
     * For example, '/index.html'.
     * 
     * @return Dk_Util_Url
     */
    function getRequestUri()
    {
        $urlStr = $this->getServerParameter('REQUEST_URI');
        if (!ereg('^http', $this->getServerParameter('REQUEST_URI'))) {
            $urlStr = 'http://'.$this->getServerParameter('HTTP_HOST') . $urlStr;
        }
        return new Dk_Util_Url($urlStr);
    }
    
    /**
     * Binds data to this request, using the name specified.
     *
     * @param string $name A key to retrieve the data.
     * @param mixed $value 
     */
    function setParameter($name, $value)
    {
        if ($value === null) {
            unset($_REQUEST[$name]);
        } else {
            $_REQUEST[$name] = $value;
        }
    }
    
    /**
     * Returns the value of a request parameter as a String, 
     * or null if the parameter does not exist. 
     *
     * You should only use this method when you are sure the parameter has 
     * only one value. If the parameter might have more than one value, use 
     * getParameterValues().
     *
     * If you use this method with a multivalued parameter, the value returned 
     * is equal to the first value in the array returned by getParameterValues. 
     *
     * @param string $name The parameter name.
     * @return mixed
     */
    function getParameter($name)
    {
        if (isset($_REQUEST[$name])) {
            if (is_array($_REQUEST[$name])) {
                $value = $_REQUEST[$name][0];
            } else {
                $value = $_REQUEST[$name];
            }
        } else {
            $value = null;
        }
        
        return $value;
    }
    
    /**
     * Returns an array of String objects containing all of the values the 
     * given request parameter has, or null if the parameter does not exist.
     *
     * If the parameter has a single value, the array has a length of 1.
     *
     * @param string $name
     * @return array
     */
    function getParameterValues($name)
    {
        if (isset($_REQUEST[$name])) {
            if (is_array($_REQUEST[$name])) {
                $value = $_REQUEST[$name];
            } else {
                $value = array($_REQUEST[$name]);
            }
        } else {
            $value = null;
        }
        return $value;
    }
    
    /**
     * Get the request array map
     *
     * @return array
     */
    function getAllParameters()
    {
        return $_REQUEST;
    }
    
    /**
     * Check if a parameter name exists in the request
     *
     * @param string $name
     * @return boolean
     */
    function exists($name)
    {
        return isset($_REQUEST[$name]);
    }
    
    /**
     * Returns an array containing the names of the parameters contained in 
     * this request.
     *
     * @return array
     */
    function getParameterNames()
    {
        return array_keys($_REQUEST);
    }

    /**
     * Returns the value of a request parameter.
     *
     * @return string The value or null if the parameter is not in the request.
     */
    private function getServerParameter($name)
    {
        if (array_key_exists($name, $_SERVER)) {
            return $_SERVER[$name];
        } else {
            return null;
        }
    }
    
    /**
     * Get the IP of the clients machine.
     *
     * @return string
     */
    function getRemoteAddr()
    {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {              // check ip from share internet
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {  // to check ip is pass from proxy
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else if (!empty($_SERVER['REMOTE_ADDR'])) {          // User remote address
            $ip = $_SERVER['REMOTE_ADDR'];
        } else {
            $ip = '0.0.0.0';
        }
        return $ip;
    }
    
    /**
     * Get the browser userAgent string
     *
     * @return string
     */
    function getUserAgent()
    {
        if (!empty($_SERVER['HTTP_USER_AGENT'])) {
           return $_SERVER['HTTP_USER_AGENT'];
        }
        return '';
    }
    
    /**
     * Get the remote hostname if available
     *
     * @return string
     */
    function getRemoteHost()
    {
        return gethostbyaddr($this->getRemoteAddr());
    }

    /**
     * Returns the query string that is contained in the request URL after the path.
     * 
     * @return string
     */
    function getQueryString()
    {
        return $_SERVER['QUERY_STRING'];
    }
    
    /**
     * Returns the name of the HTTP method with which this request was made.
     *
     * For example, GET, POST, or PUT.
     *
     * @return string
     */
    function getMethod()
    {
        return $_SERVER['REQUEST_METHOD'];
    }
    
    /**
     * Returns the raw post data.
     * 
     * @return string
     */
    function getRawPostData()
    {
        global $HTTP_RAW_POST_DATA;
        return $HTTP_RAW_POST_DATA;
    }
}
?>