<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * This object creates and loads other objects to avoid using any
 * getters and setters. Useful for DB/XML objects
 *
 * @package Util
 */
class Dk_Mapper_ObjectLoader
{
    private $nc = "\0";

    /**
     * @var array
     */
     protected static $serialTemplates = array();


    /**
     * @var Dk_Db_ObjectLoader
     */
    protected static $instance = null;




    /**
     * Sigleton, No instances can be created.
     * Use:
     *   Dk_Db_ObjectLoader::getInstance()
     */
    private function __construct() { }

    /**
     * Get an instance of this object
     *
     * @return Dk_Db_ObjectLoader
     */
    static function getInstance()
    {
        if (self::$instance == null) {
            self::$instance = new Dk_Mapper_ObjectLoader();
        }
        return self::$instance;
    }





    /**
     * Loads a database row into a new object instance.
     *
     * @param array $row
     * @param Dk_Mapper_Object mapper
     * @return return object
     */
    function doLoad($row, Dk_Mapper_Object $mapper)
    {
        $template = $this->getSerialTemplate($mapper->getColumnMaps(), $mapper->getClass());
        $serializedClass = $template[':class:'];
        /* @var $columnMap Dk_Mapper_ColumnMap */
        foreach ($mapper->getColumnMaps() as $property => $columnMap) {
            $serializedClass .= $template[$property]['start'];
            $type = $template[$property]['type'];
            $end = $template[$property]['end'];
            
            $value = $columnMap->getSerialValue($row);
            if ($columnMap->getPropertyType() == Dk_Mapper_ColumnMap::CM_BOOLEAN) {
                $value = ($value == true) ? 1 : 0;
            } elseif ($columnMap->getPropertyType() == Dk_Mapper_ColumnMap::CM_STRING) {
                $value = strlen($value) . ':"' . $value . '"';
            //} elseif (is_null($value)) {
            } elseif ($value === null) {
                $value = 'N';
                $type = '';
                $end = ';';
            }
            /*
            if (is_bool($value)) {
                $value = $value === true ? 1 : 0;
            } elseif (is_string($value)) {
                $value = strlen($value) . ':"' . $value . '"';
            } elseif (is_null($value)) {
                $value = 'N';
                $type = '';
                $end = ';';
            }
            */
            $serializedClass .= $type . $value . $end;
        }

        $serializedClass .= '}';
        
        $obj = unserialize($serializedClass);
        if ($obj === false) {
            throw new Dk_Exception('Error unserialising object: - Check variable names for correctness. ');
        }
        return $obj;
    }

    /**
     * generate the template that will be used to create the serialised object
     *
     * @param array $columnMaps
     * @param string $className
     * @return array
     */
    private function getSerialTemplate($columnMaps, $className)
    {
        static $template = null;

        if (array_key_exists($className, self::$serialTemplates)) {
            return self::$serialTemplates[$className];
        }

        $template = array();
        $properties = array();
        $class = new ReflectionClass($className);

        do {
            $properties = array_merge($properties, $class->getProperties());
            $class = $class->getParentClass();
        } while ($class != null);

        $template[':class:'] = "O:" . strlen($className) . ':"'
            . $className . '":' . count($columnMaps) . ':{';

        foreach ($properties as $property) {
            $name = $property->getName();
            if (!array_key_exists($name, $columnMaps)) {
                continue;
            }
            if ($property->isPrivate()) {
                $serializedName = "\0"
                    . $property->getDeclaringClass()->getName() . "\0"
                    . $name;
            } elseif ($property->isProtected()) {
                $serializedName = "\0*\0" . $name;
            } else {
                $serializedName = $name;
            }

            $start = 's:' . strlen($serializedName) . ':"'
                . $serializedName . '";';
            $columnMap = $columnMaps[$name];
            $type = $this->getSerialType($columnMap->getPropertyType(), $columnMap);

            if ($type{0} == 'O' || $type{0} == 'a') {
                $end = ";}";
            } else {
                $end = ";";
            }
            $template[$name] = array(
                'start' => $start,
                'type' => $type,
                'end' => $end);
        }

        self::$serialTemplates[$className] = $template;

        return $template;
    }

    /**
     * get the property type string for the serialised object
     *
     * @param string $propertyType
     * @param Dk_Mapper_ColumnMap $columnMap
     * @return string
     */
    private function getSerialType($propertyType, $columnMap)
    {
        switch ($propertyType) {
            case Dk_Mapper_ColumnMap::CM_BOOLEAN:
                $type = 'b:';
                break;
            case Dk_Mapper_ColumnMap::CM_FLOAT:
                $type = 'd:';
                break;
            case Dk_Mapper_ColumnMap::CM_INTEGER:
                $type = 'i:';
                break;
            case Dk_Mapper_ColumnMap::CM_ENCRYPT_STRING:
            case Dk_Mapper_ColumnMap::CM_STRING:
                $type = 's:';
                break;
            case Dk_Mapper_ColumnMap::CM_ARRAY:
                $type = 'a:';
                break;
            default:
                $objProperty = $columnMap->getSerialName();
                $objPropertyType = $this->getSerialType($columnMap->getSerialType(), $columnMap);
                $typeLen = strlen($propertyType);
                $type = 'O:' . $typeLen . ':"' . $propertyType . '":1:{s:' .
                    (strlen($objProperty) + $typeLen + 2) .
                    ':"' . "\0" . $propertyType . "\0" . $objProperty . '";' .
                    $objPropertyType;
                break;
        }

        return $type;
    }



    /**
     * Get an object's property and values
     *
     * @param mixed $obj
     * @return array
     */
    function getObjectValues($obj)
    {
        return $this->doGetObjectValues(serialize($obj));
    }

    /**
     * get the property-value map from a serialised object
     *
     * @param string $serializedObj
     * @return array
     */
    private function doGetObjectValues($serializedObj)
    {
        $values = array();

        $pos = strpos($serializedObj, ':', 2);
        $len = intval(substr($serializedObj, 2, $pos));
        $class = substr($serializedObj, $pos + 2, $len);

        if (substr($serializedObj, 0, 1) == 'a') {
            $classPrefix = array();
        } else {
            $classPrefix = array("\0" . $class . "\0" => $len + 2);
            while ($class = get_parent_class($class)) {
                $classPrefix["\0" . $class . "\0"] = strlen($class) + 2;
            }
        }
        $serializedObj = substr($serializedObj, strpos($serializedObj, '{') + 1, -1);

        while ($serializedObj != '') {
            $pad = 0;
            if (substr($serializedObj, 0, 2) == 'i:') {  // For array integer keys
                $len = $pos = strpos($serializedObj, ';');
                $name = substr($serializedObj, 2, $pos-2);
                $pad = 1;
            } else {
                $pos = strpos($serializedObj, ':', 2);
                $len = intval(substr($serializedObj, 2, $pos));
                $name = substr($serializedObj, $pos + 2, $len);
                $pad = strlen($len) + 6;
            }

            foreach ($classPrefix as $prefix => $prefixLen) {
                if (substr($name, 0, $prefixLen) == $prefix) {  // private vars
                    $name = substr($name, $prefixLen);
                    break;
                }
            }
            if (substr($name, 0, 3) == "\0*\0") {  // protected vars
                $name = substr($name, 3);
            }

            $serializedObj = substr($serializedObj, $len + $pad);
            $type = $serializedObj{0};
            $value = null;

            switch ($type) {
                case 'a':   // array
                case 'O':   // object

                    $pos = $this->getObjectBlockEnd($serializedObj);
                    $value = substr($serializedObj, 0, $pos+1);
                    $serializedObj = substr($serializedObj, $pos+1);
                    $value = $this->doGetObjectValues($value);
                    break;
                case 'i':  // integer
                    $pos = strpos($serializedObj, ';', 2);
                    $value = intval(substr($serializedObj, 2, $pos - 2));
                    $serializedObj = substr($serializedObj, $pos + 1);
                    break;
                case 's':  // string
                    $pos = strpos($serializedObj, ':', 2);
                    $len = intval(substr($serializedObj, 2, $pos));
                    $value = substr($serializedObj, $pos + 2, $len);
                    $serializedObj = substr($serializedObj, $pos + $len + 4);
                    break;
                case 'd':  // float
                    $pos = strpos($serializedObj, ';', 2);
                    $value = floatval(substr($serializedObj, 2, $pos - 2));
                    $serializedObj = substr($serializedObj, $pos + 1);
                    break;
                case 'b':  // boolean
                    $pos = strpos($serializedObj, ';', 2);
                    $value = (substr($serializedObj, 2, $pos - 2)) == 1;
                    $serializedObj = substr($serializedObj, $pos + 1);
                    break;
                case 'N':  // NULL values
                    $pos = strpos($serializedObj, ';', 1);
                    $value = null;
                    $serializedObj = substr($serializedObj, $pos + 1);
                    break;
                default:
                    break;
            }
            $values[$name] = $value;
        }
        return $values;
    }


    /**
     * find the char position of the closing '}' in a serialized string
     *
     * @param string $serializedObj
     */
    function getObjectBlockEnd($serializedObj)
    {
        $pos = strpos($serializedObj, ':', 2);
        $len = intval(substr($serializedObj, 2, $pos));
        $class = substr($serializedObj, $pos + 2, $len);

        if (substr($serializedObj, 0, 1) == 'a') {
            $index = $len;
        } else {
            $index = intval(substr($serializedObj, $pos+$len+4, strpos($serializedObj, '{')));
        }

        $totalPos = strpos($serializedObj, '{') + 1;
        $serializedObj = substr($serializedObj, strpos($serializedObj, '{') + 1);

        while ($index > 0) {
            if (substr($serializedObj, 0, 2) == 'i:') {  // For array integer keys
                $len = $pos = strpos($serializedObj, ';');
                $pad = 1;
            } else {
                $pos = strpos($serializedObj, ':', 2);
                $len = intval(substr($serializedObj, 2, $pos));
                $pad = strlen($len) + 6;
            }

            $totalPos += $len + $pad;
            $serializedObj = substr($serializedObj, $len + $pad);
            $type = $serializedObj{0};

            switch ($type) {
                case 'a':   // array
                case 'O':   // object
                    $pos = $this->getObjectBlockEnd($serializedObj);
                    $serializedObj = substr($serializedObj, $pos+1);
                    $totalPos += $pos+1;
                    break;
                case 'i':  // integer
                    $pos = strpos($serializedObj, ';', 2);
                    $serializedObj = substr($serializedObj, $pos + 1);
                    $totalPos += $pos+1;
                    break;
                case 's':  // string
                    $pos = strpos($serializedObj, ':', 2);
                    $len = intval(substr($serializedObj, 2, $pos));
                    $serializedObj = substr($serializedObj, $pos + $len + 4);
                    $totalPos += $pos + $len + 4;
                    break;
                case 'd':  // float
                    $pos = strpos($serializedObj, ';', 2);
                    $serializedObj = substr($serializedObj, $pos + 1);
                    $totalPos += $pos + 1;
                    break;
                case 'b':  // boolean
                    $pos = strpos($serializedObj, ';', 2);
                    $serializedObj = substr($serializedObj, $pos + 1);
                    $totalPos += $pos + 1;
                    break;
                case 'N':  // NULL values
                    $pos = strpos($serializedObj, ';', 1);
                    $serializedObj = substr($serializedObj, $pos + 1);
                    $totalPos += $pos + 1;
                    break;
                default:
                    throw new Dk_ExceptionIllegalArgument('Unknown argument, Fix your mapper or object: '. $class);
            }
            $index--;
        }
        return $totalPos;

    }

}
?>