<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */


/**
 * A data map class that maps database tables and columns to 
 *   Objects and properties.
 * 
 * @package Db
 */
class Dk_Mapper_DataMap extends Dk_Object 
{
    
    /**
     * @var array
     */
    private $columnMaps = array();
    
    /**
     * @var array
     */
    private $idColumnMaps = array();

    /**
     * @var array
     */
    private $params = array();
    
    
    
    /**
     * __construct
     * 
     */
    function __construct($params = array())
    {
        if (!is_array($params)) {
            throw new Dk_ExceptionIllegalArgument('Object is not of type array.');
        }
        $this->params = $params;
    }
    
    /**
     * Add a an element to the params array
     *
     * @param string $name
     * @param mixed $value
     */
    function addParameter($name, $value)
    {
        $this->params[$name] = $value;
    }
    
    /**
     * Get the mapper parameter array. 
     * This can be used to send data to objects that use the mapper
     *
     * @return array
     */
    function getParams()
    {
        return $this->params;
    }
    
    /**
     * Returns a parameter in the params array.
     *
     * @param string $name
     * @return mixed
     */
    function getParameter($name)
    {
        if (isset($this->params[$name])) {
            return $this->params[$name];
        }
    }
    
    /**
     * Makes an object Id from the database data.
     *
     * If the object has a multi field primary key then the object Id is the
     * concatenation of the primary key fields values.
     *
     * @param array $row
     * @return string
     */
    function makeObjectId($row)
    {
        $id = '';
        foreach ($this->idColumnMaps as $map) {
            $column = $map->getColumnName();
            $id .= $row[$column];
        }
        return $id;
    }
    
    /**
     * Sets the object ID column
     * 
     * @param string $column The column name
     * @param string $property The property name
     * @param string $propertyType The column type
     */
    function addIdColumn($column, $property, $propertyType)
    {
        $this->idColumnMaps[$property] = $this->makeColumnMap($column, $property, $propertyType);
    }
    
    /**
     * Gets the object ID columns.
     *
     * @return array An associative array of ID columns indexed by property.
     */
    function getIdColumns()
    {
        return $this->idColumnMaps;
    }
    
    /**
     * Add a column to this map
     * 
     * @param string $column The column name.
     * @param integer $property The property type.
     * @param string $propertyType The property name.
     */
    function addColumn($column, $property, $propertyType)
    {
        $this->columnMaps[$property] = $this->makeColumnMap($column, $property, $propertyType);
    }
    
    /**
     * Gets the column maps.
     * 
     * @return array
     */
    function getColumns()
    {
        return $this->columnMaps;
    }
    
    /**
     * Gets a column map by its property name
     * 
     * @return Dk_Mapper_ColumnMap
     */
    function getColumnByProperty($name)
    {
        return $this->columnMaps[$name];
    }
    
    /**
     * Make a column map
     *
     * @param string $column
     * @param string $property
     * @param string $propertyType
     * @return Dk_Mapper_ColumnMap
     * @throws Dk_RuntimeException
     */
    protected function makeColumnMap($column, $property, $propertyType)
    {
        switch($propertyType) {
            case Dk_Mapper_ColumnMap::CM_ENCRYPT_STRING :
                // TODO: fix encryption type
                //$class = 'Dk_Mapper_EncryptStringColumnMap'; 
                //break;
            case Dk_Mapper_ColumnMap::CM_STRING :
                $class = 'Dk_Mapper_StringColumnMap';
                break;
            case Dk_Mapper_ColumnMap::CM_INTEGER :
                $class = 'Dk_Mapper_IntegerColumnMap';
                break;
            case Dk_Mapper_ColumnMap::CM_FLOAT :
                $class = 'Dk_Mapper_FloatColumnMap';
                break;
            case Dk_Mapper_ColumnMap::CM_BOOLEAN :
                $class = 'Dk_Mapper_BooleanColumnMap';
                break;
            case Dk_Mapper_ColumnMap::CM_ARRAY :
                $class = 'Dk_Mapper_StringColumnMap';
                break;
            default:
                $class = $propertyType . 'ColumnMap';
                if (!class_exists($class)) {
                    throw new Dk_ExceptionIllegalArgument("Could not find ColumnMap `$class'.");
                }
                if (!is_subclass_of($class, 'Dk_Mapper_ColumnMap')) {
                    throw new Dk_ExceptionIllegalArgument($class . " is not a Dk_Mapper_ColumnMap.");
                }
        }
        return new $class($column, $property);
    }
    
}
?>