<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * This object is the base column mapper for object properties 
 * so the object loader can serialize and unserialize objects from supplied arrays
 * 
 * @package Db
 */
abstract class Dk_Mapper_ColumnMap extends Dk_Object 
{
    
    const CM_BOOLEAN = 'boolean';
    const CM_INTEGER = 'integer';
    const CM_FLOAT = 'float';
    const CM_STRING = 'string';
    const CM_ENCRYPT_STRING = 'encryptString';
    const CM_ARRAY = 'array';
    
    /**
     * @var string
     */
    private $columnName = '';
    
    /**
     * @var string
     */
    private $propertyName = '';
    
    
    
    /**
     * __construct
     * 
     * @param string $column The table name to map from.
     * @param string $property The object property to map the column to.
     */
    function __construct($columnName, $propertyName) 
    {
        $this->columnName = $columnName;
        $this->propertyName = $propertyName;
    }
    
    /**
     * The property name is the name of the property within the object to map 
     * the data value to.
     * 
     * @return string
     */
    function getPropertyName()
    {
        return $this->propertyName;
    }
    
    
    /**
     * This is the source column name.
     * EG: $row('column1' => 10, 'column2' => 'string', 'column3' => 1.00);
     * The source column names are 'column1', 'column2' and 'column 3'
     * 
     * @return string
     */
    function getColumnName()
    {
        return $this->columnName;
    }
    
    
    
    /**
     * The type of property that gets inserted into the object
     * eg: Dk_Mapper_ColumnMap::STRING or an object 'Dk_Util_Date'
     * 
     * @return string
     * @override
     */
    function getPropertyType()
    {
        return Dk_Mapper_ColumnMap::CM_STRING;
    }
    
    /**
     * Returns the object property value in its requred property type form from the row.
     * (To the object)
     * Override this to return the data in its required object form.
     * 
     * @param array $row
     * @return mixed 
     * @override
     */
    function getPropertyValue($row)
    {
        if (!array_key_exists($this->getColumnName(), $row)) {
            //vd("Failed to find value for property `{$this->getColumnName()}`");
            return;
        }
        return $row[$this->getColumnName()];
    }
    
    /**
     * Convert the object value to its native php type value 
     * (From the object)
     * 
     * @param mixed $value
     * @return mixed A php native type (string, integer, float, etc)
     * @override
     */
    function getColumnValue($row)
    {
        if (!array_key_exists($this->getPropertyName(), $row)) {
            //vd($this->getPropertyName(), $row);
            return null;
        }
        $value = $row[$this->getPropertyName()];
        if (is_array($value)) {
            $value = $row[$this->getPropertyName()][$this->getSerialName()];
        }
        return $value;
    }
    
    
    
    
    /**
     * This is a base php serialize type. (Eg: 'integer', 'string' 
     * 
     * @return string
     * @see Dk_Mapper_ColumnMap constant varibles
     * @override
     */
    function getSerialType() 
    {
        return $this->getPropertyType();
    }
    
    /**
     * Get the property name of where the value is stored in the serialized object
     * 
     * @return string
     * @override
     */
    function getSerialName() 
    {
        return $this->getPropertyName();
    }
    
    /**
     * Get the native type for the property ready for serialization
     * (From the Db or src data)
     * 
     * @param array $row
     * @return mixed A php native type (string, integer, float, etc)
     * @override
     */
    function getSerialValue($row) 
    {
        if (!array_key_exists($this->getColumnName(), $row)) {
            return;
        }
        $value = $row[$this->getColumnName()];
        if (is_array($value)) {
            $value = $row[$this->getColumnName()][$this->getSerialName()];
        }
        return $value;
        //return $this->getColumnValue($row);
    }
    
}

/**
 * A column map fo the string type
 * 
 * @package Db
 */
class Dk_Mapper_StringColumnMap extends Dk_Mapper_ColumnMap 
{
    
}

/**
 * A column map for th eInteger standard type
 * 
 * @package Db
 */
class Dk_Mapper_IntegerColumnMap extends Dk_Mapper_ColumnMap 
{
    /**
     * The type of property that gets inserted into the object
     * eg: Dk_Mapper_ColumnMap::STRING or an object 'Dk_Util_Date'
     * 
     * @return string
     * @override
     */
    function getPropertyType()
    {
        return Dk_Mapper_ColumnMap::CM_INTEGER;
    }
    
    /**
     * Returns the object property value in its requred property type form from the row.
     * Override this to return the data in its required object form.
     * 
     * @param array $row
     * @return mixed 
     * @override
     */
    function getPropertyValue($row)
    {
        $value = parent::getPropertyValue($row);
        return intval($value);
    }
    
    /**
     * Convert the object value to its native php type value 
     * 
     * @param array $row
     * @return integer A php native type (string, integer, float, etc)
     * @override
     */
    function getColumnValue($row)
    {
        return intval(parent::getColumnValue($row));
    }
}
/**
 * A column map fo the Float standard data type
 * 
 * @package Db
 */
class Dk_Mapper_FloatColumnMap extends Dk_Mapper_ColumnMap 
{
    
    /**
     * The type of property that gets inserted into the object
     * eg: Dk_Mapper_ColumnMap::STRING or an object 'Dk_Util_Date'
     * 
     * @return string
     * @override
     */
    function getPropertyType()
    {
        return Dk_Mapper_ColumnMap::CM_FLOAT;
    }
    
    /**
     * Returns the object property value in its requred property type form from the row.
     * Override this to return the data in its required object form.
     * 
     * @param array $row
     * @return mixed 
     * @override
     */
    function getPropertyValue($row)
    {
        $value = parent::getPropertyValue($row);
        return floatval($value);
    }
    
    /**
     * Convert the object value to its native php type value 
     * 
     * @param array $row
     * @return float A php native type (string, integer, float, etc)
     * @override
     */
    function getColumnValue($row)
    {
        return floatval(parent::getColumnValue($row));
    }
}



/**
 * A column map fo the string type
 * 
 * @package Db
 * @todo Fix how the value gets encrypted for the serialise function.
 */
class Dk_Mapper_EncryptStringColumnMap extends Dk_Mapper_ColumnMap 
{
    
    /**
     * The type of property that gets inserted into the object
     * eg: Dk_Mapper_ColumnMap::STRING or an object 'Dk_Util_Date'
     * 
     * @return string
     * @override
     */
    function getPropertyType()
    {
        return Dk_Mapper_ColumnMap::CM_ENCRYPT_STRING;
    }
    
    /**
     * Returns the object property value in its requred property type form from the row.
     * Override this to return the data in its required object form.
     * 
     * @param array $row
     * @return mixed 
     * @override
     */
    function getPropertyValue($row)
    {
        $value = parent::getPropertyValue($row);
        $value = Dk_Util_Encrypt::decrypt($value);
        return $value;
    }
    

    /**
     * Convert the object value to its native php type value 
     * 
     * @param array $row
     * @return mixed A php native type (string, integer, float, etc)
     * @override
     */
    function getColumnValue($row)
    {
        return Dk_Util_Encrypt::encrypt(parent::getColumnValue($row));
    }
}

/**
 * 
 * @package Db
 */
class Dk_Mapper_BooleanColumnMap extends Dk_Mapper_ColumnMap 
{
    
    /**
     * The type of property that gets inserted into the object
     * eg: Dk_Mapper_ColumnMap::STRING or an object 'Dk_Util_Date'
     * 
     * @return string
     * @override
     */
    function getPropertyType()
    {
        return Dk_Mapper_ColumnMap::CM_BOOLEAN;
    }
    
    /**
     * Returns the object property value in its requred property type form from the row.
     * Override this to return the data in its required object form.
     * 
     * @param array $row
     * @return mixed 
     * @override
     */
    function getPropertyValue($row)
    {
        $value = parent::getPropertyValue($row);
        return intval($value) == 1 ? true : false;
    }
    
    /**
     * Convert the object value to its native php type value 
     * 
     * @param array $row
     * @return mixed A php native type (string, integer, float, etc)
     * @override
     */
    function getColumnValue($row)
    {
        return (parent::getColumnValue($row) == true) ? 1 : 0;
    }
    
}
?>