<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * An email gateway object.
 * 
 * @package Mail
 */
class Dk_Mail_Gateway 
{
    
    /**
     * @var Dk_Mail_Gateway
     */
    static $instance = null;
    
    /**
     * 
     */
    private function __construct() { }
    
    /**
     * Get an instance of the email gateway
     * 
     * @return Dk_Mail_Gateway
     */
    static function getInstance() 
    {
        if (self::$instance == null) {
            self::$instance = new Dk_Mail_Gateway();
        }
        return self::$instance;
    }
    
    /**
     * Send an email message
     * 
     * @param Dk_Mail_Message $message
     */
    static function send(Dk_Mail_Message $message)
    {
        return self::getInstance()->sendMessage($message);
    }
    
    /**
     * Send a mime email message
     *
     * @param Dk_Mail_Message $message
     * @throws Dk_ExceptionRuntime
     */
    private function sendMessage(Dk_Mail_Message $message) 
    {
        $config = Dk_Config::getInstance();
        if (($config->isDebugMode() && $config->getDebugEmail() == null)) {
            error_log('No debug email found in config.');
            return false;
        }
        if ($message->getAddressList() == null || count($message->getAddressList()) == 0) {
            throw new Dk_ExceptionNullPointer("No valid address objects found in message.");
        }
        include_once('Dk/Other/htmlMimeMail5/htmlMimeMail5.php');
        
        $mail = new htmlMimeMail5();
        
        
        self::validateField($message->getSubject());
        $mail->setSubject($message->getSubject());
        
        if ($message->isHtml()) {
            $regs = array();
            $mail->setHTML($message->getBody(), $message->getImagesDir());
            eregi('<body>(.+)</body>', $message->getBody(), $regs);
            if ($regs[1] != null) {
                $mail->setText(strip_tags($regs[1]));
            } else {
                $mail->setText(strip_tags($message->getBody()));
            }
        
            $images = $message->getInlineImages();
            foreach ($images as $file => $mime) {
                $mail->addEmbeddedImage(new fileEmbeddedImage($file, $mime, new Base64Encoding()));
            }
        } else {
            $mail->setText($message->getBody());
        }
        
        $attachments = $message->getAttachments();
        foreach ($attachments as $file => $mime) {
            $mail->addAttachment(new fileAttachment($file, $mime, new Base64Encoding()));
        }
        
        $attachments = $message->getAttachmentObjs();
        foreach ($attachments as $obj) {
            $mail->addAttachment($obj);
        }
        
        if(!defined('DK_CLI')) { // if not a cli script
            $mail->setHeader('X-Sender-IP', getenv('REMOTE_ADDR'));
            $mail->setHeader('X-Referer', getenv('HTTP_REFERER'));
            self::checkReferer(array($_SERVER['HTTP_HOST']));
        }
        
        if (Dk_Config::getInstance()->isDebugMode()) {
            $mail->setSubject('Debug: '.$message->getSubject());
            return $mail->send(array(Dk_Config::getInstance()->getDebugEmail()));
        } else {
            $success = true;
            foreach ($message->getAddressList() as $address) {
                $to = explode(',', $address->getTo());
                
                if ($address->getFrom() != null) {
                    $mail->setFrom($address->getFrom());
                }
                if ($address->getBcc() != null) {
                    $mail->setBcc($address->getBcc());
                }
                if ($address->getCc() != null) {
                    $mail->setCc($address->getCc());
                }
                $b = $mail->send($to);
                $success = ($b && $success);
            }
            return $success;
        }
    }
    
    
    /**
     * check_referer() breaks up the enviromental variable
     * HTTP_REFERER by "/" and then checks to see if the second
     * member of the array (from the explode) matches any of the
     * domains listed in the $referers array (declaired at top)
     * 
     * @param array $referers
     */
    private function checkReferer($referers)
    {
        if (count($referers) > 0) {
            if (getenv('HTTP_REFERER')) {
                $temp = explode('/', getenv('HTTP_REFERER'));
                $found = false;
                while (list(,$stored_referer) = each($referers)) {
                    if (eregi('^' . $stored_referer . '$', $temp[2]))
                        $found = true;
                }
                if (!$found) {
                    throw new Dk_ExceptionRuntime("You are coming from an unauthorized domain. Illegal Referer.");
                }
            } else {
                throw new Dk_ExceptionRuntime("Sorry, but I cannot figure out who sent you here. Your browser is not sending an HTTP_REFERER.  This could be caused by a firewall or browser that removes the HTTP_REFERER from each HTTP request you submit.");
            }
        } else {
            throw new Dk_ExceptionRuntime("There are no referers defined. All submissions will be denied.");
        }
    }
    
    
    /**
     * See if a string contains any supicious coding.
     *
     * @param string $str
     * @return string
     */
    static function validateField($str) 
    {
        $badStrings = array(
           "content-type:",
           "mime-version:",
           "multipart/mixed",
           "content-transfer-encoding:",
           "bcc:",
           "cc:",
           "to:"
        );
        
        foreach($badStrings as $badString) {
            if(eregi($badString, strtolower($str))) {
                throw new Dk_ExceptionRuntime("'$badString' found. Suspected injection attempt - mail not being sent.");
            }
        }
        
        if(preg_match("/(%0A|%0D|\\n+|\\r+)/i", $str) != 0) {
            throw new Dk_ExceptionRuntime("newline found in '$str'. Suspected injection attempt - mail not being sent.");
        }
        return $str;
    } 
}
?>