<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 * @package functions
 */



/**
 * Surround a string by quotation marks. Single quote by default
 * 
 * @param string $str
 * @param char $quotes
 * @return string
 */
function enquote($str, $quote = "'")
{
    return $quote . $str . $quote;
}

/**
 * Return the string with the first character lowercased
 *
 * @param string $str
 * @return string
 * @package functions
 */
function lcfirst($str)
{
    return strtolower($str[0]) . substr($str, 1); 
}

/**
 * Count the number of bytes of a given string.
 * Input string is expected to be ASCII or UTF-8 encoded.
 * Warning: the function doesn't return the number of chars
 * in the string, but the number of bytes.
 * See http://www.cl.cam.ac.uk/~mgk25/unicode.html#utf-8
 * for information on UTF-8.
 *
 * @param string $str The string to compute number of bytes
 * @return integer The length in bytes of the given string.
 * @package functions
 */
function str2Bytes($str)
{
    // STRINGS ARE EXPECTED TO BE IN ASCII OR UTF-8 FORMAT
    // Number of characters in string
    $strlen_var = strlen($str);
    
    // string bytes counter
    $d = 0;
    
    /*
     * Iterate over every character in the string,
     * escaping with a slash or encoding to UTF-8 where necessary
     */
    for($c = 0; $c < $strlen_var; ++$c) {
        $ord_var_c = ord($str{$c});
        switch (true) {
            case (($ord_var_c >= 0x20) && ($ord_var_c <= 0x7F)) :
                // characters U-00000000 - U-0000007F (same as ASCII)
                $d++;
                break;
            case (($ord_var_c & 0xE0) == 0xC0) :
                // characters U-00000080 - U-000007FF, mask 110XXXXX
                $d += 2;
                break;
            case (($ord_var_c & 0xF0) == 0xE0) :
                // characters U-00000800 - U-0000FFFF, mask 1110XXXX
                $d += 3;
                break;
            case (($ord_var_c & 0xF8) == 0xF0) :
                // characters U-00010000 - U-001FFFFF, mask 11110XXX
                $d += 4;
                break;
            case (($ord_var_c & 0xFC) == 0xF8) :
                // characters U-00200000 - U-03FFFFFF, mask 111110XX
                $d += 5;
                break;
            case (($ord_var_c & 0xFE) == 0xFC) :
                // characters U-04000000 - U-7FFFFFFF, mask 1111110X
                $d += 6;
                break;
            default :
                $d++;
        }
        ;
    }
    ;
    return $d;
}

/*
 * PHP Override to get the MIME type of a file
 *   if the function mime_content_type does not exsit
 */
if (!function_exists('mime_content_type') && PHP_OS == 'Linux') {
    /**
     * Get the mime type from a file using a linux command
     *
     * @param string $f The file to test
     * @return string
     * @package functions
     */
    function mime_content_type($f)
    {
        return trim(exec('file -bi ' . escapeshellarg($f)));
    }
}

/*
 * dissable magic_quotes_gpc if enabled
 */
if (get_magic_quotes_gpc()) {
    /**
     * Dissable magic quotes if enabled on the server
     */
    function magicQuotesGpc()
    {
        function traverse(&$arr)
        {
            if (!is_array($arr)) {
                return;
            }
            foreach ($arr as $key => $val) {
                is_array($arr[$key]) ? traverse($arr[$key]) : ($arr[$key] = stripslashes($arr[$key]));
            }
        }
        $gpc = array(&$_COOKIE, &$_REQUEST, &$_GET, &$_POST);
        traverse($gpc);
    }
    magicQuotesGpc();
}

/*
 * Fix IE submit key name
 * When a form contains an image submit. IE uses 'submit_x' and 'submit_y'
 * as the $_REQUEST key names. Here we add the value 'submit' to the request to fix this 
 * issue.
 */
//if (isset($_SERVER['HTTP_USER_AGENT']) && strpos($_SERVER['HTTP_USER_AGENT'], 'MSIE 6.0')) {
if (isset($_SERVER['HTTP_USER_AGENT']) && strpos($_SERVER['HTTP_USER_AGENT'], 'MSIE')) {
    foreach ($_REQUEST as $key => $value) {
        if (substr($key, -2) == '_x' && !array_key_exists(substr($key, 0, -2), $_REQUEST)) {
            $newKey = substr($key, 0, -2);
            $_REQUEST[$newKey] = $value;
        }
    }
}

?>