<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * A string form field object.
 * 
 * @package Form
 */
class Dk_Form_FileField extends Dk_Form_Field 
{
  
    /**
     * @var array
     */
    private $fileInfo = array();
    
    /**
     * the max filesize in bytes
     * @var integer
     */
    protected $maxFilesize = 2000000;
    
    
    /**
     * Use fore custom filenames instead of the one supplied
     * @var string
     */
    protected $newFilename = '';
    
    /**
     * If set to false the file is not uploaded
     * @var boolean
     */
    private $moveUploded = true;
    
    /**
     * The file path for the uploaded file
     * @var Dk_Util_Path
     */
    private $path = null;
    
    
    
       
    /**
     * __construct
     * 
     * @param string $name
     * @param string Dk_Util_Path $path The upload destination path
     */
    function __construct($name, Dk_Util_Path $path) 
    {
        parent::__construct($name);
        if (!$path instanceof Dk_Util_Path) {
            throw new Dk_ExceptionIllegalArgument("Field `{$this->getName()}` declared as type `file`. Dk_Util_Path object required");
        }
        $this->path = $path;
        $this->maxFilesize = Dk_Util_Path::string2Bytes(ini_get('upload_max_filesize'));
    }
    
    
    
    /**
     * Loads the object and the form fields from the request.
     * 
     * @param array $array
     */
    function loadFromArray($array)
    {
        $name = $this->getName();
        if (!isset($_FILES[$name])) {
            return false;
        }
        $this->fileInfo = $_FILES[$name];
        $this->domValues[$name] = $this->getFilename();
        $this->setValueFromRequest($this->getFilename());
        
        if ($this->getFileError() > 0 && $this->getFileError() != UPLOAD_ERR_NO_FILE) {
            $msg = self::getErrorString($this->getFileError());
            $this->addError($msg);
            return false;
        }
        return true;
    }
    
    /**
     * Updates the fields Dom string values.
     *
     * Converts value to HTML form element (name, value) pairs and adds them
     * to the $domValues map. Called by setValue().
     *
     * @see setValue()
     * @param mixed $value
     */
    protected function setDomValues($value)
    {
        $this->domValues[$this->getName()] = $value;
    }
    
    
    
    
    
    /**
     * Get the max file size in bytes from the php.ini file.
     *
     * @return integer
     */
    function getMaxFileSize()
    {
        return $this->maxFileSize;
    }
    
    /**
     * Get the uploaded file mime type.
     *
     * @return string
     */
    function getMimeType() 
    {
        return $this->fileInfo['type'];
    }
    
    /**
     * Get eny errors generated by uploading.
     *
     * @return integer
     */
    function getFileError() 
    {
        return $this->fileInfo['error'];
    }
    
    /**
     * Get the uploded file's size in bytes.
     *
     * @return integer
     */
    function getFileSize() 
    {
        return $this->fileInfo['size'];
    }
    
    /**
     * Get the client side filename of the uploaded file.
     * 
     * @param boolean $fromFiles If set to true, forces the name to come from the $_FILES global
     * @return string
     */
    function getFilename($fromFiles = false) 
    {
        if ($this->newFilename == null || $fromFiles) {
            return self::cleanFilename($this->fileInfo['name']);
        } else {
            return $this->newFilename . '.' . strtolower(Dk_Util_Path::getFileExtension($this->fileInfo['name']));
        }
    }
    
    /**
     * Set the new custom filename set to '' to use uploaded filename
     * This is the filename without an extension
     * 
     * @param string $file
     */
    function setFilename($file)
    {
        if (substr($file, 0, 1) == '/') {
            $file = substr($file, 1);
        }
        $this->newFilename = self::cleanFilename($file);
    }
    
    /**
     * Get the file extension
     *
     * @return str
     */
    function getExtension()
    {
        $ext = explode(".", $this->getFilename());
        return strtolower(array_pop($ext));
    }
    
    /**
     * Get the stored temp file path.
     *
     * @return string
     */
    function getTmpName() 
    {
        return $this->fileInfo['tmp_name'];
    }
    
    /**
     * Test to see if a file was submitted and is in the temp folder
     * 
     * @return boolean
     */
    function hasFile()
    {
        return $this->getFileError() == UPLOAD_ERR_OK;
    }
    
    /**
     * Clean a filename.
     * If a filenames beyond the 255 limit it is concatentated.
     * 
     * @param string $file
     * @return string
     */
    static function cleanFilename($file)
    {
        if (strlen($file) > 255) {
            $arr = explode('.', $file);
            $ext = strtolower(array_pop($arr));
            $f1 = substr($file, 0, (254-strlen($ext)));
            $file = $f1 . $ext;
        } else {
            $pos = strrpos($file, '.');
            if ($pos != null) {
                $ext = strtolower(substr($file, $pos + 1));
                $file = substr($file, 0, $pos + 1) . $ext;
            }
        }
        return preg_replace('/[^a-z0-9\-\.\/]/i', '_', $file);
    }
    
    /**
     * Get the full uplod directory path
     * 
     * @return Dk_Util_Path
     */
    function getPath()
    {
        return $this->path;
    }
    
    /**
     * Upload file and return true on success
     * 
     * @return boolean
     */
    function moveUploadedFile() 
    {
        if (!$this->hasFile() || !$this->moveUploded) {
            return false;
        }
        
        $filePath = new Dk_Util_Path($this->path->getPath() . '/' . $this->getFilename());
        if (!is_dir($this->path->getPath())) {
            if (!mkdir($this->path->getPath(), 0777, true)) {
                error_log("Permission Error: Cannot create directory `{$this->path->getPath()}`");
                return false;
            }
        } else {
            if (!is_writable($this->path->getPath())) {
                error_log('Upload directory is invalid or not writable: ' . $this->path->getPath());
                return false;
            }
        }
        
        if (!move_uploaded_file($this->getTmpName(), $filePath->getPath())) {
            if (is_file($this->getTmpName())) {
                @unlink($this->getTmpName());
            }
            return false;
        }
        @chmod($filePath->getPath(), 0644);
        
        return true;
    }
    
    /**
     * Set this to false to ensure the uploded file is not moved out of the tmp folder.
     *
     * @param boolean $b
     */
    function setMoveUploaded($b)
    {
        $this->moveUploded = $b;
    }
    
    /**
     * getErrorString
     *
     * @param integer $errorId
     */
    static function getErrorString($errorId)
    {
        switch ($errorId) {
            case UPLOAD_ERR_OK:
                $str = '';
                break;
            case UPLOAD_ERR_INI_SIZE:
                $str = "The uploaded file exceeds max filesize of " . ini_get("upload_max_filesize");
                break;
            case UPLOAD_ERR_FORM_SIZE:
                $str = "The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form.";
                break;
            case UPLOAD_ERR_PARTIAL:
                $str = "The uploaded file was only partially uploaded.";
                break;
            case UPLOAD_ERR_NO_FILE:
                $str = "No file was uploaded.";
                break;
            case UPLOAD_ERR_NO_TMP_DIR:
                $str = "Missing a temporary folder.";
                break;
            case UPLOAD_ERR_CANT_WRITE:
                $str = "Failed to write file to disk";
                break;
            default:
                $str = "Unknown File Error";
        }
        return $str;
    }
    
}
?>
