<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * A basic example of an object factory.
 * 
 * <code>
 * <?php
 *    Dk_Db_ObjectFactory::getInstance()->getDb();
 * ?>
 * </code>
 *
 * @package Dk
 */
class Dk_Db_ObjectFactory 
{
    
    /**
     * @var Dk_Db_ObjectFactory
     */
    static protected $instance = null;
    
    
    /** 
     * @var Dk_Db_MyDao
     */
    protected $db = null;
    
    /**
     * @var array
     */
    protected $mappers = array();
    
    
    /**
     * This is a constructor
     * If no request session or response parameters given the default Dk objects are used.
     * 
     */
    protected function __construct() { }
    
    /**
     * Get an instance of the object factory
     * 
     * @return Dk_Db_ObjectFactory
     */
    static function getInstance() 
    {
        if (self::$instance == null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    
    /**
     * Get a database object
     *
     * @return Dk_Db_MyDao
     */
    static function getDb() 
    {
        return self::getInstance()->makeDb();
    }
    
    /**
     * Make a db object
     *
     * @return Dk_Db_MyDao
     */
    private function makeDb() 
    {
    	$config = Dk_Config::getInstance();
        if ($this->db == null) {
            $this->db = new Dk_Db_MyDao(
                 $config->getDbUser(),
                 $config->getDbPassword(),
                 $config->getDbDatabase(),
                 $config->getDbHost() );
        }
        $this->db->selectDb($config->getDbDatabase());
        return $this->db;
    }
    
    /**
     * Looks for a mapper for object of type $class
     * Class can be an array of objects, an object or a string and the 
     * mapper will try to get the most relevent classname from the supplied parameter
     * 
     * @param mixed $class Object, array, Dk_Db_Array, string
     * @return Dk_Db_Mapper
     */
    static function getDbMapper($class) 
    {
        return self::getInstance()->makeDbMapper($class);
    }
    
    /**
     * Looks for a mapper class $class . 'Mapper' or you could use the full mapper classname 'Ext_Util_ObjectMapper'
     * 
     * @param mixed $class Object, array, Dk_Db_Array, string
     * @return Dk_Db_Mapper
     */
    function makeDbMapper($class)
    {
        if (is_array($class) && count($class) > 0) {
            $obj = current($class);
            $class = get_class($obj);
        } else if ($class instanceof Dk_Db_Array) {
            $obj = $class->getFirst();
            $class = get_class($obj);
        } else if (is_object($class)) {
            $class = get_class($class);
        } 
        
        if (substr($class, -6) == 'Mapper') {
            $class = substr($class, 0, -6);
        }
        if (!array_key_exists($class, $this->mappers)) {
            if (in_array('Dk_Db_OrderInterface', class_implements($class))) { // Order Mapper
                $this->mappers[$class] = new Dk_Db_OrderMapper(Dk_Mapper_ObjectFactory::getMapper($class));
            } else {  // Basic Mapper
                $this->mappers[$class] = new Dk_Db_Mapper(Dk_Mapper_ObjectFactory::getMapper($class));
            }
        }
        return $this->mappers[$class];
    }
    
}
?>