<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */



/**
 * The base mapper object that controls the mapping of columns to objects
 *
 * @package Db
 */
class Dk_Db_Mapper extends Dk_Object 
{
    
    /**
     * @var array
     */
    static private $mappers = array();
    
    
    
    /**
     * @var Dk_Db_DaoInterface
     */
    protected $db = null;
    
    /**
     * @var Dk_Mapper_Object
     */
    protected $mapper = null;
    
    
    
    /**
     * __construct
     * 
     * @param Dk_Mapper_Object $mapper
     */
    function __construct(Dk_Mapper_Object $mapper)
    {
        $this->db = Dk_Db_ObjectFactory::getDb();
        $this->mapper = $mapper;
    }
    
    /**
     * Looks for a mapper class $class . 'Mapper' or you could use the full mapper classname 'Ext_Util_ObjectMapper'
     * 
     * @param string
     * @return Dk_Db_Mapper
     * @deprecated Will be removed in Ver 2.0 use Dk_Db_ObjectFactory
     */
    static function makeMapper($class = '')
    {
        return Dk_Db_ObjectFactory::getInstance()->makeDbMapper($class);
    }
    
    
    
    /**
     * Get the Database Access Object
     *
     * @return Dk_Db_MyDao
     */
    function getDb()
    {
        return $this->db;
    }
    
    /**
     * Get the object mapper
     *
     * @return Dk_Mapper_Object
     */
    function getMapper()
    {
        return $this->mapper;
    }
    
    /**
     * The class name this mapper is used for.
     *
     * @return string
     */
    function getClass()
    {
        return $this->getMapper()->getClass();
    }

    /**
     * Get the table from this object's mapper.
     *
     * @return string
     */
    function getTable()
    {
        $table = $this->getMapper()->getParameter('tableName');
        if ($table == null) {
            $arr = explode('_', $this->getClass());
            $table = $arr[count($arr)-1];
            $table = strtolower($table[0]) . substr($table, 1);
        }
        return $table;
    }

    
    /**
     * Return a select list of fields for a sql query
     * 
     * @param string $prepend (optional) Default is a null string
     * @return string
     */
    function getSelectList($prepend = '')
    {
        $result = '';
        if ($prepend != null && substr($prepend, -1) != '.') {
            $prepend = $prepend.".";
        }
        foreach ($this->getMapper()->getDataMap()->getIdColumns() as $map) {
            $result .= $prepend . "`" . $map->getColumnName(). '`,';
        }
        foreach ($this->getMapper()->getDataMap()->getColumns() as $map) {
            $result .= $prepend . "`" . $map->getColumnName(). '`,';
        }
        return substr($result, 0, -1);
    }
    
    /**
     * Return an update list of fields for a sql query
     * 
     * @param mixed $obj
     * @return string
     */
    function getUpdateList($obj)
    {
        $row = Dk_Mapper_ObjectLoader::getInstance()->getObjectValues($obj);
        $result = '';
        foreach ($this->getMapper()->getDataMap()->getColumns() as $map) {
            if ($map->getPropertyName() == 'modified') {
                $now = Dk_Util_Date::createDate();
                $result .= '`' . $map->getColumnName() . "` = '" . $now->getIsoDate(true) . "',";
            } else {
                $result .= '`' . $map->getColumnName() . "` = " . $this->getSqlColumnValue($map, $row) . ",";
            }
        }
        $str = substr($result, 0, -1);
        return $str;
    }
    
    /**
     * Get the insert text for a query
     * 
     * @param mixed $obj
     * @return string
     */
    function getInsertList($obj)
    {
        $columns = '';
        $values = '';
        $row = Dk_Mapper_ObjectLoader::getInstance()->getObjectValues($obj);
        
        foreach ($this->getMapper()->getDataMap()->getIdColumns() as $map) {
            $columns .= "`" . $map->getColumnName() . '`,';
            $values .= $this->getSqlColumnValue($map, $row) . ",";
        }
        foreach ($this->getMapper()->getDataMap()->getColumns() as $map) {
            $columns .= "`" . $map->getColumnName() . '`,';
            $values .= $this->getSqlColumnValue($map, $row) . ",";
        }
        return '(' . substr($columns, 0, -1) . ') VALUES(' . substr($values, 0, -1) . ')';
    }
    
    /**
     * Get the string representation of the data
     *
     * @param Dk_Mapper_ColumnMap $map
     * @param array $row
     */
    private function getSqlColumnValue($map, $row)
    {
        $value = $map->getColumnValue($row);
        if ($value === null) {
            $value = "NULL";
        } else if (is_string($value)) {
            $value = Dk_Db_MyDao::escapeString($value);
            $value = "'$value'";
        }
        return $value;
    }
    
    
    
    
    /**
     * Create a DB Array object filled
     *
     * @param Dk_Db_ResultInterface $result
     * @param string $orderBy
     * @param integer $limit
     * @param integer $offset
     * @return Dk_Db_Array
     */
    function loadDbArray(Dk_Db_ResultInterface $result, $orderBy = '', $limit = 0, $offset = 0)
    {
        $objs = new Dk_Db_Array();
        foreach($result as $row) {
            $objs[] = $this->getMapper()->loadArray($row);
        }
        $objs->setLoo(new Dk_Util_Loo($limit, $offset, $orderBy));
        $objs->setTotal($result->getTotal());
        $result->free();
        return $objs;
    }
    
    
    /**
     * Select a number of elements from a database
     *
     * @param string $where EG: "`column1`=4 AND `column2`=string"
     * @param string $orderBy EG: "`column1`, `column2` DESC"
     * @param integer $limit
     * @param integer $offset
     * @param integer $prepend Used for table aliases in a query
     * @return Dk_Db_Array
     */
    function selectFrom($from = '', $where = '', $orderBy = '', $limit = 0, $offset = 0, 
        $prepend = '', $isDistinct = false)
    {
        $limit = intval($limit);
        $offset = intval($offset);
        $prepend = Dk_Db_MyDao::escapeString($prepend);
        $isDistinct = $isDistinct === true ? true : false;
        
        if ($from == null) {
            $from = sprintf("`%s`", $this->getTable());
        }
        if ($where == null) {
            $where = "1";
        }
        
        if ($orderBy != '') {
            if (strstr(strtolower($orderBy), 'field') === false) {
                $orderBy = Dk_Db_MyDao::escapeString($orderBy);
            }
            $orderBy = "ORDER BY " . $orderBy;
        }
        $limitStr = '';
        if ($limit > 0) {
            $limitStr = sprintf("LIMIT %d, %d", $offset, $limit);
        }
        $distinct = '';
        if ($isDistinct) {
          $distinct = 'DISTINCT';
        }
        
        $query = sprintf("SELECT %s %s FROM %s WHERE %s %s %s",
              $distinct, $this->getSelectList($prepend), $from, $where, $orderBy, $limitStr);
        //vd($query);
        $result = $this->db->query($query);
        return $this->loadDbArray($result, $orderBy, $limit, $offset);
    }

    /**
     * Select a number of elements from a database
     *
     * @param string $where EG: "`column1`=4 AND `column2`=string"
     * @param string $orderBy EG: "`column1`, `column2` DESC"
     * @param integer $limit
     * @param integer $offset
     * @return Dk_Db_Array
     */
    function selectMany($where = '', $orderBy = '', $limit = 0, $offset = 0)
    {
        return $this->selectFrom('', $where, $orderBy, $limit, $offset);
    }
    
    /**
     * Create a select query using the Dk_Util_Loo object
     *
     * @param string $where
     * @param Dk_Util_Loo $loo
     * @return Dk_Db_Array
     */
    function looSelect($where, $loo = null)
    {
        if (!$loo instanceof Dk_Util_Loo) {
            $loo = new Dk_Util_Loo(0, 0, 'id');
        }
        return $this->selectFrom('', $where, $loo->getOrderBy(), $loo->getLimit(), $loo->getOffset());
    }
    

    /**
     * Select a record from a database
     *
     * @param integer $id
     * @return mixed Returns null on error
     */
    function select($id)
    {
        $idFields = $this->getMapper()->getDataMap()->getIdColumns();
        $idField = current($idFields);
        if ($idField == null) {
            throw new Dk_ExceptionNullPointer('No Column Id fields defined. Check your Object Mapper.');
        }
        $query = sprintf('SELECT %s FROM `%s` WHERE `%s` = %d LIMIT 1',
                $this->getSelectList(), $this->getTable(),
                $idField->getColumnName(), intval($id));
        $result = $this->db->query($query);
        if($result->count() > 0) {
            $obj = Dk_Mapper_ObjectLoader::getInstance()->doLoad($result->current(), $this->getMapper());
            $result->free();
            return $obj;
        }
    }

    /**
     * Insert this object into the database.
     * Returns the new insert id for this object.
     *
     * @param Dk_Object $obj
     * @return integer
     */
    function insert($obj)
    {
        $query = sprintf('INSERT INTO `%s` %s', $this->getTable(), $this->getInsertList($obj));
        $this->db->query($query);
        $id = $this->db->getInsertID();
        $obj->id = $id;
        return $id;
    }

    /**
     * Update this object in the database.
     * Returns The number of affected rows.
     *
     * @param Dk_Object $obj
     * @return integer The number of affected rows
     */
    function update($obj)
    {
        $idFields = $this->getMapper()->getDataMap()->getIdColumns();
        $idField = current($idFields);
        $query = sprintf('UPDATE `%s` SET %s WHERE `%s` = %d',
                 $this->getTable(), $this->getUpdateList($obj),
                 $idField->getColumnName(), $obj->getId());
        $this->db->query($query);
        return $this->db->getAffectedRows();
    }

    /**
     * Update a single value in a single row
     * 
     * @param integer $id
     * @param string $column
     * @param mixed $value
     * @return integer Return the number of rows affected
     */
    function updateValue($id, $column, $value)
    {
        $idFields = $this->getMapper()->getDataMap()->getIdColumns();
        $idField = current($idFields);
        $query = sprintf("UPDATE `%s` SET `%s` = '%s' WHERE `%s` = %d",
                 $this->getTable(), $column, $value,
                 $idField->getColumnName(), $id);

        $this->db->query($query);
        return $this->db->getAffectedRows();
    }

    /**
     * Delete this object from the database.
     * Returns The number of affected rows.
     * 
     * @param Dk_Object $obj
     * @return integer
     */
    function delete($obj)
    {
        $objectValues = Dk_Mapper_ObjectLoader::getInstance()->getObjectValues($obj);
        
        $where = '';
        /* @var $map Dk_Db_ColumnMap */
        foreach($this->getMapper()->getDataMap()->getIdColumns() as $map) {
            $where .= '`' . $map->getColumnName() . '` = ' . 
                $objectValues[$map->getPropertyName()] . ',';
        }
        $where = substr($where, 0, -1);
        $query = sprintf('DELETE FROM `%s` WHERE %s LIMIT 1', $this->getTable(), $where);
        $this->db->query($query);
        return $this->db->getAffectedRows();
    }

    /**
     * Delete an array of Ids from the database
     * 
     * 
     * @param array $ids
     * @return integer The number of affected rows.
     */
    function deleteGroup($ids)
    {
        $where = '';
        /* @var $map Dk_Db_ColumnMap */
        foreach($ids as $id) {
            $where .= '`id` = ' . intval($id) . ' OR ';
        }
        $where = substr($where, 0, -3);
        $query = sprintf('DELETE FROM `%s` WHERE %s LIMIT 1', $this->getTable(), $where);
        $this->db->query($query);
        return $this->db->getAffectedRows();
    }
    
    
    /**
     * Returns the object id if it is greater than 0 or the nextInsertId if is 0
     * 
     * @return integer
     */
    function getVolitileId($obj)
    {
        if ($obj->getId() == 0) {
            $id = $this->getNextInsertId();
        } else {
            $id = $obj->getId();
        }
        return $id;
    }
    
    /** 
     * Get the insert id of the last added record.
     * Taken From: http://dev.mysql.com/doc/refman/5.0/en/innodb-auto-increment-handling.html
     * 
     * @return integer The last id for auto-increment row id.
     */
    function getNextInsertId() 
    {
        $query = sprintf("SELECT MAX(`id`) AS `lastId` FROM `%s` ", $this->getTable());
        $result = mysql_query($query);
        $result = mysql_fetch_array($result);
        $id = $result['lastId']+1;
        return $id;
    }
    

    /**
     * Count records in a DB
     * 
     * @param string $from
     * @param string $where
     * @param integer $distinctId
     * @return integer
     */
    function count($from = '', $where = '')
    {
        if ($from == '') {
            $from = sprintf("`%s`", $this->getTable());
        }
        if ($where == null) {
            $where = "1";
        }
        $query = sprintf("SELECT COUNT(*) AS i FROM %s WHERE %s", $from, $where);
        
        $result = $this->db->query($query);
        $value = $result->current();
        return intval($value['i'], 10);
    }
    
}
?>