<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud <info@tropotek.com>
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * A config/registry object that configures the Sdk functionality.
 *
 *
 * @package Dk
 */
class Dk_Config extends Dk_Util_Registry
{
    /**
     * @var Dk_Config
     */
    static protected $instance = null;



    /**
     * Sigleton, No instances can be created.
     * Use:
     *   Dk_Config::getInstance()
     */
    protected function __construct() { }


    /**
     * Get an instance of this object
     * NOTE: You need to created this function for each inherited class
     *
     * @return Dk_Config
     */
    static function getInstance()
    {
        if (self::$instance == null) {
            self::$instance = new Dk_Config();
        }
        return self::$instance;
    }

    /**
     * Get a section of the config.ini that is defined by [..] brackets
     * This is usually used to setup a Widget system config options
     *
     * @return array
     */
    static function getSectionConfig($section)
    {
        $ini = parse_ini_file(self::getInstance()->getFileRoot() . '/config.ini', true);
        if (!is_array($ini)){
            $ini = array();
        }
        return $ini[$section];
    }





    /**
     * Set the host for the database libs
     *
     * @param string $host
     * @default null
     */
    function setDbHost($host)
    {
        if (!eregi('^[a-z0-9_\-\.]+$', $host)) {
            throw new Dk_ExceptionIllegalArgument('Invalid DBHost value.');
        }
        $this->setEntry('DbHost', $host);
    }

    /**
     * Get the host for the database.
     *
     * @return string
     * @default null
     */
    function getDbHost()
    {
        return $this->getEntry('DbHost');
    }

    /**
     * Set the user for the database.
     *
     * @param string $user
     * @default null
     */
    function setDbUser($user)
    {
        $this->setEntry('DbUser', $user);
    }

    /**
     * Get the database user.
     *
     * @return string
     * @default null
     */
    function getDbUser()
    {
        return $this->getEntry('DbUser');
    }

    /**
     * Set Database password.
     *
     * @param string $password
     * @default null
     */
    function setDbPassword($password)
    {
        $this->setEntry('DbPassword', $password);
    }

    /**
     * Get the database password.
     *
     * @return string
     * @default null
     */
    function getDbPassword()
    {
        return $this->getEntry('DbPassword');
    }

    /**
     * Set the database name
     *
     * @param string $database
     * @default null
     */
    function setDbDatabase($database)
    {
        $this->setEntry('DbDatabase', $database);
    }

    /**
     * Get the database name
     *
     * @return string
     * @default null
     */
    function getDbDatabase()
    {
        return $this->getEntry('DbDatabase');
    }





    /**
     * Set the site's title which will be used for emails and notices
     *
     * @param string $name
     * @default 'New Dk Project'
     */
    function setSiteTitle($name)
    {
        $this->setEntry('SiteTitle', $name);
    }

    /**
     * Get the site's title which will be used for emails and notices
     *
     * @return string
     * @default 'New Dk Project'
     */
    function getSiteTitle()
    {
        return $this->getEntry('SiteTitle');
    }
    

    /**
     * If all fatal errors will email this address.
     * If set to null no error emails are sent
     *
     * @param string $str
     * @default null
     */
    function setErrorEmail($str)
    {
        $this->setEntry('ErrorEmail', $str);
    }

    /**
     * If all fatal errors will email this address.
     * If set to null no error emails are sent
     *
     * @return string
     * @default null
     */
    function getErrorEmail()
    {
        return $this->getEntry('ErrorEmail');
    }

    /**
     * Set this sites htdoc root path.
     * Note: this must be set if no FrontController is used with modrewrite
     * Example: /~user/project/
     *
     * @param string $path
     * @default $_SERVER['PHP_SELF']
     */
    function setHtdocRoot($path)
    {
        $this->setEntry('HtdocRoot', $path);
    }

    /**
     * Get this sites htdoc root path.
     * Note: this must be set if no FrontController is used with modrewrite
     * Example: /~user/project/
     *
     * @return string
     * @default $_SERVER['PHP_SELF']
     */
    function getHtdocRoot()
    {
        return $this->getEntry('HtdocRoot');
    }

    /**
     * Set this sites filesystem root path.
     * Example: /home/user/public_html/project/
     *
     * @param string $path
     * @default dirname() of the site index.php file
     */
    function setFileRoot($path)
    {
        $this->setEntry('FileRoot', $path);
    }

    /**
     * Get this sites filesystem root path.
     * Example: /home/user/public_html/project
     *
     * @return string
     * @default dirname() of the site index.php file
     */
    function getFileRoot()
    {
        return $this->getEntry('FileRoot');
    }

    /**
     * Get this Dk filesystem root path.
     * Example: /home/user/public_html/project/lib/Dk
     * ReadOnly.
     *
     * @return string
     */
    function getDkFileRoot()
    {
        return $this->getEntry('DkFileRoot');
    }

    /**
     * Set this sites html template directory
     *
     * @param string $path
     * @default $fileRoot.'/html'
     */
    function setHtmlTemplates($path)
    {
        $this->setEntry('HtmlTemplates', $path);
    }

    /**
     * Get this sites html template directory
     *
     * @return string
     * @default $fileRoot.'/html'
     */
    function getHtmlTemplates()
    {
        return $this->getEntry('HtmlTemplates');
    }

    /**
     * Set the Data directory for this site.
     * NOTE: This directory must be writable or you will see a warning on the page
     *
     * @param string $path
     * @default $fileRoot.'/data'
     */
    function setDataDir($path)
    {
        $this->setEntry('DataDir', $path);
    }

    /**
     * Get the data directory for this site
     *
     * @return string
     * @default $fileRoot.'/data'
     */
    function getDataDir()
    {
        return $this->getEntry('DataDir');
    }








    /**
     * Set this site to use debug mode.
     * Debug mode should be set to false for live sites.
     *
     * Turns on:
     *  o Replace relative paths to home directory
     *  o Log errors E_ALL|E_STRICT
     *
     * @param boolean $boolean
     * @default flase
     */
    function setDebugMode($boolean)
    {
        if ($boolean) {
            error_reporting(E_ALL|E_STRICT);
            //error_reporting(E_ALL);
            ini_set('display_errors','On');
        } else {
            error_reporting(0);
            ini_set('display_errors','Off');
        }
        $this->setEntry('DebugMode', $boolean);
    }
    
    /**
     * Return true if this site is in debug mode
     *
     * @return boolean
     * @default flase
     */
    function isDebugMode()
    {
        return $this->getEntry('DebugMode');
    }

    /**
     * When in debug mode this email is used
     * When used in cli mode the command `hostname` is used in place of `HTTP_HOST`
     *
     * @param string $email
     * @default info@{HTTP_HOST}
     */
    function setDebugEmail($email)
    {
        $this->setEntry('DebugEmail', $email);
    }

    /**
     * When in debug mode this email is used
     *
     * @return string
     * @default info@{HTTP_HOST}
     */
    function getDebugEmail()
    {
        return $this->getEntry('DebugEmail');
    }

    /**
     * Set the error log file to use.
     * This will only works if Debug mode is enabled first
     * For a live site we want to use the default system log file.
     *
     * @param string $file
     * @default Use system error log
     */
    function setErrorLog($file)
    {
        if ($this->isDebugMode()) {
            $this->setEntry('ErrorLog', $file);
            @ini_set('error_log', $this->getEntry('ErrorLog'));
        }
    }

    /**
     * Get the error log file path.
     *
     * @return string
     * @default Use system error log
     */
    function getErrorLog()
    {
        return $this->getEntry('ErrorLog');
    }


    /**
     * Set this to true if this site has an SSL cert installed
     * NOTE: This will not work for shared SSL or SSL on a different url/path
     *
     * @param boolean $b
     * @default false
     */
    function setSslEnabled($b)
    {
        $this->setEntry('SslEnabled', $b);
    }

    /**
     * Return true if this site has an SSL cert installed
     * NOTE: This will not work for shared SSL or SSL on a different url/path
     *
     * @return boolean
     * @default false
     */
    function isSslEnabled()
    {
        return $this->getEntry('SslEnabled');
    }


    /**
     * Set the apllication tmp dir
     *
     * @param string $dir
     * @default '/tmp'
     */
    function setTmpDir($dir)
    {
        ini_set('upload_tmp_dir', $dir);
        $this->setEntry('TmpDir', $dir);
    }

    /**
     * Get the apllication tmp dir
     *
     * @return string
     * @default '/tmp'
     */
    function getTmpDir()
    {
        return $this->getEntry('TmpDir');
    }



    /**
     * Set the timezone for php.
     * See http://au.php.net/manual/en/timezones.php for a valad list of timezones
     *
     * @param string $str
     * @default Australia/Queensland
     */
    function setTimezone($str)
    {
        ini_set('date.timezone', $str);
        $this->setEntry('Timezone', $str);
    }

    /**
     * Get the current timezone
     * See http://au.php.net/manual/en/timezones.php for a valad list of timezones
     *
     * @return string
     * @default Australia/Queensland
     */
    function getTimezone()
    {
        return $this->getEntry('Timezone');
    }

    /**
     * Set the default currency for the Dk_Util_Money object
     * Valid Values: AUD, NZD, USD, THB
     *
     * @param string $str
     * @default 'AUD'
     */
    function setCurrency($str)
    {
        $eregStr = '';
        foreach (Dk_Util_Currency::$currencyList as $k => $v) {
            $eregStr .= "$k|";
        }
        $eregStr = '^(' . substr($eregStr, 0 , -1) . ')$';
        if (!ereg($eregStr, $str)) {
            throw new Dk_ExceptionIllegalArgument('Valid currency values are: AUD, NZD, USD, THB');
        }
        Dk_Util_Money::$defaultCurrencyCode = $str;
        $this->setEntry('Currency', $str);
    }

    /**
     * Get the default currency for the Dk_Util_Money object
     * Valid Values: AUD, NZD, USD, THB
     *
     * @return string
     * @default 'AUD'
     */
    function getCurrency()
    {
        return $this->getEntry('Currency');
    }

    /**
     * Set Language
     *
     * @param string $str
     * @default en (english)
     */
    function setLanguage($str)
    {
        $this->setEntry('Language', $str);
    }

    /**
     * Get Language
     *
     * @return string
     * @default en (english)
     */
    function getLanguage()
    {
        return $this->getEntry('Language');
    }


    /**
     * Set this is to fix old xmlLib libries that cannot handle the comments in javascript
     * It will remove all '<![CDATA[' and ']]>'
     *
     * @param boolean $b
     * @default false
     */
    function setCdataFix($b)
    {
        $this->setEntry('CdataFix', $b);
    }

    /**
     * Set this is to
     * The code looks like:
     *  <code>
     *  <?php
     *   $pageHtml = str_replace(array('<![CDATA[', ']]>'), array('', ''), $pageHtml);
     *  ?>
     *  </code> fix old xmlLib libries that cannot handle the comments in javascript
     *
     * @return boolean
     * @default false
     */
    function getCdataFix()
    {
        return $this->getEntry('CdataFix');
    }

    /**
     * Set the opensource status of this project.
     * This will stop certin code from executing in opencource projects
     * as this code is not required for opensource projects.
     * 
     * NOTE: The code in the non-opensource control structures where
     * isOpensource() == false may not be GPL and therfore should not be modified 
     * in any way.
     * 
     * Get the opensource status of this project
     *
     * @return string
     * @default true
     */
    function isOpensource()
    {
        return true;
    }
    
    
    
    
    /**
     * Set 
     *
     * @param string $str
     * @default 
     */
    function set($str)
    {
        $this->setEntry('', $str);
    }

    /**
     * Get 
     *
     * @return string
     * @default 
     */
    function get()
    {
        return $this->getEntry('');
    }

}
?>
