<?php

/*
 * codelighter.inc
 * This file is part of CodeLighter
 *
 * Copyright (C) 2003-2004 Kapco Media Creators
 *
 * This program is free software; permission is hereby granted, free
 * of charge, to any person obtaining a copy of this software, to deal
 * with it without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, and/or
 * sell copies of the software, and to permit persons to whom the
 * software is furnished to do so, provided that the above copyright
 * notice(s) and this permission notice appear in all copies of the
 * software.
 *
 * This software is provided ``as is'', without warranty of any kind.
 */



// Currently supported languages.

define ('LANG_C',          0);
define ('LANG_CPP',        1);
define ('LANG_HTML',       2);
define ('LANG_JAVA',       3);
define ('LANG_PERL',       4);
define ('LANG_SQL',        5);


// Currently supported output styles.

define ('STYLE_DEFAULT', 0);
define ('STYLE_NOCOLOR', 1);
define ('STYLE_EMACS',   2);



/*** Private support functions ***/

function _apply_rules ($src, $rules, $tags, $lang_p, $style_p)
{
    $rules_count = count ($rules);

    $result = '';

    $size = strlen ($src);
    while ($size > 0) {
        $rule_match = false;

        for ($i = 0; $i < $rules_count; $i++) {
            if (preg_match (
                    '/^' . $lang_p['tokens'][$rules[$i]]['regex'] . '/s',
                    $src, $m)) {

                $rule_p = $lang_p['tokens'][$rules[$i]];

                $new_tags = $style_p['tags'][$rule_p['style']];

                // Close previous tags
                for ($j = count ($tags) - 1; $j >= 0; $j--)
                    $result .= _tag_end ($tags[$j]);

                // Open new tags
                for ($j = 0; $j < count ($new_tags); $j++)
                    $result .= _tag_begin ($new_tags[$j]);

                if (count ($rule_p['subrules']) > 0)
                    $result .= _apply_rules ($m[0], $rule_p['subrules'],
                                             $new_tags, $lang_p, $style_p);
                else
                    $result .= htmlspecialchars ($m[0]);

                $match_len = strlen ($m[0]);
                $src = substr ($src, $match_len);
                $size -= $match_len;

                // Close new tags
                for ($j = count ($new_tags) - 1; $j >= 0; $j--)
                    $result .= _tag_end ($new_tags[$j]);

                // Open previous tags
                for ($j = 0; $j < count ($tags); $j++)
                    $result .= _tag_begin ($tags[$j]);

                $rule_match = true;
                break;
            }
        }

        if (! $rule_match) {
            $result .= htmlspecialchars ($src{0});
            $src = substr ($src, 1);
            $size--;
        }
    }

    return $result;
}


function _tag_begin ($tag)
{
    $value = '<' . $tag[0];

    $tag_count = count ($tag);
    for ($i = 1; $i < $tag_count; $i += 2)
        $value .= ' ' . $tag[$i] . '="' . $tag[$i + 1] . '"';
    $value .= '>';

    return $value;
}


function _tag_end ($tag)
{
    return '</' . $tag[0] . '>';
}


/**
 * code_highlight:
 * @source: The string to highlight.
 * @language: Optional. Constant that tells code_highlight() to assume
 * that the @source is written in certain language. The default
 * language is LANG_C.
 * @style. Optional. Tells code_highlight() to use a specific style to
 * produce the HTML result.
 *
 * Returns a string which represents the highlighted HTML version of
 * @source. @language should be one of the following constants:
 *
 *   LANG_C
 *   LANG_CPP
 *   LANG_HTML
 *   LANG_JAVA
 *   LANG_PERL
 *   LANG_SQL
 *
 * The default language is LANG_C. Also, an alternative @style can be
 * defined, and it can be one of:
 *
 *   STYLE_DEFAULT -- Default style.
 *   STYLE_NOCOLOR -- Style without colors. It only inserts <strong>
 *                    and <em> tags.
 *   STYLE_EMACS   -- Style that somewhat resembles the colors used in
 *                    the Emacs editor to highlight source code.
 *
 * Returns: The highlighted version of @source.
 */

function code_highlight ()
{
    // Style definitions used to build the HTML result.

    $styles = array (
        STYLE_DEFAULT => array (
            'external_tags' => array (array ('pre'),
                                      array ('font', 'face', 'Courier New')),
        
            'tags' => array ('comment' => array (
                                 array ('font', 'color', '#444444')),

                             'comment_emph' => array (
                                 array ('font', 'color', '#444444'),
                                 array ('em')),

                             'string' => array (
                                 array ('font', 'color', '#008000')),

                             'esc_string' => array (
                                 array ('font', 'color', '#77dd77')),

                             'char' => array (
                                 array ('font', 'color', '#008000')),

                             'esc_char' => array (
                                 array ('font', 'color', '#77dd77')),

                             'numeric' => array (
                                 array ('font', 'color', '#ff0000')),

                             'identifier' => array (
                                 array ('font', 'color', '#2040a0')),

                             'std_identifier' => array (
                                 array ('font', 'color', '#2040a0'),
                                 array ('strong')),

                             'type' => array (
                                 array ('font', 'color', '#2040a0'),
                                 array ('strong')),

                             'std_type' => array (
                                 array ('font', 'color', '#2040a0'),
                                 array ('strong')),
				     
                             'keyword' => array (
                                 array ('strong')),

                             'std_function' => array (
                                 array ('font', 'color', '#a52a2a'),
                                 array ('strong')),
                         
                             'include' => array (
                                 array ('font', 'color', '#0000ff'),
                                 array ('strong')),
					    
                             'preprocessor' => array (
                                 array ('strong'),
                                 array ('font', 'color', '#0000ff')),

                             'braces' => array (
                                 array ('font', 'color', '#4444ff'),
                                 array ('strong')),

                             'symbol' => array (
                                 array ('font', 'color', '#4444ff')),

                             'header' => array (
                                 array ('strong')),

                             'function_name' => array (
                                 array ('font', 'color', '#ff0000')),

                             'header_arg' => array (
                                 array ('font', 'color', '#2040a0')),

                             'regex' => array (
                                 array ('font', 'color', '#b000d0')),

                             'text' => array (
                                 array ('em')),
					    
                             'entity' => array (
                                 array ('font', 'color', '#ff0000')),

                             'deleted' => array (
                                 array ('font', 'color', '#ff0000'),
                                 array ('strong')),

                             'inserted' => array (
                                 array ('font', 'color', '#0000ff'),
                                 array ('strong')),

                             'none' => array ()
                )),


        STYLE_NOCOLOR => array (
            'external_tags' => array (array ('pre'),
                                      array ('font', 'face', 'Courier New')),
        
            'tags' => array ('comment' => array (),

                             'comment_emph' => array (
                                 array ('em')),

                             'string' => array (),

                             'esc_string' => array (),

                             'char' => array (),

                             'esc_char' => array (),

                             'numeric' => array (),

                             'identifier' => array (),

                             'std_identifier' => array (
                                 array ('strong')),

                             'type' => array (
                                 array ('strong')),

                             'std_type' => array (
                                 array ('strong')),
				     
                             'keyword' => array (
                                 array ('strong')),

                             'std_function' => array (
                                 array ('strong')),
                         
                             'include' => array (
                                 array ('strong')),
					    
                             'preprocessor' => array (
                                 array ('strong')),

                             'braces' => array (
                                 array ('strong')),

                             'symbol' => array (),

                             'header' => array (
                                 array ('strong')),

                             'function_name' => array (),

                             'header_arg' => array (),

                             'regex' => array (),

                             'text' => array (
                                 array ('em')),
					    
                             'entity' => array (),

                             'deleted' => array (
                                 array ('strong')),

                             'inserted' => array (
                                 array ('strong')),

                             'none' => array ()
                )),

        STYLE_EMACS => array (
            'external_tags' => array (array ('pre'),
                                      array ('font', 'face', 'Courier New')),
        
            'tags' => array ('comment' => array (
                                 array ('font', 'color', '#bb0000')),

                             'comment_emph' => array (
                                 array ('font', 'color', '#bb0000'),
                                 array ('em')),

                             'string' => array (
                                 array ('font', 'color', '#bb7766')),

                             'esc_string' => array (
                                 array ('font', 'color', '#cc8877')),

                             'char' => array (
                                 array ('font', 'color', '#bb7766')),

                             'esc_char' => array (
                                 array ('font', 'color', '#cc8877')),

                             'numeric' => array (
                                 array ('font', 'color', '#ff0000')),

                             'identifier' => array (
                                 array ('font', 'color', '#0000ff')),

                             'std_identifier' => array (
                                 array ('font', 'color', '#2040a0'),
                                 array ('strong')),

                             'type' => array (
                                 array ('font', 'color', '#2040a0'),
                                 array ('strong')),

                             'std_type' => array (
                                 array ('font', 'color', '#2040a0'),
                                 array ('strong')),
				     
                             'keyword' => array (
                                 array ('font', 'color', '#b000e0'),
                                 array ('strong')),

                             'std_function' => array (
                                 array ('font', 'color', '#a52a2a'),
                                 array ('strong')),
                         
                             'include' => array (
                                 array ('font', 'color', '#0000ff'),
                                 array ('strong')),
					    
                             'preprocessor' => array (
                                 array ('strong'),
                                 array ('font', 'color', '#0000ff')),

                             'braces' => array (
                                 array ('font', 'color', '#4444ff'),
                                 array ('strong')),

                             'symbol' => array (
                                 array ('font', 'color', '#000000')),

                             'header' => array (
                                 array ('strong')),

                             'function_name' => array (
                                 array ('font', 'color', '#ff0000')),

                             'header_arg' => array (
                                 array ('font', 'color', '#2040a0')),

                             'regex' => array (
                                 array ('font', 'color', '#b000d0')),

                             'text' => array (
                                 array ('em')),
					    
                             'entity' => array (
                                 array ('font', 'color', '#ff0000')),

                             'deleted' => array (
                                 array ('font', 'color', '#ff0000'),
                                 array ('strong')),

                             'inserted' => array (
                                 array ('font', 'color', '#0000ff'),
                                 array ('strong')),

                             'none' => array ()
                ))
        );


    // Rules to parse each language syntax.

    $lang_rules = array (
        LANG_C => array (
            'parent_tokens' => array (
                'comment_emph', 'comment', 'string', 'preprocessor',
                'types', 'symbols', 'char', 'numeric', 'keyword', 'braces',
                'identifier'),

            'tokens' => array (
                'comment' => array (
                    'style'    => 'comment',
                    'regex'    => '\\/\\*([^*].*?|)\\*\\/',
                    'subrules' => array ()),

                'comment_emph' => array (
                    'style'    => 'comment_emph',
                    'regex'    => '\\/\\*\\*.*?\\*\\/',
                    'subrules' => array ()),

                'string' => array (
                    'style'    => 'string',
                    'regex'    => '"(.*?[^"]|)"',
                    'subrules' => array ('esc_char')),

                'preprocessor' => array (
                    'style'    => 'preprocessor',
                    'regex'    => "#(.*?[^\\\\]|)\\n",
                    'subrules' => array ('string', 'included_file',
                                         'numeric', 'comment_emph',
                                         'comment')),

                'included_file' => array (
                    'style'    => 'string',
                    'regex'    => '<.*?>',
                    'subrules' => array ()),

                'char' => array (
                    'style'    => 'char',
                    'regex'    => '\'(\\\\)?.\'',
                    'subrules' => array ('esc_char')),

                'esc_char' => array (
                    'style'    => 'esc_char',
                    'regex'    => '\\\\.',
                    'subrules' => array ()),

                'numeric' => array (
                    'style'    => 'numeric',
                    'regex'    => '\\d\w*?\\b',
                    'subrules' => array ()),

                'identifier' => array (
                    'style'    => 'identifier',
                    'regex'    => '[a-zA-Z_]\w*?\\b',
                    'subrules' => array ()),

                'keyword' => array (
                    'style'    => 'keyword',
                    'regex'    => '(auto|break|case|const|continue|default|' .
                    'do|enum|else|extern|for|goto|if|long|register|return|' .
                    'short|signed|sizeof|static|struct|switch|typedef|' .
                    'union|unsigned|volatile|while)\\b',
                    'subrules' => array ()),

                'types' => array (
                    'style'    => 'std_type',
                    'regex'    => '\\b(int|char|float|double|void)\\b',
                    'subrules' => array ()),

                'braces' => array (
                    'style'    => 'braces',
                    'regex'    => '[\\{\\}]',
                    'subrules' => array ()),

                'symbols' => array (
                    'style'    => 'symbol',
                    'regex'    => '(\\+(\\+|=)?|\\*=?|\\/=?|-(-|=)?|%=?|&&?|' .
                    '\\|\\|?|<<?|>>?|!|\\[|\\]|\\(|\\)|\\?|:|;)',
                    'subrules' => array ())

                )),

        LANG_CPP => array (
            'parent_tokens' => array (
                'comment_emph', 'comment', 'cpp_comment', 'string',
                'preprocessor', 'types', 'symbols', 'char', 'numeric',
                'keyword', 'braces', 'std_identifier', 'identifier'),

            'tokens' => array (
                'comment' => array (
                    'style'    => 'comment',
                    'regex'    => '\\/\\*([^*].*?|)\\*\\/',
                    'subrules' => array ()),

                'comment_emph' => array (
                    'style'    => 'comment_emph',
                    'regex'    => '\\/\\*\\*.*?\\*\\/',
                    'subrules' => array ()),

                'cpp_comment' => array (
                    'style'    => 'comment',
                    'regex'    => '\\/\\/.*?\\n',
                    'subrules' => array ()),

                'string' => array (
                    'style'    => 'string',
                    'regex'    => '"(.*?[^"]|)"',
                    'subrules' => array ('esc_char')),

                'preprocessor' => array (
                    'style'    => 'preprocessor',
                    'regex'    => "#(.*?[^\\\\]|)\\n",
                    'subrules' => array ('string', 'included_file',
                                         'numeric', 'comment_emph',
                                         'comment', 'cpp_comment')),

                'included_file' => array (
                    'style'    => 'string',
                    'regex'    => '<.*?>',
                    'subrules' => array ()),

                'char' => array (
                    'style'    => 'char',
                    'regex'    => '\'(\\\\)?.\'',
                    'subrules' => array ('esc_char')),

                'esc_char' => array (
                    'style'    => 'esc_char',
                    'regex'    => '\\\\.',
                    'subrules' => array ()),

                'numeric' => array (
                    'style'    => 'numeric',
                    'regex'    => '\\d\w*?\\b',
                    'subrules' => array ()),

                'identifier' => array (
                    'style'    => 'identifier',
                    'regex'    => '[a-zA-Z_]\w*?\\b',
                    'subrules' => array ()),

                'std_identifier' => array (
                    'style'    => 'std_identifier',
                    'regex'    => '(true|false|cin|cout)\\b',
                    'subrules' => array ()),

                'keyword' => array (
                    'style'    => 'keyword',
                    'regex'    => '(auto|bool|break|case|catch|const|' .
                    'continue|class|default|delete|do|dynamic_cast|enum|' .
                    'else|extern|explicit|for|friend|goto|if|inline|long|' .
                    'mutable|namespace|new|operator|overload|private|' .
                    'protected|public|register|reinterpret_cast|return|' .
                    'short|signed|sizeof|static|static_cast|struct|switch|' .
                    'template|this|throw|typedef|typeid|typename|union|' .
                    'unsigned|using|virtual|volatile|wchar_t|while)\\b',
                    'subrules' => array ()),

                'types' => array (
                    'style'    => 'std_type',
                    'regex'    => '\\b(int|char|float|double|void)\\b',
                    'subrules' => array ()),

                'braces' => array (
                    'style'    => 'braces',
                    'regex'    => '[\\{\\}]',
                    'subrules' => array ()),

                'symbols' => array (
                    'style'    => 'symbol',
                    'regex'    => '(\\+(\\+|=)?|\\*=?|\\/=?|-(-|=)?|%=?|&&?|' .
                    '\\|\\|?|<<?|>>?|!|\\[|\\]|\\(|\\)|\\?|:|;)',
                    'subrules' => array ())

                )),

        LANG_HTML => array (
            'parent_tokens' => array ('comment', 'entity', 'tag'),

            'tokens' => array (
                'comment' => array (
                    'style'    => 'comment',
                    'regex'    => '<!--.*?-->',
                    'subrules' => array ()),

                'entity' => array (
                    'style'    => 'entity',
                    'regex'    => '&.*?;',
                    'subrules' => array ()),

                'tag' => array (
                    'style'    => 'std_identifier',
                    'regex'    => '<.*?>',
                    'subrules' => array ('string')),

                'string' => array (
                    'style'    => 'string',
                    'regex'    => '(["\']).*?\\1',
                    'subrules' => array ()),

/* TODO
                    {
                        'name'       => 'tag',
                        'regex'      => '<(/|!)?[-.a-zA-Z0-9]*.*?>',
                        'style'      => 'predefined identifier',
                            {
                                'name'       => 'brackets',
                                'regex'      => '[<>]',
                                'style'      => 'braces',
                                'childregex' => []
                            },
                            {
                                'name'       => 'attribute',
                                'regex'      => '[^\'" ]+(?=.)',
                                'style'      => 'identifier',
                                'childregex' => []
                            }
                            ]
                    }
                    ]
 
*/


                )),

        LANG_JAVA => array (
            'parent_tokens' => array (
                'comment', 'comment_emph', 'cpp_comment', 'double_string',
                'single_string', 'include', 'keyword', 'type',
                'braces', 'numeric', 'identifier', 'symbol'),

            'tokens' => array (
                'comment' => array (
                    'style'    => 'comment',
                    'regex'    => '\\/\\*([^*].*?|)\\*\\/',
                    'subrules' => array ()),

                'comment_emph' => array (
                    'style'    => 'comment_emph',
                    'regex'    => '\\/\\*\\*.*?\\*\\/',
                    'subrules' => array ()),

                'cpp_comment'  => array (
                    'style'    => 'comment',
                    'regex'    => '\\/\\/.*?\\n',
                    'subrules' => array ()),

                'double_string' => array (
                    'style'    => 'string',
                    'regex'    => '"(.*?[^\\\\]|)"',
                    'subrules' => array ('esc_char')),

                'single_string' => array (
                    'style'    => 'string',
                    'regex'    => '\'(.*?[^\\\\]|)\'',
                    'subrules' => array ('esc_char')),

                'esc_char' => array (
                    'style'    => 'esc_char',
                    'regex'    => '\\\\.',
                    'subrules' => array ()),

                'numeric' => array (
                    'style'    => 'numeric',
                    'regex'    => '\\d\w*?\\b',
                    'subrules' => array ()),

                'include' => array (
                    'style'    => 'preprocessor',
                    'regex'    => '(import|package)\\b.*?\\n',
                    'subrules' => array (
                        'comment_emph', 'comment', 'cpp_comment',
                        'esc_char')),

                'keyword' => array (
                    'style'    => 'keyword',
                    'regex'    =>
'(break|case|catch|continue|default|do|else|false|finally|for|if|' .
'instanceof|new|null|return|super|switch|this|throw|throws|true|try|' .
'while)\\b',
                    'subrules' => array ()),

                'type' => array (
                    'style'    => 'std_type',
                    'regex'    =>
'(abstract|boolean|byte|char|class|double|extends|final|float|implements|' .
'int|interface|long|native|private|protected|public|short|static|' .
'synchronized|transient|void|volatile)\\b',
                    'subrules' => array ()),

                'braces' => array (
                    'style'    => 'braces',
                    'regex'    => '[\\{\\}\\(\\)\\[\\]]',
                    'subrules' => array ()),

                'identifier' => array (
                    'style'    => 'identifier',
                    'regex'    => '[a-zA-Z_]\w*?\\b',
                    'subrules' => array ()),

                'symbol' => array (
                    'style'    => 'symbol',
                    'regex'    => '[\\*\\-\\+=:;%&\\|<>!]',
                    'subrules' => array ())
                )),


        LANG_PERL => array (
            'parent_tokens' => array (
                'comment', 'variable', 'double_string', 'single_string',
                'interpolated_q_string', 'literal_q_string', 'keyword',
                'braces', 'subroutine', 'std_function', 'predefined_vars',
                'identifier'),

            'tokens' => array (
                'comment' => array (
                    'style'    => 'comment',
                    'regex'    => '#.*?\\n',
                    'subrules' => array ()),

                'variable' => array (
                    'style'    => 'identifier',
                    'regex'    => '[$@%][a-zA-Z_]\w*?\\b',
                    'subrules' => array ()),

                'esc_char' => array (
                    'style'    => 'esc_char',
                    'regex'    => '\\\\.',
                    'subrules' => array ()),

                'double_string' => array (
                    'style'    => 'string',
                    'regex'    => '"(.*?[^\\\\]|)"',
                    'subrules' => array ('esc_char', 'variable')),

                'single_string' => array (
                    'style'    => 'string',
                    'regex'    => '\'(.*?[^\\\\]|)\'',
                    'subrules' => array ('esc_char')),

                'keyword' => array (
                    'style'    => 'keyword',
                    'regex'    => '(new|if|until|while|elsif|else|unless|' .
                    'for|foreach|BEGIN|END|eq|ne|not|and|or)\\b',
                    'subrules' => array ()),

                'predefined_vars' => array (
                    'style'    => 'std_identifier',
                    'regex'    => '\\b($_|$a|$b|$\\d|$&|$`|$\'|$\\+|' .
                    '$^([CDFHNLAEIMOPRSTVWX]|)|' .
                    '@\\+|$\\*|$\\.|$\\/|$\\|$,|$\\\\|$"|$;|$#|$%|$=|$-|' .
                    '@-|$~|$^|$:|$\\?|$!|%!|$@|$$|$<|$>|$\\(|$\\)|' .
                    '$\\[|$\\]|%^H|$ARGV|@ARGV|@F|@INC|@_|%INC|%ENV|$ENV|' .
                    '%SIG|$SIG)\\b',
                    'subrules' => array ()),

                'braces' => array (
                    'style'    => 'braces',
                    'regex'    => '[\\{\\}]',
                    'subrules' => array ()),

                'std_function' => array (
                    'style'    => 'std_function',
                    'regex'    => '(-[rwxoRWXOezsfdlpSbctugkTBMAC]|abs|' .
                    'accept|alarm|atan2|bind|binmode|bless|caller|chdir|' .
                    'chmod|chomp|chop|chown|chr|chroot|close|closedir|' .
                    'connect|continue|cos|crypt|dbmclose|dbmopen|defined|' .
                    'delete|die|do|dump|each|endgrent|endhostent|' .
                    'endnetent|endprotoent|endpwent|endservent|eof|eval|' .
                    'exec|exists|exit|exp|fcntl|fileno|flock|fork|format|' .
                    'formline|getc|getgrent|getgrgrid|getgrnam|' .
                    'gethostbyaddr|gethostbyname|gethostent|getlogin|' .
                    'getnetbyaddr|getnetbyname|getnetent|getpeername|' .
                    'getpgrp|getppid|getpriority|getprotobyname|' .
                    'getprotobynumber|getprotoent|getpwent|getpwnam|' .
                    'getpwuid|getservbyname|getservbyport|getservent|' .
                    'getsockname|getsockopt|glob|gmtime|goto|grep|hex|' .
                    'import|index|int|ioctl|join|keys|kill|last|lc|lcfirst|' .
                    'length|link|listen|local|localtime|lock|log|lstat|' .
                    'm|map|mkdir|msgctl|msgget|msgrcv|msgsnd|my|next|no|' .
                    'oct|open|opendir|ord|our|pack|package|pipe|pop|pos|' .
                    'print|printf|prototype|push|q|qq|qr|quotemeta|qw|qx|' .
                    'rand|read|readdir|readline|readlink|readpipe|recv|' .
                    'redo|ref|rename|require|reset|return|reverse|' .
                    'rewinddir|rindex|rmdir|s|scalar|seek|seekdir|select|' .
                    'semctl|semget|semop|send|setgrent|sethostent|' .
                    'setnetent|setpgrp|setpriority|setprotoent|setpwent|' .
                    'setservent|setsockopt|shift|shmctl|shmget|shmget|' .
                    'shmread|shmwrite|shutdown|sin|sleep|socket|' .
                    'socketpair|sort|splice|split|sprintf|sqrt|srand|stat|' .
                    'study|sub|substr|symlink|syscall|sysopen|sysread|' .
                    'sysseek|system|syswrite|tell|telldir|tie|tied|time|' .
                    'times|tr|truncate|uc|ucfirst|umask|undef|unlink|' .
                    'unpack|unshift|untie|use|utime|values|vec|wait|' .
                    'waitpid|wantarray|warn|write|y)\\b',
                    'subrules' => array ()),

                'identifier' => array (
                    'style'    => 'identifier',
                    'regex'    => '[a-zA-Z_]\w*?\\b',
                    'subrules' => array ()),

                'interpolated_q_string' => array (
                    'style'    => 'none',
                    'regex'    => 'q[qrxw](\\W)(.*?[^\\\\]|)\\1',
                    'subrules' => array ('std_function',
                                         'interpolated_delimited_string')),

                'literal_q_string' => array (
                    'style'    => 'none',
                    'regex'    => 'q(\\W)(.*?[^\\\\]|)\\1',
                    'subrules' => array ('std_function',
                                         'literal_delimited_string')),

                'interpolated_delimited_string' => array (
                    'style'    => 'string',
                    'regex'    => '(\\W)(.*?[^\\\\]|)\\1',
                    'subrules' => array ('esc_char', 'variable')),

                'literal_delimited_string' => array (
                    'style'    => 'string',
                    'regex'    => '(\\W)(.*?[^\\\\]|)\\1',
                    'subrules' => array ('esc_char')),

                'subroutine' => array (
                    'style'    => 'none',
                    'regex'    => 'sub\s+\w+',
                    'subrules' => array ('std_function', 'sub_name')),

                'sub_name' => array (
                    'style'    => 'function_name',
                    'regex'    => '[a-zA-Z_]\w*?\\b',
                    'subrules' => array ())



/* TODO
                {
                    'name'       => 'regex matching I',
                    'regex'      => '(?:\\b| )?(?:/(?:\\\\/|[^/\\n])*(?:/[gimesox]*)|s([^\w\s])(?:\\\\\\2|[^\\2\\n])*?(\\2)[^(\\2)\\n]*?(\\2[gimesox]*))',
                    'style'      => 'regex',
                    'childregex' => []
                },
                {
                    'name'       => 'regex matching II',
                    'regex'      => '(?:\\b| )(?:m|qq?|tr|y)([^\w\s])(?:\\\\\\2|[^\\2\\n])*(?:\\2[gimesox]*)',
                    'style'      => 'regex',
                    'childregex' => []
                },
                {
                    'name'       => '<< stuff',
                    'regex'      => '<<(?:("|\')([^\\n]*)\\2|\\w*).*?^\\3$',
                    'style'      => 'text',
                    'childregex' => []
                },

                TODO
                {
                    'name'       => 'POD',
                    'regex'      => '^=.*?^(?:=cut|\\Z)',
                    'style'      => 'doc comment',
                    'childregex' => []
                }
                ]

*/
                )),


        LANG_SQL => array (
            'parent_tokens' => array ('comment', 'comment_c', 'keyword',
                                      'string', 'std_identifier', 'symbol',
                                      'numeric', 'type', 'identifier'),

            'tokens' => array (
                'comment' => array (
                    'style'    => 'comment',
                    'regex'    => '--.+?\\n',
                    'subrules' => array ()),

                'comment_c' => array (
                    'style'    => 'comment_emph',
                    'regex'    => '\\/\\*.*?\\*\\/',
                    'subrules' => array ()),

                'keyword' => array (
                    'style'    => 'keyword',
                    'regex'    => '(?i)(add|alter|and|as|begin|by|close|' .
                    'constraint|create|desc|end|exception|fetch|foreign|' .
                    'from|function|grant|group|if|in|into|insert|is|key|' .
                    'not|null|on|open|or|order|others|out|package|pragma|' .
                    'primary|references|replace|return|select|set|show|' .
                    'table|to|when|where)\\b',
                    'subrules' => array ()),

                'std_identifier' => array (
                    'style'    => 'std_identifier',
                    'regex'    => '(?i)(false|true)\\b',
                    'subrules' => array ()),

                'string' => array (
                    'style'    => 'string',
                    'regex'    => '\'(.*?[^\\\\]|)\'',
                    'subrules' => array ('esc_char')),

                'esc_char' => array (
                    'style'    => 'esc_char',
                    'regex'    => '\\\\.',
                    'subrules' => array ()),

                'type' => array (
                    'style'    => 'std_type',
                    'regex'    => '(?i)(blol|char|character|date|integer|' .
                    'numeric|varchar|varying)\\b',
                    'subrules' => array ()),

                'symbol' => array (
                    'style'    => 'symbol',
                    'regex'    => '(,|%|<|>|:?=|\\(|\\))',
                    'subrules' => array ()),

                'numeric' => array (
                    'style'    => 'numeric',
                    'regex'    => '\\d\w*?\\b',
                    'subrules' => array ()),

                'identifier' => array (
                    'style'    => 'identifier',
                    'regex'    => '[a-zA-Z_]\w*?\\b',
                    'subrules' => array ())

                ))

        );

    if (func_num_args () < 1) {
        trigger_error ('Wrong parameter count for code_highlight()');
        return false;
    }

    $result = '';
    $code = func_get_arg (0);


    // Default values

    if (func_num_args () >= 2) {
        $lang = func_get_arg (1);
        if (! in_array ($lang, array (LANG_C, LANG_CPP, LANG_HTML, LANG_JAVA,
                                      LANG_PERL, LANG_SQL))) {
            trigger_error ('code_highlight(): Invalid language');
            return false;
        }
    } else {
        $lang = LANG_C;
    }

    if (func_num_args () >= 3) {
        $style = func_get_arg (2);
        if (! in_array ($style, array (STYLE_DEFAULT, STYLE_NOCOLOR,
                                       STYLE_EMACS))) {
            trigger_error ('code_highlight(): Invalid style');
            return false;
        }
    } else {
        $style = STYLE_DEFAULT;
    }


    for ($i = 0, $n = count ($styles[$style]['external_tags']); $i < $n; $i++)
        $result .= _tag_begin ($styles[$style]['external_tags'][$i]);

    $lang_p  = &$lang_rules[$lang];
    $style_p = &$styles[$style];

    $result .= _apply_rules ($code, $lang_p['parent_tokens'], array (),
                             $lang_p, $style_p);

    for ($i = count ($styles[$style]['external_tags']) - 1; $i >= 0; $i--)
        $result .= _tag_end ($styles[$style]['external_tags'][$i]);
    
    return $result;
}

?>
 