<?php
/**
* class XLog
*
* The XLog is a PHP 5 oriented object class to log messages into one or multiple files.
* 
* @category   Log
* @author     Costin Trifan <costintrifan@yahoo.com>
* @copyright  2009 Costin Trifan
* @licence    MIT License http://en.wikipedia.org/wiki/MIT_License
* @version    1.0
* 
* Copyright (c) 2009 Costin Trifan <http://june-js.com/>
* 
* Permission is hereby granted, free of charge, to any person
* obtaining a copy of this software and associated documentation
* files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use,
* copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following
* conditions:
* 
* The above copyright notice and this permission notice shall be
* included in all copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
* HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
* WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
* FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*/
class XLog
{
	private function __clone() {}
	private function __wakeup() {}

	protected
		$log_file 	= '',	# The absolute path to the log file.
		$message 	= '',	# The message to write into the log file
		$line		= '',	# The line where the error has occurred [ ie: __LINE__ ]
		$file 		= '';	# The file where the error has occurred [ ie: __FILE__ ]

	# Error messages
	protected
		$error_file_not_found 	= "Error: Although you have enabled error logging, the log file could not be found!",
		$error_file_open 		= "Error: Couldn't open the log file. Check for permissions!",
		$error_no_message 		= "Error: The content to write into the log file is missing...";



	# .ctor
	/**
	* Constructor
	*
	* @param string $log_file The absolute path to the log file.
	*/
	public function __construct( $log_file='' )
	{
		$this->SetLogFile($log_file);
	}


	/**
	* Set the active log file.
	*
	* @access private
	* @return boolean
	*/
	private function SetLogFile( $log_file='' )
	{
		if ($log_file)
			$this->log_file = $log_file;
	}


	/**
	* Check to see whether or not the log file exists.
	*
	* @access private
	* @return boolean
	*/
	private function FileExists( $log_file )
	{
		return file_exists($log_file);
	}


	/**
	* Clear the log file
	*/
	public function Clear( $log_file='' )
	{
		if (!$log_file)
			$log_file = $this->log_file;

		if (!$this->FileExists($log_file))
			$this->ExitWithError($this->error_file_not_found);
	
		if (($h = fopen($log_file, "w")) !== FALSE)
			fclose($h);
		else
			$this->ExitWithError($this->error_file_open);
	}


	/**
	* Show the content from the log file
	*
	* @param string $log_file The absolute path to the log file.
	*/
	public function GetContent( $log_file='' )
	{
		if (!$log_file)
			$log_file = $this->log_file;

		if (!$this->FileExists($log_file))
			$this->ExitWithError($this->error_file_not_found);

		# Check if the file is php; if it is, remove php's tags and comments
		$info = pathinfo($log_file);
		$file_extension  = strtoupper($info['extension']);


		if (($h = @fopen($log_file, "r")) !== FALSE)
		{
			# Get content
			$content = file_get_contents($log_file);
			@fclose($h);

			if ($file_extension == 'PHP')
			{
				$begin_file = '<?php /*';
				$end_file 	= '*/ ?>';

				$content_length 	= strlen($content);
				$begin_file_length 	= strlen($begin_file);
				$end_file_length 	= strlen($end_file);
				# Strip php's tags and comments
				$content 			= substr($content, $begin_file_length, -$end_file_length);
			}
		}
		return html_entity_decode($content, ENT_QUOTES, "UTF-8");
	}


	/**
	* Write into the log file
	*
	* @param string $message The message to write into the log file.
	* @param string $file  The file where the error has occurred [ ie: __FILE__ ].
	* @param number $line The line where the error has occurred [ ie: __LINE__ ].
	* @param boolean $clear_before Whether or not to delete the existent content from the log file before writting the new one.
	* @param string $log_file The absolute path to the log file.
	*
	* @return void
	*/
	public function Write($message='', $file=NULL, $line=NULL, $clear_before=FALSE, $log_file='')
	{
		if (!$message) $this->ExitWithError($this->error_no_message);
	
		# Setup arguments
		$this->message  = htmlentities($message, ENT_QUOTES, "UTF-8");
		$this->file		= $file;
		$this->line		= $line;
	
		if ($log_file)
			$this->log_file = $log_file;

		if (!$this->FileExists($this->log_file))
			$this->ExitWithError($this->error_file_not_found);

		if ($clear_before)
			$this->Clear($this->log_file);

		# Detect the file's extension so the appropriate function can be called
		$info = pathinfo($this->log_file);
		$file_extension  = strtoupper($info['extension']);

		switch ($file_extension)
		{
			case 'INI' 	: $this->WriteIni(); break;
			case 'PHP' 	: $this->WritePhp(); break;
			default 	: $this->WriteAny(); break;
		}
	}


	/**
	* Write log into an ini file
	*
	* @access protected
	*/
	protected function WriteIni()
	{
		if (($h = fopen($this->log_file, "r+")) !== FALSE)
		{
			$initial_content = file_get_contents($this->log_file);
		
			$br = "\r\n";
			$content  = $br.";---------------------------------------------------------------------------------------".$br;
			$content .= "Date \t= ".date("M d Y H:i:s", time()).$br;
			$content .= "IP \t= ".$_SERVER['REMOTE_ADDR'].$br;
			$content .= "Page \t= ".$_SERVER['HTTP_HOST'].$_SERVER['REQUEST_URI'].$br;
			$content .= "File \t= ".$this->file.$br;
			$content .= "Line \t= ".$this->line.$br;
			$content .= "Error \t= ".$this->message.$br;
			$content .= ";---------------------------------------------------------------------------------------".$br;
			$content .= $br.$initial_content;

			$content = trim($content);
			@fwrite($h, $content, strlen($content));
			@fclose($h);
		}
		else $this->ExitWithError($this->error_file_open);
	}


	/**
	* Write log into a php file
	*
	* @access protected
	*/
	protected function WritePhp()
	{
		if (($h = fopen($this->log_file, "r+")) !== FALSE)
		{
			$br 		= "\r\n";
			$begin_file = '<?php /*';
			$end_file 	= '*/ ?>';

			$initial_content = trim(file_get_contents($this->log_file));

			$content_length 	= strlen($initial_content);
			$begin_file_length 	= strlen($begin_file);
			$end_file_length 	= strlen($end_file);
			# Strip php's tags and comments
			$initial_content 	= substr($initial_content, $begin_file_length, -$end_file_length);


			$content  = "[".date("M d Y H:i:s", time())."]".$br;
			$content .= "---------------------------------------------------------------------------------------".$br;
			$content .= "IP: \t".$_SERVER['REMOTE_ADDR'].$br;
			$content .= "Page: \t".$_SERVER['HTTP_HOST'].$_SERVER['REQUEST_URI'].$br;
			$content .= "File: \t".$this->file.$br;
			$content .= "Line: \t".$this->line.$br;
			$content .= "Error: \t".$this->message.$br;
			$content .= "---------------------------------------------------------------------------------------".$br;
			$content .= $initial_content;

			$content = trim($content);
			$content = $begin_file.$br.$content.$br.$end_file;
			@fwrite($h, $content, strlen($content));
			@fclose($h);
		}
		else $this->ExitWithError($this->error_file_open);
	}


	/**
	* Write log into any other file type
	*
	* @access protected
	*/
	protected function WriteAny()
	{
		if (($h = fopen($this->log_file, "r+")) !== FALSE)
		{
			$initial_content = file_get_contents($this->log_file);
		
			$br = "\r\n";
			$content  = "[".date("M d Y H:i:s", time())."]".$br;
			$content .= "---------------------------------------------------------------------------------------".$br;
			$content .= "IP: \t".$_SERVER['REMOTE_ADDR'].$br;
			$content .= "Page: \t".$_SERVER['HTTP_HOST'].$_SERVER['REQUEST_URI'].$br;
			$content .= "File: \t".$this->file.$br;
			$content .= "Line: \t".$this->line.$br;
			$content .= "Error: \t".$this->message.$br;
			$content .= "---------------------------------------------------------------------------------------".$br;
			$content .= $br.$initial_content;

			$content = trim($content);
			@fwrite($h, $content, strlen($content));
			@fclose($h);
		}
		else $this->ExitWithError($this->error_file_open);
	}


	/**
	* Get the size of the log file.
	* Original source code: http://www.php.net/manual/en/function.filesize.php#84034
	*
	* @return string
	*/
	public function GetFileSize( $log_file='', $round=0 , $add_space=TRUE)
	{
		if (!$log_file)
			$log_file = $this->log_file;

		$size 	= filesize($log_file);
		$sizes 	= array('B', 'KB', 'MB', 'GB');
		$total 	= count($sizes);

		for ($i=0; $size > 1024 && $i < $total; $i++)
			$size /= 1024;

		if ($add_space)
			return round($size,$round).' '.$sizes[$i];
		else
			return round($size,$round).$sizes[$i];
	}


	/**
	* Display the error message
	*
	* @param string $error_message The message to be displayed in case of an error.
	* @access protected
	* @return void
	*/
	protected function ExitWithError( $error_message='' )
	{
		exit("<strong>".$error_message."</strong>");
	}

}
// >> END class XLog
?>