<?php
/********************************************************************
 *
 *  Project: SEO URL TOOLS
 *	Author: Predrag Supurovic
 *	
 *  The goal of this library is to provide some URL 
 *  (universal resource locator) manipulation functions with support for SEO
 *
 *  This is free software. You may use it and redistribute it. You may change code to suit your needs. 
 *  You  may distribute changed code only within your aplication if it is free and with freely available source.
 *  Free status means both free to access and use and free of charge. If you use this code, changed or not 
 *  changed, you must visibly state that you use it. You may make contributions to this original code. 
 *
 *  (c)2007 Copyright by Predrag Supurovic
 *  (c)2007 Copyright by DataVoyage, http://www.datavoyage.com/
 *	All rights reserved
 *
/********************************************************************/
/*

	Using SEO URL's requires you to get organized. You have to decide what syntax URL's would use and make strict 
	rules about it. Also you have to make strict procedures of creating URL-s. 
	
	To make this library we had to make such rules, we had to make such rules. If you want to use this library you 
	must understand and folow those rules. They are strict but not hard to follow. 
	
	First thing, we had to define URL syntax. It should be simply enought to be easily parsed and used, and complex 
	enough to allow various usage. We hope we achieved both goals by defining URL syntax as this:

	URL may have two modes: qulery mode and seo path mode. Query mode is standard URL path containing usual URL elements. 
	All parameters in URL are defined as queries. SEO mode is also known as pretty URL. It offers SEO optimised contents 
	in URL. Parameters may be provided in URL path and as quiery parameters. URL paths are used primarily.
	
	Structure of the URL is defined as:

	query mode: [scheme]://[user]:[pass]@[host]:[port]/[root]/[doc]?[section]&[path]&[options]&[query]#[fragment]
	seo path mode: [scheme]://[user]:[pass]@[host]:[port]/[root]/[section]/[path].[options]?[query]#[fragment]

	[scheme]	scheme of the url (http/https/ftp....)
	[user]		username
	[pass]		password
	[host]		domain part of the url
	[port]		server port number
	[root]		root path of the site address
	[doc]		document name if in query url mode
	[section]	name of the section. It is actually the first directory item of the pretty url part of the url. 
				Section item must match one of the predefined sections. If it does not match, then predefined 
				default section is used, and not parsed from the url. you may use sections to sepaate different
				parts of the site. Most obvious example would be public and admin section. Your application may use
				this information to load different site template for different sections, apply diferent access 
				rules etc...
	[path]		may contain none or more items separated by / each item is used as parameter. You may access them 
				by order or by name if you set path_names. As path does not provide names, names are distinguished
				from order of values provided in path. The first parameter is assotiated to the firt name in 
				path_names, second to the econd etc... Path names should be custom defined. If not custom defined, 
				p0, p1, p2, p3... will be used instead
	[options]	document options. Used for general options like language, page number or client type. Option items
				are separated by dots (.) and ordered. You may access them by name if you define them using 
				url_add_param(). As option values are not named in url, names are assotiated due to type and value 
				of the option. This class recognizes two types: numeric and char (string). Numeric type may contain
				numbers, and it is allowed to have just one numeric type option. String option may contain several
				predefined values. You set those values through url_add_param(). Althrough you may have several 
				string options, they must not have the same values because name of the option is distinguished from 
				it's possible values.
	[query]		query part of the url. items contain name and value, and you may access them by name. query values may 
				always show as calssic url query parameters (i.e. ?par1=val1&par2=val2)but in SEO URLs they also may show 
				in path as /par1=val1/par2=val2/. ACtually, when string url is zreated the second syntah will be enforced 
				in SEO URL's
	[fraction]	part of the url behind #
	
	When URL is defined as such, it is easier to parse it to it's elements, and also it is easier to construct URL 
	with proper syntax if needed elements are provided. Ant that is where this scripts starts: you have to stop seeing URL 
	as a string, but as a group of elemnts that define string. 
	
	We decided to use array structure for URL definition. It consists of neccessary elements that define URL. When you have
	to display URL, whch means to present it as string, pfovided mechanism wil create such string for you if you provide 
	neccessary elements. Also, when you have to read URL elements from string, you also have mechanism to do so. Read 
	elements will be provided as array structure. Althrough some URL manipulation functions of this library do accept 
	both string or array structure for URL, it is advised that you use url aray structure while dealing URL-s within 
	your application. 
	
	You may convert form one strucutre to another at will. Pay attention that conversion may behave diferently depending
	if class is in query url or pretty url mode. You may change mode at will in any time.
	
	Url array structure:
	  [scheme]		// scheme from url	
	  [user]		// username from url
	  [pass]		// password from url
	  [host]		// host from url
	  [port]		// port from url
	  [root]		//
	  [section]		// section from url
	  [path]		// path from url
	    [][name][value]
	    [][name][value]
	    [][name][value]
	  [options]		// options from url
	    [][name][value]
	    [][name][value]
	    [][name][value]
	  [query]		// query from url
	    [][name][value]
	    [][name][value]
	    [][name][value]
	  [fragment]	// fragment from url
	  
	  During the conversion form string to array structure, class will set apropriate items in array structure, if 
	  it is possible to distinguish them. Pay ettention that conversion works differently depending on mode (query
	  url or pretty url). When conversion form array to string occures, class also tries to create url syntax 
	  depending on url mode (query url or pretty url). Items that are not acceptable in query mode url syntax 
	  (section, path, options) will be used as simple query parameters.


	For SEO URLs to work you have to set url rewriting on apache server. Thing is, that you do not have to set 
	complex rules and also you do not have to set them suit your url naming scheme. Tehre is universal rewriting 
	rule which does all the job. Actual URL parsing is all done in PHP. That means that all settings are doen in 
	PHP and not Apache.
	
	Simply, place .htaccess file in root of the site which contains this:
	
	==== .htaccess ===
	RewriteEngine on
	RewriteCond %{REQUEST_FILENAME} !-f
	RewriteCond %{REQUEST_FILENAME} !-d
	RewriteRule ^(.*)$ index.php [L,QSA]
	==== .htaccess ===



	ChangeLog:

  Version: 1.0.38
  	- in progress
	- added version_info();
	- seo_url_tools() has been changed. parameter $p_use_root_script is added to this function, so it is now 
	  possible to force usage of script name in URL. This allowed SEO URLS to work even if rewrite engine 
	  is not turned on.
	- set_pretty_url_mode() also got parameter $p_use_root_script to match seo_url_tools()

  Version: 1.0.37
	- demo included in distribution

  Version: 1.0.36
	- added replacement_add() and clean_url_string()

	Version: 1.0.35
	- bug in get_url_param(). It did not read specified parameters properly

	Version: 1.0.34
	- the first public release

*/

class seo_url_tools {
	var	$version_name = 'SEO URL Tools';
	var $version_number =  '1.0.38';

	//
	// function seo_url_tools()
	// constructor. Initialises object. You may provide URL mode on initialisation if you want to. 
	// Also, you may set if root document will be used in URL's. If pretty url mode is on and 
	// root document is also enforced, then pretty urls will work even if rewrite engine is not
	// turned on (at least it works on Apache 1.3 with PHP 4.3).
	//
	// You may set or change mode later in any time.
	//
	function seo_url_tools($p_mode = false, $p_use_root_script = false) {

		$this->pretty_urls_mode = $p_mode;
		$this->use_root_script = $p_use_root_script;

		$this->document_host = $_SERVER['HTTP_HOST'];

		$m_script_name = $_SERVER['SCRIPT_NAME'];

		if ($this->use_root_script) {
			$this->document_root = str_replace('\\', '/', $_SERVER['SCRIPT_NAME'] . '/');
		} else {
			$this->document_root = str_replace('\\', '/', dirname ($_SERVER['SCRIPT_NAME']) . '/');
		}
		$this->document_root_path = str_replace('\\', '/', dirname ($_SERVER['SCRIPT_NAME']) . '/');

		$this->full_current_url_string = $this->get_full_current_url('string');

		$this->default_section = '';
		$this->default_port = '80';	
		$this->sections = Array();
		$this->options = Array();
		$this->path_names = Array();
		$this->default_path_value = '';		

		$this->url_replacements = array (
			" " => "_",
			"\/" => "_",
			"\\"	=> "_",
			"'" => "-",
			'"' => '-',
			'?' => '',
			'&' => '_'
		);

	}

	//
	//  function set_seo_urls_mode ($p_mode, $p_use_root_script = '') {
	//  Set URL mode: true - soe url (pretty url), false - standard, query url
	//  Set if root script is used in URL's (true - yes, false - no)
	//
	function set_pretty_url_mode ($p_mode, $p_use_root_script = '') {
		$p_mode = ! empty ($p_mode);
		$this->pretty_urls_mode = $p_mode;

		if (is_bool ($p_use_root_script)) {
			$this->use_root_script = $p_use_root_script;
		}
		$this->seo_url_tools ($this->pretty_urls_mode, $this->use_root_script);

	}

	//
	// function init_current_url()
	// Initialises current URL (URL of current doucment). It should be called once when all other 
	//  properties of this class are set. If current URL is not initialised, this function is 
	// called automaticaly through other functions that require current URL
	//
	function init_current_url() {
		$this->current_url_array = $this->url2array($this->full_current_url_string);
		$this->current_url_string = $this->array2url($this->current_url_array);

	}



	/**************************************************************************
	* URL FUNCTIONS
	***************************************************************************/
	//
	//	function get_full_current_url ($p_result_type = 'string')
	//  Returns full URL of the current document
	//	Output type may be string or array, which you specify as parameter. If type is not 
	//  specified, it defaults to string
	//
	function get_full_current_url ($p_result_type = 'string') {
		$m_result = '';
		if (empty ($_SERVER['HTPS']) or ($_SERVER['HTPS'] == 'off')) {
			$m_result .= 'http';
		} else {
			$m_result .= 'https';
		}
		$m_result .= '://';
		$m_result .= $_SERVER['HTTP_HOST'];
		if ($_SERVER['SERVER_PORT'] !== '80') {
			$m_result .= ':';
			$m_result .= $_SERVER['SERVER_PORT'];
		}
		$m_result .= '/';		
		if (isset ($_SERVER['REQUEST_URI'])) {
//			$m_uri = substr ($_SERVER['REQUEST_URI'],1);
			$m_uri = $_SERVER['REQUEST_URI'];

			if ( ($this->use_root_script) and (strpos ($m_uri, $this->document_root) === false) ) {
				$m_uri = str_replace ($this->document_root_path, $this->document_root, $m_uri); 
			}

			$m_result .= substr ($m_uri,1);
		} else {
		  $m_result .= '';
		}
		if ($p_result_type !== 'string') $m_result = $this->url2array ($m_result);

		return $m_result;
	}

	//
	//	function get_current_url ($p_result_type = 'string')
	//	Returns current URL without unnecessary items (default values are removed)
	//	By other means it works the same as get_full_current_url ()
	//
	function get_current_url ($p_result_type = 'string') {

		if (empty ($this->current_url_string)) {
			$this->init_current_url();
		}

		if ($p_result_type == 'string') {
			$m_result = $this->current_url_string;
		} else {
			$m_result = $this->current_url_array;
		}

		return $m_result;
	}
	
	//
	//	function get_exact_current_url ()
	//  Returns exact URL of the current document as sent in request from user
	//	Output value is a string 
	//
	function get_exact_current_url () {
		$m_result = '';
		if (empty ($_SERVER['HTPS']) or ($_SERVER['HTPS'] == 'off')) {
			$m_result .= 'http';
		} else {
			$m_result .= 'https';
		}
		$m_result .= '://';
		$m_result .= $_SERVER['HTTP_HOST'];
		$m_result .= ':';
		$m_result .= $_SERVER['SERVER_PORT'];
		$m_result .= '/';
		if (isset ($_SERVER['REQUEST_URI'])) {
		  $m_result .= substr ($_SERVER['REQUEST_URI'],1);
		} else {
		  $m_result .= '';
		}
		if ($p_result_type !== 'string') $m_result = $this->url2array ($m_result);

		return $m_result;
	}
	
	
	
	//
	// function get_root_url($p_url)
	// returns string URL of the root of the site. This function is meaned to provide easy way to 
	// construct url that leads to index of the whole site. It pays attention to URL options so 
	// they are preserved if neccessary. For instance if language is set, get_root_url() will give 
	// you URL that leads to index page but in current language. If parameer is not set, then 
	// it uses current url
	//
	function get_root_url($p_url = '') {
		if (!empty ($p_url)) {
			if (! is_array ($p_url)) {
				$m_url = $this->url2array ($p_url);
			} else {
				$m_url = $p_url;
			}
		} else {
			$m_url = $this->get_current_url('array');
		}



		$m_root_url = $this->clean_url_properties ($m_url, 'params, section, fragment');
		return ($this->array2url($m_root_url));
	}

	//
	// 	function force_array_url ($p_url)
	//  it takes url in any of two formats (string or array) and returns array structure
	//
	function force_url_array ($p_url) {
		if (! is_array ($p_url)) {
			$m_url = $this->url2array ($p_url);
		} else {
			$m_url = $p_url;
		}
		return $m_url;
	}

	//
	// 	function force_url_string ($p_url)
	//  it takes url in any of two formats (string or array) and returns string structure
	//
	function force_url_string ($p_url) {
		if (! is_string ($p_url)) {
			$m_url = $this->array2url ($p_url);
		} else {
			$m_url = $p_url;
		}
		return $m_url;
	}
	
	//
	// 	function url2array ($p_url_str) {
	//	converts string url to array url structure. This actually parses string to url items. 
	//   You should have all class properties set to be ableto parse URL according to your URL specification
	//
	function url2array ($p_url_str) {

		$p_url_str = $this->clean_string_url ($p_url_str);
	
		$m_url_split = parse_url ($p_url_str);

		$m_result = Array();

		if (empty ($m_url_split['path'])) {
			$m_url_split['path'] = '';
		}

		if (!empty ($this->document_root)) {
			$m_tmp_str = substr ($m_url_split['path'], 0, strlen ($this->document_root));
			if ($m_tmp_str == $this->document_root) {
				$m_result['root'] = $this->document_root;
				$m_url_split['path'] = substr ($m_url_split['path'], strlen ($this->document_root));
			}
		}
		
		if (empty ($m_url_split['path']) and $this->pretty_urls_mode) {
			$m_url_split['path'] = $this->default_path_value;
		}
		
		$m_result = array_merge ($m_result, $m_url_split);

		if ($this->pretty_urls_mode and isset ($m_url_split['path'])) {

			$m_path = split ('\.', $m_url_split['path'], 2);

			$m_result['path'] = Array();
			$m_tmp = preg_split ("/\//", $m_path[0]);

			foreach ($m_tmp as $m_key => $m_value) {
				if (! empty ($m_value)) {
					$m_path_split [] = $m_value;
				}
			}
			
			if (empty ($m_path_split)) $m_path_split[$this->get_path_name(0)] = $this->default_path_value;
			
			$m_item_count = 0;
			if (count ($m_path_split) > 0) { 
				foreach ($m_path_split as $m_key => $m_item) {
					$m_corected_key = $m_key;
					if (($m_key == 0) and $this->section_exists ($m_item)) {
						$m_result['section'] = $m_item;
						$m_item_count--;
					} else {
						if (! empty ($m_item)) {
							if (strpos ($m_item, '=')) {
								if (!isset ($m_result['query']) or ! is_array ($m_result['query'])) {
									unset ($m_result['query']);
									$m_result['query'] = Array();
								}

								$m_item_query = split ("=", $m_item,2);
								$m_result['query'][$m_item_query[0]] = $m_item_query[1];	
								$m_item_count--;								
							} else {
								if (isset ($this->path_names[$m_item_count])) {
										$m_result['path'][$this->path_names[$m_item_count]] = $m_item;
								} else {
										$m_result['path']['p' . $m_item_count] = $m_item;
								}
							}
						}
					}
					$m_item_count++;
				}
			}
			if ($this->pretty_urls_mode) {

				$m_result['options'] = Array();
				if (empty ($m_path[1])) $m_path[1] = '';
				$m_options_split = preg_split ("/\./", $m_path[1]);

				foreach ($m_options_split as $m_item) {
					if (! empty ($m_item)) {
						$m_option_name = $this->match_option ($m_item);
						if (! empty ($m_option_name)) {
							$m_result['options'][$m_option_name] = $m_item;
						}
					}
				}
			} else {

				if (!empty ($m_path[1])) {
					$m_result['path'][count($m_result['path'])-1] .= '.' . $m_path[1];
				}
			}
		} else {
			if (isset ($m_result['path'])) {
				$m_tmp = $m_result['path'];
				unset ($m_result['path']);
				$m_result['doc']= $m_tmp;
			}

		} // if ($this->pretty_urls_mode and isset ($m_url_split['path']))

		if (isset ($m_url_split['query'])) {
			$m_query_split = preg_split ("/\&/", $m_url_split['query']);
			$m_item_count = count ($m_query_split);
			
			if (! is_array ($m_result['query'])) {
				unset ($m_result['query']);
				$m_result['query'] = Array();
			}

			for ($i = 0; $i < $m_item_count; $i++) {
				$m_split = preg_split ("/=/", $m_query_split[$i]);
				$m_query_val = isset ($m_split[1]) ? $m_split[1] : null;
				if ($this->option_exists ($m_split[0])) {
					$m_result['options'][$m_split[0]] = $m_query_val;
				} else {
					$m_result['query'][$m_split[0]] = $m_query_val;
				}
			}

			if (!empty ($m_result['query']['section'])) {
				$m_result['section'] = $m_result['query']['section'];
				unset ($m_result['query']['section']);
			}

		}
		
		$m_result = $this->clean_item ($m_result);
		return $m_result;
	} // function url2array

	
	//
	// 	function array2url ($p_rurl) 
	//  Converts url array structure into string. Use this when you need to dislay URL in document. 
	//  You should have all class properties set to be able to construct URL that suits your URL specifications
	//
	function array2url ($p_url_array, $p_skip_defaults = true) {

		$m_query = '';
		$m_result = '';

		if (! is_array ($p_url_array)) {
			trigger_error('$p_url_array parameter of array2url() must be array', E_USER_ERROR);		
		}

		if (! empty ($p_url_array['scheme'])) $m_result .= $p_url_array['scheme'] . '://';

		if (! empty ($p_url_array['name'])) { 
			$m_result .= $p_url_array['name'];
				if (! empty ($m_result)) {
					$m_result .= ':' . $p_url_array['pass'];
				}
			$m_result .= '@';
		}

		if (! empty ($p_url_array['host'])) $m_result .= $p_url_array['host'];
		
		if (! empty ($p_url_array['root'])) $m_result .= $p_url_array['root'];		
		
		if ($this->pretty_urls_mode == 0) {
			if (! empty ($p_url_array['doc'])) { 
				$m_result .= $p_url_array['doc'];
			}
			
			if (! empty ($p_url_array['section']) and ! $this->is_section_default ($p_url_array['section']) and
				empty ($p_url_array['query']['section'])) {
				$p_url_array['query']['section'] = $p_url_array['section'];
			}


			if (! empty ($p_url_array['path'])) { 
				foreach ($p_url_array['path'] as $m_key => $m_item) {
						$p_url_array['query'][$m_key] = $m_item;
				}
			}			

			if (! empty ($p_url_array['options'])) { 
				foreach ($p_url_array['options'] as $m_key => $m_item) {
					if (! $this->is_option_default($m_key, $m_item) and empty ($p_url_array['query'][$m_key]))
						$p_url_array['query'][$m_key] = $m_item;
				}
			}			
			
			if (! empty ($p_url_array['query'])) { 
				$m_query_2 = $this->url_build_query ($p_url_array['query']);
				if (!empty ($m_query)) $m_query .= '&';
				$m_query .= $m_query_2;
			}

			if (!empty ($m_query)) $m_result .= '?' . $m_query;
			
			
		} else {
			if (! empty ($p_url_array['section']) and ! $this->is_section_default ($p_url_array['section'])) {
					$m_result .= '/' . $p_url_array['section'];
			}

			$m_options = '';
			if (! empty ($p_url_array['options']) and $this->pretty_urls_mode) {
				foreach ($p_url_array['options'] as $m_key => $m_item) {
					if (! empty ($m_item) and ! $this->is_option_default($m_key, $m_item)) 
						$m_options .= '.' . $m_item;
				}
			}			

			if (! empty ($p_url_array['path']) and (count ($p_url_array['path']) > 0) ) { 
				foreach ($p_url_array['path'] as $m_key => $m_item) {
						if (!(empty ($m_item) or (
							    (count ($p_url_array['path']) == 1) and 
							    $this->is_menu_path_default($m_item) and 
							    empty ($m_options))))
						{
							$m_result .= '/' . $m_item;
						}
				}
			} else {
			  if (! empty ($m_options)) $m_result .= '/' . $this->default_path_value;
			
			}
			
			if (! empty ($p_url_array['query'])) { 
				$m_query = $this->url_build_query ($p_url_array['query'],'','/');
				if (! empty ($m_query)) $m_result .= '/';
				$m_result .= $m_query;
			}
		
			$m_result .= $m_options;


		}
		
		if (! empty ($p_url_array['fragment'])) $m_result .= '#' . $p_url_array['fragment']; 
		
		$m_result = $this->clean_string_url ($m_result);

		return $m_result;

	} // function seo_array2url
	
	
	//
	// 	function add_query ($p_query, $p_param_name, $param_value) 
	//  this is internal function. It creates string in proper query format to be added to string url path
	//
	function add_query ($p_query, $p_param_name, $param_value) {
		if (!empty ($p_query)) $p_query .= '&';
		$p_query .= "$p_param_name=$param_value";
		return $p_query;
	}

	//
	// url_build_query ($data, $prefix=null, $sep='', $key='')
	// internal function. It creates url query part from array 
	//
	function url_build_query ($data, $prefix=null, $sep='', $key='') {

		$ret = array();
		foreach ((array)$data as $k => $v) {
			if (!empty ($v)) {
				if (is_int($k) && $prefix != null) {
					$k = $prefix.$k;
				};
				if (!empty($key)) {
					$k = $key."[".$k."]";
				};

				if (is_array($v) || is_object($v)) {
					array_push($ret, $this->rl_build_query ($v,"",$sep,$k));
				}
				else {
					array_push ($ret, $k. '=' . $v);
				};
			};
		};

		if(empty($sep)) {
			$sep = ini_get("arg_separator.output");
		};

		return implode($sep, $ret);
	}
	
	/****************************************************************************/
	// PARAMETERS
	/****************************************************************************/	

	
	//
	// function url_add_param ($p_url, $p_param, $p_clean_what = nil)
	//
	// Adds specified list of parameters to specified url. If $p_url already contains parameter 
	// this will change it's value. If new value for existing parameter is empty, parameter 
	// will be removed from url. $purl may be string or array structure.
	//
	// $p_param contains defining parameters. May be string or array structured in the same manner as url array struct.
	//
	// If string parameters are used they will be parsed and array structure filled in. If query mode, string values 
	// cannot have definitions of section, path parameters and options in path part of url as that part of the string 
	// would be treated as document name. If you need to change section, path parameters or options, use query 
	// parameters in url string (worse option) or pass values as array structure.
	//
	// section, path and options parts of structure are used to create prettyurl params if pretty url mode is used
	// if query mode is on, these options would be treated as query parameters. That allows you to use the same 
	// structure to set parameters and respective url syntax would be used depending on url mode (pretty url or query url).
	//
	// if $p_clean_what is specified, url will be cleaned before params added. For parameter options check 
	//
	// Result function is array structure with altered URL parameters
	//
	function url_add_param ($p_url, $p_param, $p_clean_what = null) {
	
		$m_url = $this->force_url_array ($p_url);
		
		$m_url = $this->clean_url_properties ($m_url, $p_clean_what);
		
		$m_param = $this->force_url_array ($p_param);
		
		if (! isset ($m_url['path'])) $m_url['path'] = Array();
		if (! isset ($m_url['options'])) $m_url['options'] = Array();
		if (! isset ($m_url['query'])) $m_url['query'] = Array();

		if (!empty ($m_param['scheme'])) $m_url['scheme'] = $m_param['scheme'];
		if (!empty ($m_param['user'])) $m_url['user'] = $m_param['user'];
		if (!empty ($m_param['pass'])) $m_url['pass'] = $m_param['pass'];
		if (!empty ($m_param['host'])) {
			$m_url['host'] = $m_param['host'];
			if (empty ($m_param['root'])) $m_url['root'] = '';
			if (empty ($m_param['doc'])) $m_url['doc'] = '';
		}
		if (!empty ($m_param['port'])) $m_url['port'] = $m_param['port'];		
		if (!empty ($m_param['root'])) $m_url['root'] = $m_param['root'];
		if (!empty ($m_param['doc'])) $m_url['doc'] = $m_param['doc'];
		if (!empty ($m_param['section'])) $m_url['section'] = $m_param['section'];
		if (!empty ($m_param['fragment'])) $m_url['fragment'] = $m_param['fragment'];

		if (isset ($m_param['path']) and (count ($m_param['path']) > 0)) $m_url['path'] = $m_param['path'];
		
		if (isset ($m_param['options'])) {
			$m_url['options'] = array_merge ($m_url['options'], $m_param['options']);
		}
	
		if (isset ($m_param['query'])) {
			$m_url['query'] = array_merge ($m_url['query'], $m_param['query']);		
		}
		
		$m_result = $this->clean_item ($m_url);

		return $m_result;
	} // url_add_param	
	
	
	//
	// function url_current_add_param ($p_params)
	//
	// Adds specified list of parameters to current url (url of current document).
	// it wilrks similar to url_add_param() with exception that you may specify type of
	// result through parameter $p_result_type. It is useful when you need to change few 
	// parameters to current URL and display it in a document.
	//
	function url_current_add_param ($p_param, $p_result_type = 'string', $p_clean_what = null) {

		$m_param = $this->force_url_array ($p_param);
		$m_url = $this->get_current_url ('array');
		$m_result = $this->url_add_param ($m_url, $m_param, $p_clean_what);
		
		if ($p_result_type == 'string') $m_result = $this->array2url ($m_result);

		return $m_result;
	}	
	
	
	//
	// 	function quick_current_url ($p_option='', $p_item='', $p_page='', $p_lang='', $p_client='')
	//	Changes some parameter values in current URL
	//  This function is targeted to situation where you have simple way to radically change URL by setting lots 
	//  of parameters but to avoid creating complex array structure. For instance, when you create URL of menu 
	//  option you want to remove most of the parameters from current url and keep or change one or two.
	//	
	function quick_current_url ($p_option='', $p_item='', $p_page='', $p_lang='', $p_client='') {

		$m_params['path']['opt'] = getMenuUrl ($p_option);
		$m_params['path']['item'] = $p_item;
		$m_params['option']['page'] = $p_page;
		$m_params['option']['lang'] = $p_lang;
		$m_params['option']['client'] = $p_client;

		return $this->url_current_add_param ($m_params); 

	}
	
	
		
	/****************************************************************************/
	// SECTIONS
	/****************************************************************************/	
	//
	// 	function set_sections ($p_sections = '', $p_default)
	//  This defines possible sections in URL. Url section will be recognized in URL only if section is 
	//  previously defined through this option. You may specify list of sections as array or comma separated sring.
	//  You also may specify name of the section that will be used as default if section is not specified in URL.
	//  On string url ceation, if url section is defautl section, section would not be displayed in URL.
	//
	function set_sections ($p_sections = '', $p_default = false) {
		if (!is_array($p_sections)) {
			$p_sections = explode (',', $p_sections);
		}
		if (count($p_sections) == 0) {
			$p_sections[] = '';
		}
        if (empty ($p_default)) $p_default = $p_sections[0];
		
		$this->sections = $p_sections;
		$this->default_section = $p_default;
		
		unset ($this->current_url_string);

	} // function set_sections
	
	//
	// 	function section_exists ($p_section)
	//  check if specified section name exists in defined sections list
	//
	function section_exists ($p_section) {
		return in_array ($p_section, $this->sections);
	} // function is_default_section
	
	//
	// 	function get_section_default ($p_section)
	//  returns name of the default section
	//
	function get_section_default ($p_section) {
		return ($this->default_section);
	} // function get_default_section	

	//
	// 	function is_section_default ($p_section)
	//  checks if specified section name is default section
	//
	function is_section_default ($p_section) {
		return ($p_section == $this->default_section);
	} // function is_default_section	

	
	//**************************************************************************/
	// PORT
	//**************************************************************************/	
	
	//
	// 	function set_default_port ($p_value)
	//  Sets default port in url. This means that if portis not part of the url, this port will be 
	//  assumed. Also, on string url creation, if url port is default port, then port will not be shown in url string.
	//  By default, http port is used, port 80
	//
	function set_default_port ($p_value) {
		if (is_numeric ($p_value)) {
			$this->default_port = $p_value;
		} else {
			trigger_error('$p_value parameter of set_port() must be numeric', E_USER_ERROR);		
		}
		unset ($this->current_url_string);
	} // function set_default_port 
	

	//
	// 	function is_port_default ($p_value)
	//  checks if specified port is default
	//
	function is_port_default ($p_value) {
		return ($p_value == $this->default_port);
	} // function is_default_port



	//**************************************************************************/
	// OPTIONS
	//**************************************************************************/	
	
	//
	// 	function set_option ($p_name, $p_type, $p_default_value='', $p_values='')
	//  Adds option to option list. You must specify option name and type. 
	//  $p_name is name that would beused to access this option value
	//  $p_type is type of the option and it may be 'numeric' or 'char'. You may define
	//  just one numeric type parameter and unlimied nubmer of char parameters
	//  $p_default value is optional. You may specify default value of the option. That means if option value 
	//  is not specified in URL, default value will be asumed. Also, when UTL string is reated and this URL 
	//  option has default value, URL option will be ommited from string.
	//  $p_values is not required for numeric type as in numeric type, value may be any number. But, in char <br />
	//  type option, $p_values specifies allowed values for this specific option. This is important, as string 
	//  URL does not contain option name, so parser mustgess option name by comparing parsed value to allowed 
	//  values. If it get match than it assotiates value to this option name. That introduces limit that if you 
	//  have several char type options, they must not have the same any allowed value. $p_values may be array list <br />
	//  or string with comma separated list of values
	//
	function set_option ($p_name, $p_type, $p_default_value='', $p_values='') {

		$m_pos = in_array ($p_type, Array ('char', 'numeric'));

		if ($m_pos === false) {
			trigger_error('$p_type must have value of "char" (character) or "numeric" (numeric)', E_USER_ERROR);
		}

		if ($p_type == 'char') {
			if (!is_array($p_values)) $p_values = explode (',', $p_values);
			if (count($p_values) == 0) trigger_error('$p_values must have at least one record', E_USER_ERROR);
			if (empty ($p_default_value)) $p_default_value = $p_values[0];
		}

		if ($p_type == 'numeric') {
			$p_values = null;
			if (empty ($p_default_value)) $p_default_value = 0;
		}

		$this->options[$p_name]['id'] = $p_name;
		$this->options[$p_name]['type'] = $p_type;
		$this->options[$p_name]['values'] = $p_values;
		$this->options[$p_name]['default_value'] = $p_default_value;
		
		unset ($this->current_url_string);

	} // function set_option

	//
	// 	function option_exists ($p_option)
	// checks if option of specified name is defined
	//
	function option_exists ($p_option) {
		return (isset ($this->options[$p_option]));
	} // function option_exists

	//
	// 	function get_option_default ($p_option)
	//  returns default value of specified option name
	//
	function get_option_default ($p_option) {
		if ($this->option_exists ($p_option)) {
			return ($this->options[$p_option]['default_value']);
		} else {
			return '';
		}
	} // function get_option_default


	//
	// function is_option_default ($p_option, $p_value)
	// checks if $p_value is default value of the option named $p_option
	//
	function is_option_default ($p_option, $p_value) {
		return ($p_value == $this->get_option_default ($p_option));
	}
	
	
	//
	// 	function match_option ($p_value) {
	//  Returns name of the option, which $p_value belongs to.
	//
	function match_option ($p_value) {
		$m_result = '';

		foreach ($this->options as $m_option) {
			if (($m_option['type'] == 'numeric') and is_numeric ($p_value)){
				$m_result = $m_option['id'];
				break;
			}

			if (($m_option['type'] == 'char') and ! (array_search($p_value, $m_option['values']) === false) ) {
				$m_result = $m_option['id'];
				break;
			}
		}
		return $m_result;

	} // function match_option


	
	//**************************************************************************/
	// PATH NAMES
	//**************************************************************************/	
	
	//
	// 	function set_path_names ($p_names) {
	//  Sets path names. Specify path names as comma delimited string. You must 
	//  specify path names in order they should show in path. As path values are not named, values read from 
	//  path will be assigned to path names accorting to order of appearance. If you do not specify path names 
	//  path names will be automaticaly generated as p folowed by path order. The first path valeu will be assotiated 
	//  to p0, the second to p1 etc. If you specify path names then then they will replace autogenerated names. Of for 
	//  some reason url contains more path values than custom defined path names, automatic path_names will be used for
	//  those not custom defined. Autoamtic path_names will contain p folowed by order or the value (including named 
	//   values). i.e. fourth value will be named p3.
	//
	function set_path_names ($p_names) {
		if (! empty ($p_names)) {
			$this->path_names = explode (',', $p_names);
		} else {
			$this->path_names = Array();
		}
		unset ($this->current_url_string);
	}


	//
	// 	function path_name_exists ($p_value) {
	//  Checks if name $_p_value is defined as path name
	//
	function path_name_exists ($p_value) {
		return (in_array ($p_value, $this->path_names));

	}
	
	//
	// 	function get_path_name ($p_value) {
	//  Returns name of the path_name by order. $p_value is path_name order number starting from zero. 
	//  If there is no custom defined path name, autoamtic path name is generated acording 
	//  to parameter order
	//
	function get_path_name ($p_value) {
		if (!empty ($this->path_names[$p_value])) {
			return ($this->path_names[$p_value]);
		} else {
			return ('p' . $p_value);		
		}

	}	
	
	//
	// 	function set_default_path_value ($p_value) {
	//  Sets default path value. This is the value used if path values are not specified in URL.
	//
	function set_default_path_value ($p_value) {
		if (! empty ($p_value)) {
			$this->default_path_value = $p_value;
		} else {
			$this->default_path_value = '';
		}

	}
	
	//
	// function is_menu_path_default($m_value)
	// Checks if the first path value is default value. 
	//
	function is_menu_path_default($p_value) {
		$m_value = explode ('-', $p_value, 2);
		$m_def_value = explode ('-', $this->default_path_value, 2);
		return ($m_value[0] == $m_def_value[0]);
	} // function is_default_path_value
	
	
	
	//**************************************************************************/
	// MISCELANEOUS
	//**************************************************************************/	
	
	//
	// function url_root ()
	// Return root level of site
	// This works if function is called from script which is in the root of the site, 
	// or if script is included by other script which is in the root. Pay etention that this returns apsolute 
	// root of the site regardles of current parameters. If you need url of site index use get_root_url()
	//
	function url_root() {
		$m_result = "http://" . $this->document_host  . $this->document_root;	
		if (substr($m_result, -1,1) !== "/"){
			$m_result .= '/';
		}

		return $m_result;
	}
	
	/*
	* function clean_url_properties ($p_url, $p_clean_what = nil)
	*
	* Cleans specified items of the url (all or from comma separated list)
	* 
	* Possible clean options:
	*	all			remove all items from the url
	*	params		remove path and query
	*	vars		remove section, path, options, query and fragment
	*	scheme		remove scheme from url	
	*	user		remove username from url
	*	pass		remove password from url
	*	host		remove host from url
	*	port		remove port from url
	*	root		remove root from url
	*	doc			remove document from url
	*	section		remove section from url
	*	path		remove path from url
	*	options		remove options from url
	*	query		remove query from url
	*	fragment	remove fragment from url
	*
	* Returns url in format same as input format (if input is array, output would be array too.
	*/
	function clean_url_properties ($p_url, $p_clean_what) {

		$m_is_array = is_array ($p_url);
		$m_url = $this->force_url_array ($p_url);

		$m_clean_what = split (',', $p_clean_what);

		foreach ($m_clean_what as $m_item) {
			if ($m_item == 'scheme' or $m_item == 'all') unset ($m_url['scheme']);		
			if ($m_item == 'user' or $m_item == 'all') unset ($m_url['user']);
			if ($m_item == 'pass' or $m_item == 'all') unset ($m_url['pass']);
			if ($m_item == 'host' or $m_item == 'all') unset ($m_url['host']);
			if ($m_item == 'port' or $m_item == 'all') unset ($m_url['port']);
			if ($m_item == 'root' or $m_item == 'all') unset ($m_url['root']);
			if ($m_item == 'section' or $m_item == 'vars' or $m_item == 'all') unset ($m_url['section']);						
			if ($m_item == 'doc' or $m_item == 'all') unset ($m_url['doc']);						
			if ($m_item == 'path' or $m_item == 'vars' or $m_item == 'params' or $m_item == 'all') unset ($m_url['path']);
			if ($m_item == 'options' or $m_item == 'vars' or $m_item == 'all') unset ($m_url['options']);
			if ($m_item == 'query' or $m_item == 'vars' or $m_item == 'params' or $m_item == 'all') unset ($m_url['query']);
			if ($m_item == 'fragment' or $m_item == 'vars' or $m_item == 'all') unset ($m_url['fragment']);
		}
		if (! $m_is_array) $m_url = $this->array2url ($m_url);
		return ($m_url);
	}
	
	
	//
	//  function clean_item ($p_value)
	//  Removes empty items in array
	//
	function clean_item ($p_value) {
		if (is_array ($p_value)) {
			if ( count ($p_value) == 0) {
				unset ($p_value);
				$p_value = null;
			} else {
				foreach ($p_value as $m_key => $m_value) {
					$m_value = $this->clean_item ($m_value);
					if (empty ($m_value) or (count ($m_value) == 0)) {
						unset ($p_value[$m_key]);
						$p_value[$m_key] = null;
					}
				}
			}
		} else {
			if (empty ($p_value)) {
				unset ($p_value);
				$p_value = null;
			}
		}
		return $p_value;
	}

	
	
	//
	// 	function get_url_param ($p_url, $p_param_type, $p_param_name = '')
	//  Returns valueof the URL parameter from specified url
	//  $p_url is url that shoud be parsed. It may be string or array
	//  $p_param_type is type of parameter (section, path, option...)
	//  $p_param_name is name of parameter which value you want to get
	//
	function get_url_param ($p_url, $p_param_type, $p_param_name = '') {

			$m_url = $this->force_url_array ($p_url);

			switch ($p_param_type) {


				case 'scheme' :
				case 'user' :
				case 'pass' :
				case 'host' :
				case 'root' :
				case 'doc' :
				case 'fraction' :

					if (isset ($m_url[$p_param_type])) {
					    	$m_result = $m_url[$p_param_type];
					} else {
				    		$m_result = '';
					}
					break;


				case 'port' :
					if (isset ($m_url['port'])) {
				    		$m_result = $m_url['port'];
					} else {
				    		$m_result = $this->default_port;
					}
					break;

				case 'section' :
					if (isset ($m_url['section'])) {
				    		$m_result = $m_url['section'];
					} else {
				    		$m_result = $this->default_section;
					}
					break;


				case 'path' :
						if (isset ($m_url['path'][$p_param_name])) {
				    		$m_result = $m_url['path'][$p_param_name];
						} else {
							if (isset ($m_url['query'][$p_param_name])) {
					    		$m_result = $m_url['query'][$p_param_name];
							} else {
 					    		$m_result = '';
							}
						}
						break;


				case 'options' : 
					if (isset ($m_url[$p_param_type][$p_param_name])) {
				    		$m_result = $m_url[$p_param_type][$p_param_name];
					} else {
						if (isset ($this->options[$p_param_name]['default_value'])) {
							$m_result = $this->options[$p_param_name]['default_value'];
						} else {
							$m_result = null;
						}
					}
					break;


				case 'query' : 
					if (isset ($m_url[$p_param_type][$p_param_name])) {
				    		$m_result = $m_url[$p_param_type][$p_param_name];
					} else {
						if (isset ($this->query[$p_param_name]['default_value'])) {
							$m_result = $this->query[$p_param_name]['default_value'];
						} else {
							$m_result = null;
						}
					}
					break;


				}

		return $m_result;
	}


	//
	// function clean_string_url ($p_value)
	// Cleans url strnig from multiplied / in path.
	//
	function clean_string_url ($p_value) {

		$m_result = str_replace('://', ':::', $p_value);
		$m_result = str_replace('//', '/', $m_result);
		$m_result = str_replace(':::', '://', $m_result);		
		return ($m_result);

	}

	//
	// force_correct_current_url()
	// Checks if URL user typed in browser matches correct url for the document, and if not, 
	// enforces correct URL
	//
	function force_correct_current_url() {
		$m_url = trim ($this->get_full_current_url('string'),'/');
		$m_parsed_url = $this->url2array($m_url);
		$m_reparsed_url = $this->array2url($m_parsed_url);	
		if ($m_url !== $m_reparsed_url) {
			header("HTTP/1.1 301 Moved Permanently");
			header('Location: ' . $m_reparsed_url);
		}
	}

	//
	// clean_url_string ($p_url_string) 
	// Clean url string from notwanted characters (or substrings). It repalces characters not wallowed in url 
	// with proper replacements. You may customize replacament definitions by using replacements_add() or by 
	// setting contents of file _url_transform.php which should be in the same directory where this library resides.
	//
	function clean_url_string ($p_url_string) {
		$custom_transform = dirname (__FILE__) . '/_url_transform.php';

		if (file_exists ($custom_transform)) {
			include ('_url_transform.php');
			if (!empty ($REPLACEMENTS) and is_array ($REPLACEMENTS))
				 $this->replacements_add ($REPLACEMENTS);
		}

		$m_url_string = strtr($p_url_string, $this->url_replacements);

		$m_url_string = urlencode ($m_url_string);
     
		return $m_url_string;
	}   

	//
	// function replacements_add ($p_array)
	// Sets list of replacements for clean_url_string(). Input is array list of strings that should be replaced 
	// and replacements.
	//
	function replacements_add ($p_array) {
		$this->url_replacements = array_merge ($this->url_replacements, $p_array);
	}
	
	//
	// function version_info ($p_item = '')
	// Raturns version infroamtion. Input parameter may be empty, 'name' or 'number'. Empty parameter
	// returns version info as array, 'name' returns name and 'number' returns number.
	//
	function version_info ($p_item = '') {
		switch ($p_item) {
			case 'name': 
				$m_result = $this->version_name;
				break;
			case 'number' :
				$m_result = $this->version_number;
				break;
			default:
				$m_result['name'] = $this->version_name;
				$m_result['number'] = $this->version_number;
		}
		return ($m_result);
	}

	
} // class

?>