/*
 * Dash Media Player: External Interface Gateway 
 * Copyright (c) 2008 TMT Digital LLC.
 *
 * Author:  Travis Tidwell | http://www.travistidwell.com
 *
 * Dependencies:
 *    jQuery Library
 *
 * This script is used as a connection interface gateway to control the player from an external source.
 * It is also used as a means for communication between multiple instances of the Dash Media Player
 * on the page.  This allows you to separate the Playlist and the actual
 * player on the page, but yet have them behave as a single player.  For more information on how you can
 * integrate this great feature on your website, visit http://www.tmtdigital.com/remoteplaylist
 *
 * License: GPL
 */

var dashReady = false;
var dashObjects = new Object;
var dashCallback = function( args ){};
$(document).ready(function() {
   dashReady = true;  
});

function isDashReady() {
   return dashReady;
}

function dashRegisterObject( id ) {
   dashObjects[id] = true;
}

/** 
 * Loads a single node into the player.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The node ID of the node you would like for this player to load.
 */
function dashLoadNode( id, nodeId ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          dashObj.loadNode( nodeId );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Loads a media file in the player.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The file that you would like for the player to load. 
 */
function dashLoad( id, file ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          dashObj.load( file );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Plays a media file.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The file that you would like for the player to play.  
 *          This is not necessary if you use dashLoadNode or dashLoad before making this call.
 *          If you call this function with the file provided, then the player will load that file
 *          before actually playing it.
 */
function dashPlay( id, file ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          dashObj.play( file );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Pauses the media file that is playing in the player.
 *
 * @param - The ID of the player you would like to send this request too.
 */
function dashPause( id ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          dashObj.pause();
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Stops the media file that is playing in the player.
 *
 * @param - The ID of the player you would like to send this request too.
 */
function dashStop( id ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          dashObj.stop();
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }

   return false;
}

/** 
 * Seeks the media file that is playing to the time specified.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The time that you would like to seek the track too.
 */
function dashSeek( id, seekTime ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          dashObj.seek( seekTime );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Sets the volume of the media being played.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The volume that you would like to set the media too.
 */
function dashVolume( id, vol ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          dashObj.volume( vol );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Gets the volume of the media being played.
 *
 * @param - The ID of the player you would like to send this request too.
 */
function dashGetVolume( id ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          return dashObj.getVolume();
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return 0;
}

/** 
 * Sets the player into Full Screen mode..
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - Boolean:  True - FullScreen, False - Normal
 */
function dashSetFullScreen( id, full ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          dashObj.setFullScreen( full );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Maximizes the player by getting rid of the playlist.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - Boolean:  True - Maximize, False - Minimize
 * @param - Boolean:  Used to indicate if you want the transition to be tweened.
 */
function dashSetMaximize( id, max, tween ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          dashObj.setMaximize( max, tween );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Maximizes the player by getting rid of the playlist.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - Boolean:  True - Show Menu, False - Hide Menu
 * @param - Boolean:  Used to indicate if you want the transition to be tweened.
 */
function dashSetMenu( id, menu, tween ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {    
          dashObj.setMenu( menu, tween );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Sees if the player has already loaded a node.
 *
 * @param - The ID of the player you would like to send this request too.
 *
 * @return - Boolean (True if is has loaded content, False otherwise)
 */
function dashIsNodeLoaded( id ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {
         return (dashObj.isNodeLoaded());
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Loads a playlist.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The name of the playlist (view) you would like to load.
 */
function dashLoadPlaylist( id, playlist ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {      
          dashObj.loadPlaylist( playlist );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Loads the previous item in the playlist.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - Indicate if you would like for the player to loop to the end of the list if it is already
 *          on the first item.
 * @param - Indicate if you would like for the playlist to play the file after it loads it.
 */
function dashLoadPrev( id, loop, playAfter ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {      
          dashObj.loadPrev( loop, playAfter );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Loads the next item in the playlist.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - Indicate if you would like for the player to loop to the beginning of the list if it is already
 *          on the last item.
 * @param - Indicate if you would like for the playlist to play the file after it loads it.
 */
function dashLoadNext( id, loop, playAfter ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.loadNext( loop, playAfter );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Loads the previous page in the playlist.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - Indicate if you would like for the player to loop to the last page if it is already
 *          on the first page.
 */
function dashPrevPage( id, loop ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {      
          dashObj.prevPage( loop );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Loads the next page in the playlist.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - Indicate if you would like for the player to loop to the beginning page if it is already
 *          on the last page.
 */
function dashNextPage( id, loop ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.nextPage( loop );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Sets the filter argument for the playlist currently loaded.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The argument that you would like to pass to your playlist to filter the content.
 * @param - The index of the argument.
 */
function dashSetFilter( id, argument, index ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.setFilter( argument, index );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

function dashDebug( arg ) {
   //alert( arg.toString() );
   $('#dashdebug').append( arg + "<br/>" );
}

/** 
 * Resets the controls.
 *
 * @param - The ID of the player you would like to send this request too.
 */
function dashResetControls( id ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.resetControls();
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Enables/Disables the controls.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - true - Enable the controls :  false - Disable the controls.
 */
function dashEnableControls( id, enable ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.enableControls(enable);
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Sets the state of the controls.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The state you would like to set this control bar too.
 *             "play"  - Play State
 *             "pause" - Pause State
 */
function dashSetControlState( id, state ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.setControlState(state);
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Sets the total time print out of the controls.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The time that you would like to set on the controls.
 */
function dashSetControlTime( id, time ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.setControlTime(time);
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Sets the volume of the controls.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The volume that you would like to set on the controls.
 */
function dashSetControlVolume( id, volume ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.setControlVolume(volume);
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Sets the progress indication of the controls.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The progress that you would like to set on the controls.
 */
function dashSetControlProgress( id, progress ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.setControlProgress(progress);
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Sets the seek indication of the controls.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The seek that you would like to set on the controls.
 */
function dashSetControlSeek( id, seek ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.setControlSeek(seek);
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Updates the controls given the play time and total time..
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The playhead time.
 * @param - The total time of the media being played.
 */
function dashControlUpdate( id, playTime, totalTime ) {
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.controlUpdate(playTime, totalTime);
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Calls a Dash Media Player service routine.  The player must have the external service
 * flag set to true in order for this to work properly.
 *
 * @param - The ID of the player you would like to send this request too.
 * @param - The command that you would like to send to the service.
 * @param - The callback that will get called with the response.
 */

function dashServiceCall() {
   var id = arguments[0];
   arguments.shift();
   var command = arguments[0];
   arguments.shift();
   dashCallback = arguments[0];
   arguments.shift();
   
   try {
      var dashObj = getDashObject( id );
      if( dashObj ) {         
          dashObj.serviceCall( command, arguments );
          return true;
      }
   }  
   catch(e) {
      alert( e );
   }
   
   return false;
}

/** 
 * Return from the service call.
 *
 * @param - An array of the arguments that gets passed back.
 */

function dashServiceReturn( args ) {
   dashCallback( args );   
}

/** 
 * Gets the Dash Player Object given an ID
 *
 * @param - The ID of the player you would like to send this request too.
 */
function getDashObject( id ){
   var dashObj = null;
	
   if( dashReady ) {   
      if(navigator.appName.indexOf("Microsoft") != -1) {
         dashObj = window[id];
      }
      else {
         if(document[id].length != undefined) {
            dashObj = document[id][1];
         }
         else {
            dashObj = document[id];
         }
      }
   }
   else {
      alert( id + " not found" );
   }

	return dashObj;
}