<?
/*
	Contacular - a super simple contact form code base for web developers
	Copyright (C) 2009-2010 Jordan Hall

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
	
	*** Contacular 0.14 'human' ***
*/

// *** Class Declaration ***

class ContacularForm
{
	// *** Member Variable Declaration ***

	private $attribution = true;
	private $advancedValidation = true;
	private $fields = array();
	private $recipients = array();
	private $seperator;
	private $errors = array();
	private $recaptchaPublicKey = null;
	private $recaptchaPrivateKey = null;
	
	// *** Public Member Functions ***
	
	/* 
	Function: Constructor
	Purpose: Called upon instantiated to perform initial setup of the form fields if a specific '$type' is passed.
	Arguments:
		$type - Determines the type of form to generate
		$width - Overrides default width of form fields (in pixels) - useful for making form fit (in sidebars for example)
	*/	
	public function __construct($type=false, $width=null)
	{
		if ($type) $this->setup($type, $width);
	}
	
	/* 
	Function: addRecipient
	Purpose: Used to add a new e-mail address to the list of e-mail recipients associated with this form
	Arguments:
		$recipient - E-mail address recipient to associate with this form
	*/
	public function addRecipient($recipient)
	{
		$this->recipients[] = $recipient;
	}
	
	/* 
	Function: setSeperator
	Purpose: Used to set the character or string which seperates the form field names from the data entry elements
	Arguments:
		$seperator - Character or string used to seperate form field names from data entry elements (e.g. ':', ' -' or ':-')
	*/
	public function setSeperator($seperator)
	{
		$this->seperator = $seperator;
	}
	
	/* 
	Function: setAttribution
	Purpose: Sets whether or not the attribution text, 'Powered by Contacular', is displayed on generated forms
	Recommendation: Leaving this text displayed gives credit to the original author, and helps promote Contacular
	Arguments:
		$attribution - Boolean used to determine whether or not the 'Powered by Contacular' text is displayed
	*/
	public function setAttribution($attribution = true)
	{
		$this->attribution = $attribution;
	}
	
	/* 
	Function: setRecaptchaPublicKey
	Purpose: Sets the Recaptcha public key obtained from recaptcha.net and enables Recaptcha checking
	Arguments:
		$recaptchaPublicKey - Recaptcha public key obtained from recaptcha.net
	*/
	public function setRecaptchaPublicKey($recaptchaPublicKey)
	{
		$this->recaptchaPublicKey = $recaptchaPublicKey;
	}
	
	/* 
	Function: setRecaptchaPrivateKey
	Purpose: Sets the Recaptcha public key obtained from recaptcha.net and enables Recaptcha checking
	Arguments:
		$recaptchaPublicKey - Recaptcha public key obtained from recaptcha.net
	*/
	public function setrecaptchaPrivateKey($recaptchaPrivateKey)
	{
		$this->recaptchaPrivateKey = $recaptchaPrivateKey;
	}
	
	/* 
	Function: setAdvancedValidation
	Purpose: Sets whether or not advanced validation checking is performed (such as DNS A record checking on e-mail address validation)
	Recommendation: Enabled by default (recommended) to ensure maximum validation, but could potentially cause false positives.
	Arguments:
		$advancedValidation - Boolean used to determine whether or not advanced validation techniques are used
	*/
	public function setAdvancedValidation($advancedValidation = true)
	{
		$this->advancedValidation = $advancedValidation;
	}
	
	/* 
	Function: getCode
	Purpose: Generate the defined form HTML code and return it for output or other use
	Returns: String - Generated HTML code for form
	Arguments:
		None
	*/
	public function getCode()
	{
		// Check sanity of ContacularForm object
		if ($sanity = $this->sanityCheck()) return $sanity;
		
		// Generate HTML code
		$url = $this->getURL();
		$code = "\n<!-- Contactular Form Start (http://contacular.co.uk/) -->\n";
		$code .= "<form method=\"post\" action=\"".$url."\" name=\"contactularform\">";
		$code .= "<table>";
		foreach ($this->fields as $field)
		{
			$code .= "<tr>";
			$code .= "<td><label for=\"".$field['name']."\">".$field['label'].$this->seperator." </label></td>";
			$code .= "<td>";
			$code .= $this->getFormFieldText($field);
			$code .= "</td>";
			$code .= "</tr>";
		}
		if ($this->recaptchaPublicKey && $this->recaptchaPrivateKey)
		{
			require_once('recaptchalib.php');
			$code .= "<tr><td></td><td>";
			$code .= recaptcha_get_html($this->recaptchaPublicKey);
			$code .="</td></tr>";
		}
		$code .= "<tr><td></td><td><input style=\"font-family: inherit;\" type=\"submit\" name=\"contactularform_submit\" value=\"Send\" /> ".$this->getAttributionText()."</td></tr>";
		$code .= "</table>";
		$code .= "</form>";
		$code .= "\n<!-- Contactular Form End -->\n";
		return $code;
	}
	
	/* 
	Function: processResponse
	Purpose: Processes the response from the form POST and deals with any necessary validation and output such as e-mailing out
	Returns: Boolean - true if form POST has been handled by this function or false if no contacularform submission was detected
	Arguments:
		$post - The $_POST PHP super global array from the page to which the form values are sent, usually '$_POST'
	*/
	public function processResponse($post)
	{
		if (!$post['contactularform_submit'])
		{
			// If no contacularform_submit was posted then the post did not come from a contacular form submission, thus we have nothing to do.
			return false;
		}
		
		// Check sanity of ContacularForm object
		if ($this->sanityCheck()) return false;
		
		// Perform validation
		foreach ($this->fields as $field)
		{
			if ($field['type']=="email")
			{
				if (!$this->validateEmail($post[$field['name']]))
				{
					$this->addError("This e-mail address does not appear to be valid.");
				}
			}
			if ($field['type']=="mandatorytext" || $field['type']=="mandatorytextarea")
			{
				if (!trim($post[$field['name']]))
				{
					$this->addError("The field '".$field['label']."' is required.");
				}
			}
		}
		if ($this->recaptchaPublicKey && $this->recaptchaPrivateKey)
		{
			require_once('recaptchalib.php');
			$resp = recaptcha_check_answer ($this->recaptchaPrivateKey, $_SERVER["REMOTE_ADDR"], $_POST["recaptcha_challenge_field"], $_POST["recaptcha_response_field"]);
			if (!$resp->is_valid) 
			{
				$this->addError("The reCAPTCHA was not entered correctly.");
			}
		}
		if (count($this->errors)!=0) return false;
		
		// Build e-mail content
		$subject = "Response from Contacular Form at ".$_SERVER['SERVER_NAME'];
		$body = "Details from the form are shown below.\n\n";
		foreach ($this->fields as $field)
		{
			if (!$post[$field['name']] && $field['type']=="checkbox") $post[$field['name']] = "Not ticked";
			$body .= $field['label'].": ".stripslashes($post[$field['name']])."\n";
		}
		$from = "From: ".$_SERVER['SERVER_NAME']." <".$this->getFromEmail().">";
		
		// Send out e-mail(s)
		foreach ($this->recipients as $recipient)
		{
			mail($recipient, $subject, $body, $from);
		}
		return true;
	}
	
	/* 
	Function: getErrors
	Purpose: After calling processResponse, this function can be called to retrieve any validation errors
	Returns: String - a list of validation errors (if any) seperated by a <br/> tag
	Arguments:
		None
	*/
	public function getErrors()
	{
		foreach ($this->errors as $error)
		{
			$output .= $error."<br/>";
		}
		return $output;
	}
	
	// *** Private Member Functions ***
	
	private function setup($type, $width=null)
	{
		switch($type)
		{
			case "simple":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("message", "Message", "mandatorytextarea", 100, $width);
				break;
				
			case "simplesubject":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("subject", "Subject", "mandatorytext", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("message", "Message", "mandatorytextarea", 100, $width);
				break;
				
			case "simpleresponse":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("message", "Message", "mandatorytextarea", 100, $width);
				$this->addField("response_desired", "Response desired", "checkbox");
				break;
		
			case "callback":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("from_telephone", "Telephone", "mandatorytext", $width);
				break;
				
			case "enquiry":
				$this->addField("from_title", "Title", "title", null, 70);
				$this->addField("from_firstname", "First name", "mandatorytext", null, $width);
				$this->addField("from_lastname", "Surname", "mandatorytext", null, $width);
				$this->addField("from_telephone", "Telephone", "text", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("enquiry", "Enquiry", "mandatorytextarea", 100, $width);
				break;
				
			case "cataloguerequest":
				$this->addField("from_title", "Title", "title", null, 70);
				$this->addField("from_firstname", "First name", "mandatorytext", null, $width);
				$this->addField("from_lastname", "Surname", "mandatorytext", null, $width);
				$this->addField("from_telephone", "Telephone", "text", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("addressline1", "Address Line 1", "mandatorytext", null, $width);
				$this->addField("addressline2", "Address Line 2", "text", null, $width);
				$this->addField("addressline3", "Address Line 3", "text", null, $width);
				$this->addField("city", "City", "mandatorytext", null, $width);
				$this->addField("county", "County / State", "mandatorytext", null, $width);
				$this->addField("postcode", "Post / ZIP Code", "mandatorytext", null, $width);
				$this->addField("country", "Country", "mandatorytext", null, $width);
				break;
				
			case "contact":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("from_telephone", "Telephone", "text", null, $width);
				$this->addField("message", "Message", "mandatorytextarea", 100, $width);
				break;
				
			case "contactresponse":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("from_telephone", "Telephone", "text", null, $width);
				$this->addField("message", "Message", "mandatorytextarea", 100, $width);
				$this->addField("response_desired", "Response desired", "checkbox");
				break;
				
			case "companycontact":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("from_company", "Company", "text", null, $width);
				$this->addField("from_telephone", "Telephone", "text", null, $width);
				$this->addField("message", "Message", "mandatorytextarea", 100, $width);
				break;
				
			case "companycontactreferrer":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("from_company", "Company", "text", null, $width);
				$this->addField("from_telephone", "Telephone", "text", null, $width);
				$this->addField("referrer", "How did you find us?", "referrer", null, $width);
				$this->addField("referrer_details", "Details on how you found us", "text", null, $width);
				$this->addField("message", "Message", "mandatorytextarea", 100, $width);
				break;
				
			case "comment":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("from_website", "Website", "text", null, $width);
				$this->addField("message", "Message", "mandatorytextarea", 100, $width);
				break;
				
			case "development":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("from_subject", "Subject", "subject_development", null, $width);
				$this->addField("message", "Message", "mandatorytextarea", 100, $width);
				break;
				
			case "simplenewsletter":
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("subscribe_to_newsletter", "Subscribe to newsletter", "checkbox");
				break;
				
			case "newsletter":
				$this->addField("from_name", "Name", "mandatorytext", null, $width);
				$this->addField("from_email", "E-mail", "email", null, $width);
				$this->addField("subscribe_to_newsletter", "Subscribe to newsletter", "checkbox");
				break;
				
			default:
				echo "Contacular error: Supplied form type '".$type."' is not valid.";
		}
	}
	
	private function addField($name, $label, $type="text", $height=null, $width=null)
	{
		if (!$height) $height = 25;
		if (!$width) $width = 250;
		$newField = &$this->fields[];
		$newField['name'] = $name;
		$newField['label'] = $label;
		$newField['type'] = $type;
		$newField['width'] = $width;
		$newField['height'] = $height;
	}
	
	private function getFormFieldText($field)
	{
		switch ($field['type'])
		{
			case "textarea":
			case "mandatorytextarea":
				return "<textarea style=\"width: ".$field['width']."px; height: ".$field['height']."px;\" name=\"".$field['name']."\" id=\"".$field['name']."\" ></textarea>";
				break;
				
			case "checkbox":
				return "<input name=\"".$field['name']."\" id=\"".$field['name']."\" type=\"checkbox\" value=\"Ticked\" />";
				
			case "title":
				return $this->getSelectFormFieldText($field, array("Mr", "Mrs", "Miss", "Dr"));
				break;
				
			case "subject_development":
				return $this->getSelectFormFieldText($field, array("General", "Bug Report", "Feature Request"));
				break;
				
			case "referrer":
				return $this->getSelectFormFieldText($field, array("Not sure / Do not wish to say", "Link from another site", "Search engine", "Recommended by a friend", "E-mail campaign", "Advert (Internet)", "Advert (Paper-based)", "Other"));
				break;
				
			case "email":
				return "<input style=\" width: ".$field['width']."px; height: ".$field['height']."px;\" type=\"text\" name=\"".$field['name']."\" id=\"".$field['name']."\" />";
				break;
			
			case "mandatorytext":
				return "<input style=\" width: ".$field['width']."px; height: ".$field['height']."px;\" type=\"text\" name=\"".$field['name']."\" id=\"".$field['name']."\" />";
				break;
				
			default:
				return "<input style=\" width: ".$field['width']."px; height: ".$field['height']."px;\" type=\"".$field['type']."\" name=\"".$field['name']."\" id=\"".$field['name']."\" />";
				break;
		}
	}
	
	private function getSelectFormFieldText($field, $options)
	{
		$code = "<select style=\"width: ".$field['width']."px; height: ".$field['height']."px;\" name=\"".$field['name']."\" id=\"".$field['name']."\">";
		foreach ($options as $option)
		{
			$code .= "<option value=\"".$option."\">".$option."</option>";
		}
		$code .= "</select>";
		return $code;
	}
	
	private function getAttributionText()
	{
		if ($this->attribution)
		{
			return "<span style=\"font-size: 75%;\">Powered by <a href=\"http://contacular.co.uk/\" target=\"_blank\" title=\"Contacular contact form\">Contacular</a></span>";
		}
		return;
	}
	
	private function getFromEmail()
	{
		return "contacularbot@".$_SERVER['SERVER_NAME'];
	}
	
	private function sanityCheck()
	{
		if (!$this->fields) return "Contacular error: No field(s) were defined.";
		if (!$this->recipients) return "Contacular error: No recipient(s) were defined.";
	}
	
	private function validateEmail($email)
	{
		$pattern = '/^([a-z0-9])(([-a-z0-9._])*([a-z0-9]))*\@([a-z0-9])' .
'(([a-z0-9-])*([a-z0-9]))+' . '(\.([a-z0-9])([-a-z0-9_-])?([a-z0-9])+)+$/i';
		if (!preg_match ($pattern, $email)) 
		{
			return false;
		}
		else
		{
			if ($this->advancedValidation)
			{
				$email_parts = explode("@", $email);
				$domain = $email_parts[1];
				return checkdnsrr($domain, "A");
			}
			else
			{
				return true;
			}
		}
	}
	
	private function addError($text)
	{
		$this->errors[] = $text;
	}
	
	private function getURL()
	{
		$url = 'http';
		if ($_SERVER["HTTPS"] == "on") 
		{
			$url .= "s";
		}
		$url .= "://";
		if ($_SERVER["SERVER_PORT"] != "80") 
		{
			$url .= $_SERVER["SERVER_NAME"].":".$_SERVER["SERVER_PORT"].$_SERVER["REQUEST_URI"];
		}
		else
		{
			$url .= $_SERVER["SERVER_NAME"].$_SERVER["REQUEST_URI"];
		}
		return $url;
	}
	
}

?>
