<?php
// ZebraFeeds - copyright (c) 2006 Laurent Cazalet
// http://www.cazalet.org/zebrafeeds
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// ZebraFeeds feed class

/* created from a single channel or a list of channel

if multiple channels:
for each channel

load feed
aggregate in items array (sorting by date)
set title and description 

if single channel
load feed
update title and description


is an exact replacement for a magpie RSS object
except it contains severa feeds aggregated
and channel is all virtual
it merges a bunch of MagpieRSS objects into one,
sorting all items by date
*/


class feed {


    // aggregated, normalized items
    // if we aggregate several feeds, index is timestamp
    var $items;

    var $image;
    var $channel;

    // merging/filter options
    var $trimtype;
    var $trimsize;
    var $matchExpression;
    var $userFunction;

    // timestamp before which we don't want news
    var $_earliest;
    var $isVirtual;

    //function feed($trimtype = 'none', $trimsize = 0, $match = '') {
        function feed(&$options, &$rss) {
            $this->items = array();
            $this->image = array();
            $this->channel = array();
            $this->_earliest = 0;

            $this->trimtype = $options['trimtype'];
            $this->trimsize = $options['trimsize'];
            $this->matchExpression = $options['match'];
            $this->userFunction  = $options['userfunction'];

            // get timestamp we don't want to go further
            if ($this->trimtype == 'hours') {
                // earliest is the timestamp before which we should ignore news
                $this->_earliest = time() - (3600 * $this->trimsize);
            }
            if ($this->trimtype =='days') {
                // earliest is the timestamp before which we should ignore news

                // get timestamp of today at 0h00
                $todayts = strtotime(date("F j, Y"));

                // substract x-1 times 3600*24 seconds from that
                // x-1 because the current day counts, in the last x days
                $this->_earliest = $todayts -  (3600*24*($this->trimsize-1));
            }       

            if ($rss != null) {
                $this->channel = &$rss->channel;
                $this->channel['image'] = &$rss->image;
                $this->channel['isvirtual'] = false;
                $this->isVirtual = false;

                // integrate this RSS object's items
                $this->mergeItems($rss);
            } else {
                /* we need: title, description, xmlurl, id, 
                * but for now, just init this */
                $this->channel['title'] = 'ZebraFeeds aggregated channels';
                $this->channel['isvirtual'] = true;
                $this->channel['last_fetched'] = 0;
                $this->isVirtual = true;

            }


        }    



        /* function to call after all RSS have been merged
        in order to finalize processing, like sorting and trimming */
        function aggregate($sort = true) {
            if ($sort) {
                $this->sortItems();
            }
            $this->trimItems();
        }


        function mergeWith( &$rss ) {
            $this->touch($rss);
            $this->mergeItems($rss);
        }
        
        /* merge the news items from the RSS object into our list of items
        but before, do some stuff, like
        - normalize items
        - keep only the ones we want
        - add additional data to items
        */
        function mergeItems(&$rss) {

            foreach ($rss->items as $item) {

                $comparedate = (isset($item['date_timestamp'])) ? $item['date_timestamp']: 0;
                $basetime = time();
                // optionally exclude news with date in future
                if (ZF_NOFUTURE == 'yes') {
                    if ($comparedate > $basetime ) {
                        if (ZF_DEBUG==4) {
                            zf_debug('News item \"'.$item['title'].'\" has future date. Skipped.'.$basetime.' lower than '.$comparedate);
                        }
                        continue;
                    }
                }

                if ($this->trimtype == 'hours' || $this->trimtype =='days') {
                    // consider onlyrecent items
                    if (ZF_DEBUG==4) {
                        zf_debug( "comparing item date ".date("F j, Y, g:i a",$comparedate)."(".$comparedate.") to earliest wanted ". $this->_earliest ." : ".date("F j, Y, g:i a",$this->_earliest));
                    }

                    if ( $comparedate >= $this->_earliest) {
                        if (ZF_DEBUG==4) {
                            zf_debug( 'News item within time frame');
                        }

                    } else {
                        if (ZF_DEBUG==4) {
                            zf_debug( 'News item outside time frame');
                        }
                        continue;

                    }
                } else {
                    // no particular trim
                    if (ZF_DEBUG==4) {
                        zf_debug( 'Item accepted: '.$item['title'].' accepted, no time frame');
                    }
                }

                // last check: do we need to match, and does this news match
                if ( strlen($this->matchExpression) > 0) {

                    if (!$this->itemMatches($item)) {
                        if (ZF_DEBUG==4) {
                            zf_debug( 'News item DOES NOT match. rejected');
                        }
                        continue;
                    } else {
                        if (ZF_DEBUG==4) {
                            zf_debug( 'News item match');
                        }
                    }
                }

                if (strlen($this->userFunction) > 0) {
                    call_user_func($this->userFunction, array(&$item));
                    if ($item['discarded']) {
                        // finally add our item to the news array
                        if (ZF_DEBUG==4) {
                            zf_debug('Item discarded by user function: \"'.$item['title'].'\"');
                        }
                        continue;
                    }
                }
                // finally add our item to the news array
                if (ZF_DEBUG==4) {
                    zf_debug('Item integrated: \"'.$item['title'].'\" (trimtype: '.$this->trimtype.')');
                }
                $this->items[] = $item;

            }// foreach item of feed


        }

        // retain the most recent fetch date of all feeds integrated in this one
        function touch(&$rss) {
            if ($this->channel['isvirtual']){

                // if the feed we are merging into is more recent, the date of this feed is touched
                if ($this->channel['last_fetched'] < $rss->channel['last_fetched']) {
                    $this->channel['last_fetched'] = $rss->channel['last_fetched'];
                }
            }
        }

        /* matching function. check an expression against 
        title and description of an item
        defaut: case insensitive keyword match
        could be regexp
        return true if match
        */
        function itemMatches(&$item) {
            $subject = strip_tags($item['title']) . ' ' .strip_tags($item['description']);
            //echo "checking ".$item['title']." for ". $exp.":".strpos(strtolower($subject), strtolower($exp))."<br/>";
            return !(strpos(strtolower($subject), strtolower($this->matchExpression))===false); 
        }


        /* get rid of superfluous items exceeding our limit */
        function trimItems() {
            // trim items by number
            if ($this->trimtype == 'news') {
                if (ZF_DEBUG==4) {
                    zf_debug( 'trimming to '.$this->trimsize.' news');
                }
                $this->items = array_slice($this->items, 0, $this->trimsize);
            }
        }


        /* sort our aggregated items */
        function sortItems() {
            if (ZF_DEBUG==4) {
                zf_debug('sorting items');
            }

            /* sort by timestamp */
            usort($this->items, 'zf_compareItemsDate');

        }




    }


    /* compare the date of two news items
    used as callback in the sorting items call*/
    function zf_compareItemsDate($a, $b) {
        return $a["date_timestamp"] < $b["date_timestamp"];

    }


?>
