<?php

/**
 * This file defines all functions related to managing form submissions.
 *
 * @copyright Benjamin Keen 2007
 * @author Benjamin Keen <ben.keen@gmail.com>
 * @package 1-5-1
 * @subpackage Submissions
 */


// -------------------------------------------------------------------------------------------------


/**
 * Retrieves everything about a form submission for use in a display / edit form submission page.
 *
 * @param integer $form_id The unique form ID.
 * @param integer $submission_id The unique submission ID.
 * @return array Returns an array of hashes. Each index is a separate form field and it's value is
 *           a hash of information about it, such as value, field type, field size, etc.
 */
function ft_get_submission($form_id, $submission_id)
{
	global $g_table_prefix;

	$link = ft_db_connect();

	// get the form template info
	$template_info = mysql_query("
		 SELECT *
		 FROM   {$g_table_prefix}form_templates
		 WHERE  form_id = $form_id
		 ORDER BY list_order
							");

	// get the form submission info
	$submission_info = mysql_query("
		 SELECT *
		 FROM   {$g_table_prefix}form_{$form_id}
		 WHERE  submission_id = $submission_id
							");

	$submission = mysql_fetch_assoc($submission_info);

	// combine the two into an (ordered) array of hashes
	$return_arr = array();
	while ($info = mysql_fetch_assoc($template_info))
	{
		// if the submission contains contents for this field, add it
		if (array_key_exists($info['col_name'], $submission))
			$info["content"] = $submission[$info['col_name']];

		$return_arr[] = $info;
	}

	ft_db_disconnect($link);

	return $return_arr;
}


/**
 * Retrieves ONLY the submission data itself. If you require "meta" information about the submision
 * such as it's field type, size, database table name etc, use {@link
 * http://www.formtools.org/developerdoc/1-4-6/Submissions/_code---submissions.php.html#functionget_submission
 * get_submission}.
 *
 * @param integer $form_id The unique form ID.
 * @param integer $submission_id The unique submission ID.
 * @return array Returns a hash of submission information.
 */
function ft_get_submission_info($form_id, $submission_id)
{
	global $g_table_prefix;

	$link = ft_db_connect();

	// get the form submission info
	$submission_info = mysql_query("
		 SELECT *
		 FROM   {$g_table_prefix}form_{$form_id}
		 WHERE  submission_id = $submission_id
							");

	$submission = mysql_fetch_assoc($submission_info);

	ft_db_disconnect($link);

	return $submission;
}


/**
 * Gets the number of submissions made through a form.
 *
 * @param integer $form_id The unique form ID.
 * @return integer The number of submissions (finalized).
 */
function ft_get_submission_count($form_id)
{
	global $g_table_prefix;

	$link = ft_db_connect();

	// get the form submission info
	$query = mysql_query("
		 SELECT count(*)
		 FROM   {$g_table_prefix}form_{$form_id}
		 WHERE  is_finalized = 'yes'
							");

	$result = mysql_fetch_array($query);
	$submission_count = $result[0];

	ft_db_disconnect($link);

	return $submission_count;
}


/**
 * Updates an individual form submission.
 *
 * This updates all field types, including files. Note: it does not DELETE files - that's handled
 * separately by {@link http://www.formtools.org/developerdoc/1-4-6/Submissions/_code---submissions.php.html#functiondelete_file_submission
 * delete_file_submission}.
 *
 * @param array $infohash This parameter should be a hash (e.g. $_POST or $_GET) containing the
 *             various fields from the update submission page. The contents of it change for each
 *             form content.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_update_submission($infohash)
{
	global $g_table_prefix, $g_multi_val_delimiter, $LANG;

	$success = true;
	$message = $LANG["notify_form_submission_updated"];

	// check all required fields were entered
	if (empty($infohash['form_id']) || empty($infohash['submission_id']))
	{
		// add notification message
		$success = false;
		$message = $LANG["notify_no_form_or_submission_id"];
		return array($success, $message);
	}

	$infohash = ft_clean_hash($infohash);


	$form_id       = $infohash['form_id'];
	$submission_id = $infohash['submission_id'];

	$form_template = ft_get_form_template($form_id);
	$db_column_names = array();

	$query = array();
	$file_fields = array();
	while ($row = mysql_fetch_assoc($form_template))
	{
		// don't try to update system fields
		if ($row["field_type"] == "system")
			continue;

		// keep track of the file fields & their IDs. These will be used to upload the files, if need be
		if ($row['field_type'] == "file")
			$file_fields[$row['col_name']] = $row['field_id'];
		else
		{
			if (isset($infohash[$row["col_name"]]))
			{
				if (is_array($infohash[$row["col_name"]]))
					$query[] = $row["col_name"] . " = '" . join("$g_multi_val_delimiter", $infohash[$row["col_name"]]) . "'";
				else
					$query[] = $row["col_name"] . " = '" . $infohash[$row["col_name"]] . "'";
			}
			else
				$query[] = $row["col_name"] . " = ''";
		}
	}
	$set_query = join(",\n", $query);


	$query = "
		UPDATE {$g_table_prefix}form_{$form_id}
		SET    $set_query
		WHERE  submission_id = $submission_id
					 ";

	$link = ft_db_connect();
	$result = mysql_query($query);
	ft_db_disconnect($link);

	if (!$result)
		return array(false, $LANG["notify_submission_not_updated"]);


	// now the submission exists in the database, upload any files
	if (!empty($file_fields))
	{
		$problem_files = array();

		while (list($key, $fileinfo) = each($_FILES))
		{
			// if nothing was included in this field, just ignore it
			if (empty($fileinfo['name']))
				continue;

			if (array_key_exists("$key", $file_fields))
			{
				list ($success2, $message2) = ft_upload_file($form_id, $submission_id, $file_fields[$key], $fileinfo);
				if (!$success2)
					$problem_files[] = array($fileinfo['name'], $message2);
			}
		}

		if (!empty($problem_files))
		{
			$message = $LANG["notify_submission_updated_file_problems"] . "<br /><br />";
			foreach ($problem_files as $problem)
				$message .= "&bull; <b>{$problem[0]}</b>: $problem[1]<br />\n";

			return array(false, $message);
		}
	}

	return array($success, $message);
}


/**
 * Deletes a form submission.
 *
 * If required, deletes any files that were uploaded along with the original submission. If one or
 * more files associated with this submission couldn't be deleted (either because they didn't exist
 * or because they didn't have permissions) the submission IS deleted, but it returns an error
 * indicating which files caused problems.
 *
 * @param integer $form_id The unique form ID.
 * @param mixed $delete_ids A single submission ID / an array of submission IDs / "all". This column
 *               determines which submissions will be deleted.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_delete_submission($form_id, $delete_ids)
{
	global $g_table_prefix, $LANG;

	$form_info = ft_get_form($form_id);
	$auto_delete_submission_files = $form_info['auto_delete_submission_files'];

	// stick the submission ID / IDs into an array
	if (!is_array($delete_ids))
	{
		// if we're deleting all submissions, DON'T add a where clause. That will cause all
		// submissions in this form to be deleted. Also, get a list of all submission IDs to delete;
		// this is used to examine each, one by one, to delete any associated files
		if ($delete_ids == "all")
		{
			// now delete the submission
			$link = ft_db_connect();
			$submission_id_qry = mysql_query("
				SELECT submission_id
				FROM {$g_table_prefix}form_{$form_id}
									");
			ft_db_disconnect($link);

			$ids = array();
			while ($row = mysql_fetch_assoc($submission_id_qry))
				$submission_ids[] = $row["submission_id"];
		}

		// otherwise, delete only those submissions in the list
		else
		{
			$submission_ids = array($delete_ids);
			$where_clause = "WHERE submission_id = $submission_id";
		}
	}
	else
	{
		$submission_ids     = $delete_ids;
		$submission_ids_qry = array();
		for ($i=0; $i<count($submission_ids); $i++)
			$submission_ids_qry[] = "submission_id = {$submission_ids[$i]}";

		$where_clause = "WHERE " . join(" OR ", $submission_ids_qry);
	}

	$file_delete_problems = array();
	$form_template = ft_get_form_template($form_id);
	$form_has_file_field = false;

	// loop the form templates to find out if there are any file fields. If there are - and the user
	// configured it - delete any associated files
	while ($field_info = mysql_fetch_assoc($form_template))
	{
		if ($field_info['field_type'] == "file")
		{
			$form_has_file_field = true;

			// store the filename we're about to delete BEFORE deleting it. The reason being,
			// if the delete_file_submission function can't find the file, it updates the database record
			// (i.e. overwrites the file name with "") and returns a message indicating what happened.
			// If this wasn't done, in the event of a file being removed/renamed by another process, the
			// user could NEVER remove the filename from their interface. This seems the least inelegant
			// solution. By storing the filename here, we can display it to the user to explain what
			// happened.
			if ($auto_delete_submission_files == "no")
				continue;

			for ($i=0; $i<count($submission_ids); $i++)
			{
				$submission_id = $submission_ids[$i];

				$submission_info = ft_get_submission_info($form_id, $submission_id);
				$filename = $submission_info[$field_info['col_name']];

				// if no filename was stored, it was empty - just continue
				if (empty($filename))
					continue;

				list($success, $message) = ft_delete_file_submission($form_id, $submission_id, $field_info['field_id']);
				if (!$success)
					$file_delete_problems[] = array($filename, $message);
			}
		}
	}


	// now delete the submission
	$link = ft_db_connect();
	mysql_query("
		 DELETE FROM {$g_table_prefix}form_{$form_id}
		 $where_clause
							");
	ft_db_disconnect($link);


	if ($auto_delete_submission_files == "yes")
	{
		if (empty($file_delete_problems))
		{
			$success = true;

			if (count($submission_ids) > 1)
				$message = ($form_has_file_field) ? $LANG["notify_submissions_and_files_deleted"] :
					$LANG["notify_submissions_deleted"];
			else
				$message = ($form_has_file_field) ? $LANG["notify_submission_and_files_deleted"] :
					$LANG["notify_submission_deleted"];
		}
		else
		{
			$success = false;

			if (count($submission_ids) > 1)
				$message = $LANG["notify_submissions_deleted_with_problems"] . "<br /><br />";
			else
				$message = $LANG["notify_submission_deleted_with_problems"] . "<br /><br />";

			foreach ($file_delete_problems as $problem)
				$message .= "&bull; <b>{$problem[0]}</b>: $problem[1]<br />\n";
		}
	}
	else
	{
		$success = true;

		if (count($submission_ids) > 1)
			$message = $LANG["notify_submissions_deleted"];
		else
			$message = $LANG["notify_submission_deleted"];
	}


	// update sessions to ensure the first submission date and num submissions are correct
	_ft_load_form_values($form_id);

	return array($success, $message);
}


/**
 * Deletes a file that has been uploaded through a particular form submission file field.
 *
 * Now say that 10 times fast.
 *
 * @param integer $form_id The unique form ID.
 * @param integer $submission_id A unique submission ID.
 * @param integer $field_id A unique form field ID.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_delete_file_submission($form_id, $submission_id, $field_id)
{
	global $g_table_prefix, $LANG;

	// get the column name and upload folder for this field
	$field_info = ft_get_form_field($field_id);
	$col_name    = $field_info['col_name'];
	$file_folder = $field_info['file_upload_dir'];

	// if the column name wasn't found, the $field_id passed in was invalid. Return false.
	if (empty($col_name))
		return array(false, $LANG["notify_submission_no_field_id"]);


	$query = "
		SELECT $col_name
		FROM   {$g_table_prefix}form_{$form_id}
		WHERE  submission_id = $submission_id;
						";

	$link = ft_db_connect();
	$result = mysql_query($query);
	$file_info = mysql_fetch_row($result);
	$file = $file_info[0];

	$update_database_record = false;
	$success = true;
	$message = "";

	if (!empty($file))
	{
		if (@unlink("$file_folder/$file"))
		{
			$success = true;
			$message = $LANG["notify_file_deleted"];
			$update_database_record = true;
		}
		else
		{
			$success = false;
			$message = $LANG["notify_file_not_deleted"];
		}
	}

	// if need be, update the database record to remove the reference to the file in the database. Generally this
	// should always work, but in case something funky happened, like the permissions on the file were changed to
	// forbid deleting, I think it's best if the record doesn't get deleted to remind the admin/client it's still there.
	if ($update_database_record)
	{
		$query = mysql_query("
			UPDATE {$g_table_prefix}form_{$form_id}
			SET    $col_name = ''
			WHERE  submission_id = $submission_id;
						 ");
	}

	ft_db_disconnect($link);

	return array($success, $message);
}


/**
 * Deprecated function. Use ft_finalize_submission instead.
 *
 * @deprecated
 */
function finalize_submission($form_id, $submission_id)
{
	ft_finalize_submission($form_id, $submission_id);
}


/**
 * For use by programmers to finalize a submission (i.e. make it appear in the client's user
 * interface).
 *
 * @param integer $form_id The unique form ID.
 * @param integer $submission_id A unique submission ID.
 * @return boolean $success True on success, false otherwise.
 */
function ft_finalize_submission($form_id, $submission_id)
{
	global $g_table_prefix;

	// check the form_id is valid
	$form_info = ft_get_form($form_id);
	if (empty($form_info))
		return false;

	$query = "
		UPDATE {$g_table_prefix}form_$form_id
		SET    is_finalized = 'yes'
		WHERE  submission_id = $submission_id
					 ";

	$link = ft_db_connect();
	$result = mysql_query($query);
	ft_db_disconnect($link);

	return true;
}


/**
 * Uploads a file for a particular form submission field.
 *
 * This is to be called AFTER the submission has already been added to the database - it uploads the
 * file to the specified folder then updates the database record.
 *
 * @param integer $form_id The unique form ID.
 * @param integer $submission_id A unique submission ID.
 * @param integer $field_id A unique field ID.
 * @param array $fileinfo An index from the $_FILES array (containing all data about the file)
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 *               [2]: If success, the filename of the uploaded file
 */
function ft_upload_file($form_id, $submission_id, $field_id, $fileinfo)
{
	global $g_table_prefix, $LANG;

	// get the column name and upload folder for this field
	$field_info = ft_get_form_field($field_id);
	$col_name    = $field_info['col_name'];
	$file_upload_dir = $field_info['file_upload_dir'];
	$file_upload_url = $field_info['file_upload_url'];
	$file_upload_max_size = $field_info['file_upload_max_size'];
	$file_upload_types = $field_info['file_upload_types'];

	// if the column name wasn't found, the $field_id passed in was invalid. Return false.
	if (empty($col_name))
		return array(false, $LANG["notify_submission_no_field_id"]);

	// convert any whitespace chars in filename to underscores [list may be expanded later]
	$filename     = preg_replace("/\s+/", "_", $fileinfo["name"]);
	$tmp_filename = $fileinfo["tmp_name"];
	$filesize     = $fileinfo["size"];

	// check file size
	if ($filesize > $file_upload_max_size)
		return array(false, $LANG["notify_file_too_large"]);

	// check upload folder is valid and writable
	if (!is_dir($file_upload_dir) || !is_writable($file_upload_dir))
		return array(false, $LANG["notify_invalid_field_upload_folder"]);

	// check file extension is valid. Note: this is "dumb" - it just tests for the file extension, not
	// the actual file type based on it's header info [this is done because I want to allow users to permit
	// uploading of any file types, and I can't know the headers of all of them]
	$is_valid_extension = true;
	if (!empty($file_upload_types))
	{
		$is_valid_extension = false;
		$raw_extensions = explode(",", $file_upload_types);

		foreach ($raw_extensions as $ext)
		{
			// remove whitespace and periods
			$clean_extension = str_replace(".", "", trim($ext));

			if (preg_match("/$clean_extension$/", $filename))
				$is_valid_extension = true;
		}
	}

	// all checks out!
	if ($is_valid_extension)
	{
		// check for duplicate filenames and get a unique name
		$unique_filename = ft_check_duplicate_filename($file_upload_dir, $filename);

		// copy file to uploads folder and remove temporary file
		if (rename($tmp_filename, "$file_upload_dir/$unique_filename"))
		{
			@chmod("$file_upload_dir/$unique_filename", 0777);

			// update the database
			$query = "
				UPDATE {$g_table_prefix}form_{$form_id}
				SET    $col_name = '$unique_filename'
				WHERE  submission_id = $submission_id
							 ";

			$link = ft_db_connect();
			mysql_query($query);
			ft_db_disconnect($link);

			return array(true, $LANG["notify_file_uploaded"], $unique_filename);
		}
		else
			return array(false, $LANG["notify_file_not_uploaded"]);
	}

	// not a valid extension. Inform the user
	else
		return array(false, $LANG["notify_unsupported_file_extension"]);
}


/**
 * Creates and returns a search for any form, and any subset of its columns, returning results in
 * any column order and for any single page subset (or all pages).
 *
 * As of version 1.4.4 this is used for ALL pages that list the submissions - or a subset of them.
 * i.e. print preview, excel download, admin submission listings, client submission listings.
 *
 * @param integer $form_id The unique form ID.
 * @param mixed $results_per_page an integer, or "all".
 * @param integer $page The current page number - or empty string, if this function is returning all
 *              results in one page (e.g. printer friendly page).
 * @param string $order A string of form: "{db column}_{ASC|DESC}"
 * @param mixed $columns An array containing which database columns to return, or a string:
 *              "all" - which returns all columns (e.g. print preview page).
 * @param array $search_fields an optional hash with these keys:<br/>
 *                search_field<br/>
 *                search_date<br/>
 *                search_keyword<br/>
 * @return array Returns a hash with these keys:<br/>
 *                ["search_query"] => resource ID of the MySQL select query<br/>
 *                ["num_results"]  => the TOTAL number of results (not just the X that will appear
 *                                    in the current page, listed in the "search_query" key
 */
function ft_search_submissions($form_id, $results_per_page, $page_num, $order, $columns, $search_fields = array())
{
	global $g_table_prefix;

	// sorting by column, format: col_x-desc / col_y-asc
	list($column, $direction) = split("-", $order);
	$field_info = ft_get_form_field_by_colname($form_id, $column);

	if ($field_info["data_type"] == "number")
		$order_by = "CAST($column as SIGNED) $direction";
	else
		$order_by = "$column $direction";

	// determine the LIMIT clause
	$limit_clause = "";
	if ($results_per_page != "all")
	{
		if (empty($page_num))
			$page_num = 1;
		$first_item = ($page_num - 1) * $results_per_page;

		$limit_clause = "LIMIT $first_item, $results_per_page";
	}

	// determine the SELECT clause
	$select_clause = "";
	if (!is_array($columns) && $columns == "all")
	{
		$select_clause = " * ";
	}
	else
	{
		// if submission_id isn't included, add it - it'll be needed at some point
		if (!in_array("submission_id", $columns))
			$columns[] = "submission_id";

		$select_clause = join(", ", $columns);
	}

	// if there are filters defined, employ the filters for this form
	$filter_clause = "";
	if (!empty($_SESSION["ft"]["form_{$form_id}_filters"]))
		$filter_clause = "AND " . join(" AND ", $_SESSION["ft"]["form_{$form_id}_filters"]);

	// if search fields were included, build an addition to the WHERE clause
	$search_where_clause = "";
	if (!empty($search_fields))
	{
		$clean_search_fields = ft_clean_hash($search_fields);

		$search_field   = $clean_search_fields["search_field"];
		$search_date    = $clean_search_fields["search_date"];
		$search_keyword = $clean_search_fields["search_keyword"];

		// search field can either be "all" or a database column name. "submission_date"
		// has a special meaning in that it allows searching by specific date ranges
		switch ($search_field)
		{
			case "all":
				if (!empty($search_keyword))
				{
					// get all columns
					$col_info = ft_get_form_column_names($form_id);
					$col_names = array_keys($col_info);
					unset($col_names["is_finalized"]);
					unset($col_names["submission_date"]);

					$clauses = array();
					foreach ($col_names as $col_name)
						$clauses[] = "$col_name LIKE '%$search_keyword%'";

					$search_where_clause = "AND (" . join(" OR ", $clauses) . ") ";
				}
				break;

			case "submission_date":
				if (!empty($search_date))
				{
					// search by number of days
					if (is_numeric($search_date))
					{
						$days = $search_date;
						$search_where_clause = "AND (DATE_SUB(curdate(), INTERVAL $days DAY) < submission_date) ";
					}

					// otherwise, return a specific month
					else
					{
						list($month, $year) = split("_", $search_date);

						$month_start = mktime(0, 0, 0, $month, 1, $year);
						$month_end   = mktime(0, 0, 0, $month+1, 1, $year);

						$start = date("Y-m-d", $month_start);
						$end   = date("Y-m-d", $month_end);

						$search_where_clause = "AND (submission_date > '$start' AND submission_date < '$end') ";
					}

					if (!empty($search_keyword))
					{
						// get all columns
						$col_info = ft_get_form_column_names($form_id);
						$col_names = array_keys($col_info);
						unset($col_names["is_finalized"]);
						unset($col_names["submission_date"]);

						$clauses = array();
						foreach ($col_names as $col_name)
							$clauses[] = "$col_name LIKE '%$search_keyword%'";

						$search_where_clause .= "AND (" . join(" OR ", $clauses) . ") ";
					}
				}
				break;

			// here, the user is searching one of their own custom fields
			default:
				if (!empty($search_keyword) && !empty($search_field))
					$search_where_clause = "AND $search_field LIKE '%$search_keyword%'";
				break;
		}
	}

	// now build our query
	$full_query = "
			SELECT $select_clause
			FROM   {$g_table_prefix}form_{$form_id}
			WHERE  is_finalized = 'yes'
						 $search_where_clause
						 $filter_clause
			ORDER BY $order_by
						 $limit_clause
								";
	$count_query = "
			SELECT *
			FROM   {$g_table_prefix}form_{$form_id}
			WHERE  is_finalized = 'yes'
						 $search_where_clause
						 $filter_clause
								 ";

	$link = ft_db_connect();
	$search_query = mysql_query($full_query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$full_query</i>", mysql_error());

	$num_results  = mysql_num_rows(mysql_query($count_query));
	ft_db_disconnect($link);

	$return_hash["search_query"] = $search_query;
	$return_hash["num_results"]  = $num_results;

	return $return_hash;
}


/**
 * Returns all submission IDs in a search result set. This is used on the item details pages (admin
 * and client) to build the << previous / next >> links.
 *
 * @param integer $form_id The unique form ID
 * @param mixed   $results_per_page an integer, or "all"
 * @param string  $order A string of form: "{db column}_{ASC|DESC}"
 * @param array   $search_fields an optional hash with these keys:<br/>
 *                  search_field<br/>
 *                  search_date<br/>
 *                  search_keyword<br/>
 * @return string An HTML string.
 */
function ft_get_search_submission_ids($form_id, $results_per_page, $order, $search_fields = array())
{
	global $g_table_prefix;

	// sorting by column, format: col_x-desc / col_y-asc
	list($column, $direction) = split("-", $order);
	$field_info = ft_get_form_field_by_colname($form_id, $column);

	if ($field_info["data_type"] == "number")
		$order_by = "CAST($column as SIGNED) $direction";
	else
		$order_by = "$column $direction";

	// determine the LIMIT clause
	$limit_clause = "";
	if ($results_per_page != "all")
	{
		if (empty($page_num))
			$page_num = 1;
		$first_item = ($page_num - 1) * $results_per_page;

		$limit_clause = "LIMIT $first_item, $results_per_page";
	}

	// if there are filters defined, employ the filters for this form
	$filter_clause = "";
	if (!empty($_SESSION["ft"]["form_{$form_id}_filters"]))
		$filter_clause = "AND " . join(" AND ", $_SESSION["ft"]["form_{$form_id}_filters"]);

	// if search fields were included, build an addition to the WHERE clause
	$search_where_clause = "";
	if (!empty($search_fields))
	{
		$clean_search_fields = ft_clean_hash($search_fields);

		$search_field   = $clean_search_fields["search_field"];
		$search_date    = $clean_search_fields["search_date"];
		$search_keyword = $clean_search_fields["search_keyword"];

		// search field can either be "all" or a database column name. "submission_date"
		// has a special meaning in that it allows searching by specific date ranges
		switch ($search_field)
		{
			case "all":
				if (!empty($search_keyword))
				{
					// get all columns
					$col_info = ft_get_form_column_names($form_id);
					$col_names = array_keys($col_info);
					unset($col_names["is_finalized"]);
					unset($col_names["submission_date"]);

					$clauses = array();
					foreach ($col_names as $col_name)
						$clauses[] = "$col_name LIKE '%$search_keyword%'";

					$search_where_clause = "AND (" . join(" OR ", $clauses) . ") ";
				}
				break;

			case "submission_date":
				if (!empty($search_date))
				{
					// search by number of days
					if (is_numeric($search_date))
					{
						$days = $search_date;
						$search_where_clause = "AND (DATE_SUB(curdate(), INTERVAL $days DAY) < submission_date) ";
					}

					// otherwise, return a specific month
					else
					{
						list($month, $year) = split("_", $search_date);

						$month_start = mktime(0, 0, 0, $month, 1, $year);
						$month_end   = mktime(0, 0, 0, $month+1, 1, $year);

						$start = date("Y-m-d", $month_start);
						$end   = date("Y-m-d", $month_end);

						$search_where_clause = "AND (submission_date > '$start' AND submission_date < '$end') ";
					}

					if (!empty($search_keyword))
					{
						// get all columns
						$col_info = ft_get_form_column_names($form_id);
						$col_names = array_keys($col_info);
						unset($col_names["is_finalized"]);
						unset($col_names["submission_date"]);

						$clauses = array();
						foreach ($col_names as $col_name)
							$clauses[] = "$col_name LIKE '%$search_keyword%'";

						$search_where_clause .= "AND (" . join(" OR ", $clauses) . ") ";
					}
				}
				break;

			// here, the user is searching one of their own custom fields
			default:
				if (!empty($search_keyword) && !empty($search_field))
					$search_where_clause = "AND $search_field LIKE '%$search_keyword%'";
				break;
		}
	}

	// now build our query
	$full_query = "
			SELECT submission_id
			FROM   {$g_table_prefix}form_{$form_id}
			WHERE  is_finalized = 'yes'
						 $search_where_clause
						 $filter_clause
			ORDER BY $order_by
								";

	$link = ft_db_connect();
	$search_query = mysql_query($full_query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$full_query</i>", mysql_error());

	ft_db_disconnect($link);

	$submission_ids = array();
	while ($row = mysql_fetch_assoc($search_query))
		$submission_ids[] = $row["submission_id"];

	return $submission_ids;
}

