<?php

/**
 * This file defines all general functions used throughout Form Tools.
 *
 * @copyright Benjamin Keen 2007
 * @author Benjamin Keen <ben.keen@gmail.com>
 * @package 1-5-1
 * @subpackage General
 */


// -------------------------------------------------------------------------------------------------


/**
 * Open a database connection.
 *
 * After connecting, you should always call disconnect_db() to close it when you're done.
 *
 * @return resource Returns a reference to the open connection.
 */
function ft_db_connect()
{
	global $g_db_hostname, $g_db_username, $g_db_password, $g_db_name, $g_unicode;

	$link = @mysql_connect($g_db_hostname, $g_db_username, $g_db_password) or
		ft_handle_error("Couldn't connect to database: <b>$g_db_hostname</b>", "");
	@mysql_select_db($g_db_name) or
		ft_handle_error("couldn't find database <b>$g_db_name</b>", "");

	// if required, set all queries as UTF-8
	if ($g_unicode)
		@mysql_query("SET NAMES 'utf8'", $link);

	return $link;
}


/**
 * Closes a database connection.
 *
 * @param resource Closes the connection included in this parameter.
 */
function ft_db_disconnect($link)
{
	@mysql_close($link);
}


/**
 * Retrieves all values from the "settings" table
 *
 * @return array All settings in a hash.
 */
function ft_get_settings()
{
	global $g_table_prefix;

	$link = ft_db_connect();
	$query  = mysql_query("
					SELECT setting_name, setting_value
					FROM   {$g_table_prefix}settings
					");
	$hash = array();
	while ($row = mysql_fetch_assoc($query))
		$hash[$row['setting_name']] = $row['setting_value'];

	ft_db_disconnect($link);

	return $hash;
}


/**
 * Called by the Administrator from the Program Settings page; updates various values in the
 * settings table.
 *
 * @param array $infohash This parameter should be a hash (e.g. $_POST or $_GET) containing the
 *             various fields from the Program Settings admin page.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_update_settings($infohash)
{
	global $g_table_prefix, $g_root_url, $LANG;

	$success = true;
	$message = $LANG["notify_setup_options_updated"];

	$infohash = ft_clean_hash($infohash);

	// validate all fields
	$rules = array();
	$rules[] = "required,program_name,{$LANG["validation_no_program_name"]}";
	$rules[] = "required,footer_text,{$LANG["validation_no_footer_text"]}";
	$rules[] = "required,administrator_email,{$LANG["validation_no_admin_email"]}";
	$rules[] = "valid_email,administrator_email,{$LANG["validation_invalid_admin_email"]}";
	$rules[] = "required,logout_url,{$LANG["validation_no_logout_url"]}";
	$rules[] = "required,default_language,{$LANG["validation_no_default_language"]}";
	$rules[] = "required,date_format,{$LANG["validation_no_date_format"]}";
	$rules[] = "required,num_clients_per_page,{$LANG["validation_no_num_clients_per_page"]}";
	$rules[] = "digits_only,num_clients_per_page,{$LANG["validation_invalid_num_clients_per_page"]}";
	$rules[] = "required,num_forms_per_page,{$LANG["validation_no_num_forms_per_page"]}";
	$rules[] = "digits_only,num_forms_per_page,{$LANG["validation_invalid_num_forms_per_page"]}";
	$rules[] = "required,default_css,{$LANG["validation_no_default_css"]}";
	$errors = validate_fields($infohash, $rules);

	if (!empty($errors))
	{
		$success = false;
		array_walk($errors, create_function('&$el','$el = "&bull;&nbsp; " . $el;'));
		$message = join("<br />", $errors);
		return array ($success, $message, "");
	}

	$link = ft_db_connect();

	$program_name         = trim($infohash['program_name']);
	$footer_text          = $infohash['footer_text'];
	$administrator_email  = trim($infohash['administrator_email']);
	$logout_url           = $infohash['logout_url'];
	$default_language     = $infohash['default_language'];
	$timezone_offset      = $infohash['timezone_offset'];
	$date_format          = $infohash['date_format'];
	$num_clients_per_page = $infohash['num_clients_per_page'];
	$num_forms_per_page   = $infohash['num_forms_per_page'];
	$emailing_method      = $infohash['emailing_method'];
	$default_css          = $infohash['default_css'];

	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$program_name' WHERE setting_name = 'program_name'");
	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$footer_text' WHERE setting_name = 'footer_text'");
	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$administrator_email' WHERE setting_name = 'administrator_email'");
	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$logout_url' WHERE setting_name = 'logout_url'");
	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$default_language' WHERE setting_name = 'default_language'");
	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$timezone_offset' WHERE setting_name = 'timezone_offset'");
	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$date_format' WHERE setting_name = 'date_format'");
	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$num_clients_per_page' WHERE setting_name = 'num_clients_per_page'");
	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$num_forms_per_page' WHERE setting_name = 'num_forms_per_page'");
	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$emailing_method' WHERE setting_name = 'emailing_method'");
	$query = mysql_query("UPDATE {$g_table_prefix}settings SET setting_value = '$default_css' WHERE setting_name = 'default_css'");
	ft_db_disconnect($link);

	// reset the session values
	$_SESSION['ft']['program_name']        = stripslashes($program_name);
	$_SESSION['ft']['footer_text']         = stripslashes($footer_text);
	$_SESSION['ft']['administrator_email'] = $administrator_email;
	$_SESSION['ft']['logout_url']          = $logout_url;
	$_SESSION['ft']['default_css']         = $default_css;

	return array($success, $message);
}


/**
 * A handy, generic function used throughout the site to output messages to the user - the content
 * of which are returned by the various functions. It can handle multiple messages (notifications
 * and/or errors) by passing in arrays for each of the two parameters.
 *
 * Ultimately, one of the goals is to move to complete consistency in the ways the various functions
 * handle their return values. Specifically, an array with the following indexes:<br/>
 *    [0] T/F (or an array of T/F values),<br/>
 *    [1] error/success message string (or an array of strings)<br/>
 *    [2] other information, e.g. new IDs (optional).
 *
 * @param boolean $results This parameter can be EITHER a boolean or an array of booleans if you
 *          need to display multiple messages at once.
 * @param boolean $messages The message to output, or an array of messages. The indexes of each
 *          corresponds to the success/failure boolean in the $results parameter.
 */
function ft_display_message($results, $messages)
{
	global $LANG;

	// if there are no messages, just return
	if (empty($messages))
		return;

	$notifications = array();
	$errors        = array();

	if (is_array($results))
	{
		for ($i=0; $i<=count($results); $i++)
		{
			if     ($results[$i])  $notifications[] = $messages[$i];
			elseif (!$results[$i]) $errors[]        = $messages[$i];
		}
	}
	else
	{
		if     ($results)  $notifications[] = $messages;
		elseif (!$results) $errors[]        = $messages;
	}


	// display notifications
	if (!empty($notifications))
	{
		if (count($notifications) > 1)
		{
			array_walk($notifications, create_function('&$el','$el = "&bull;&nbsp; " . $el;'));
			$display_str = join("<br />", $notifications);
		}
		else
			$display_str = $notifications[0];

		echo "<div class='notify'>$display_str</div><br />";
	}

	// display errors
	if (!empty($errors))
	{
		// if there were notifications displayed, add a little padding to separate the two sections
		if (!empty($notifications)) { echo "<br />"; }

		if (count($errors) > 1)
		{
			array_walk($errors, create_function('&$el','$el = "&bull;&nbsp; " . $el;'));
			$display_str = join("<br />", $errors);
			$title_str = $LANG["word_errors"];
		}
		else
		{
			$display_str = $errors[0];
			$title_str = $LANG["word_error"];
		}

		echo "<div class='error'><span>$title_str</span><br /><br />$display_str</div><br />";
	}
}


/**
 * Handy function to help manage long strings by adding either an ellipsis or inserts a inserts a
 * <br /> at the position specified, and returns the result.
 *
 * @param string $str The string to manipulate.
 * @param string $length The max length of the string / place to insert <br />
 * @param string $flag "ellipsis" / "page_break"
 * @return string The modified string.
 */
function ft_trim_string($str, $length, $flag = "ellipsis")
{
	$new_string = "";
	if (mb_strlen($str) < $length)
		$new_string = $str;
	else
	{
		if ($flag == "ellipsis")
			$new_string = mb_substr($str, 0, $length) . "...";
		else
		{
			$parts = mb_str_split($str, $length);
			$new_string = join("<br />", $parts);
		}
	}

	return $new_string;
}


/**
 * A multibyte version of str_split. Splits a string into chunks and returns the pieces in
 * an array.
 *
 * @param string $string The string to manipulate.
 * @param integer $split_length The number of characters in each chunk.
 * @return array an array of chunks, each of size $split_length. The last index contains the leftovers.
 *      If <b>$split_length</b> is less than 1, return false.
 */
function mb_str_split($string, $split_length = 1)
{
	if ($split_length < 1)
		return false;

	$result = array();
	for ($i=0; $i<mb_strlen($string); $i+=$split_length)
		$result[] = mb_substr($string, $i, $split_length);

	return $result;
}


/**
 * Displays basic &lt;&lt; 1 2 3 >> navigation for lists, each linking to the current page.
 *
 * @param integer $num_results The total number of results found.
 * @param integer $num_per_page The max number of results to list per page.
 * @param integer $current_page The current page number being examined (defaults to 1).
 * @param string $pass_along_str The string to include in nav links.
 */
function ft_display_page_nav($num_results, $num_per_page, $current_page = 1, $pass_along_str)
{
	global $g_max_nav_pages, $LANG;

	// display the total number of results found
	$range_start = ($current_page - 1) * $num_per_page + 1;
	$range_end   = $range_start + $num_per_page - 1;
	$range_end   = ($range_end > $num_results) ? $num_results : $range_end;

	echo "<div>{$LANG["phrase_total_results_c"]} <b>$num_results</b>&nbsp;";

	// if there's more than one page, display a message showing which numbers are being shown
	if ($num_results > $num_per_page)
	{
		$replacement_info = array("startnum" => $range_start, "endnum" => $range_end);
		echo ft_replace_placeholders($LANG["phrase_viewing_range"], $replacement_info);
	}

	// calculate total number of pages
	$total_pages  = ceil($num_results / $num_per_page);

	// piece together additional query string values
	if (!empty($pass_along_str))
		$query_str = "&" . $pass_along_str;

	// determine the first and last pages to show page nav links for
	$half_total_nav_pages  = floor($g_max_nav_pages / 2);
	$first_page = ($current_page > $half_total_nav_pages) ? $current_page - $half_total_nav_pages : 1;
	$last_page  = (($current_page + $half_total_nav_pages) < $total_pages) ? $current_page + $half_total_nav_pages : $total_pages;

	if ($total_pages > 1)
	{
		echo "<br />{$LANG["word_page_c"]} ";

		for ($page=$first_page; $page<=$last_page; $page++)
		{
			// if we're not on the first page, provide a "<<" (previous page) link
			if ($current_page != 1 && $page == $first_page)
				echo "<a href='{$_SERVER['PHP_SELF']}?page=" . ($current_page-1) . "{$query_str}'>&lt;&lt;</a>&nbsp;";

			echo "<a href='{$_SERVER['PHP_SELF']}?page=$page{$query_str}'>";
			if ($page == $current_page)
				echo "<b>$page</b>";
			else
				echo $page;

			echo "</a> ";

			// if required, add a final ">>" (next page) link
			if ($current_page != $total_pages && $page == $last_page)
				echo "<a href='{$_SERVER['SCRIPT_NAME']}?page=" . ($current_page+1) . "{$query_str}'>&gt;&gt;</a>";
		}
	}

	echo "</div><br />";
}


/**
 * Displays basic << 1 2 3 >> navigation for lists, each linking to the current page.
 *
 * This function has exactly the same purpose as display_page_nav, except that the pages are
 * hidden/shown with DHTML instead of separate server-side calls per page. This technique is better
 * for lists that contain a smaller number of items, e.g. client accounts.
 *
 * ASSUMPTION: the JS counterpart function with the same function is defined in the calling page.
 * That function does all the work of hiding/showing pages, updating the "viewing X-Y"
 * text, enabling disabling the << and >> arrows, and storing the current page in sessions. This
 * function merely sets up the base HTML + JS.
 *
 * @param integer $num_results The total number of results found.
 * @param integer $num_per_page The max number of results to list per page.
 * @param integer $current_page The current page number being examined (defaults to 1).
 */
function ft_display_dhtml_page_nav($num_results, $num_per_page, $current_page = 1)
{
	global $LANG;

	// display the total number of results found
	$range_start = ($current_page - 1) * $num_per_page + 1;
	$range_end   = $range_start + $num_per_page - 1;
	$range_end   = ($range_end > $num_results) ? $num_results : $range_end;

	echo "<div>{$LANG["phrase_total_results_c"]} <b>$num_results</b>&nbsp;";

	// if there's more than one page, display a message showing which numbers are being shown
	if ($num_results > $num_per_page)
	{
		$replacement_info = array("startnum" => "<span id='nav_viewing_num_start'>$range_start</span>", "endnum" => "<span id='nav_viewing_num_end'>$range_end</span>");
		echo ft_replace_placeholders($LANG["phrase_viewing_range"], $replacement_info);
	}

	// calculate total number of pages
	$total_pages  = ceil($num_results / $num_per_page);

	if ($total_pages > 1)
	{
		echo "<br />{$LANG["word_page_c"]} ";

		// always show a "<<" (previous page) link. Its contents can be changed with JS
		echo "<span id='nav_previous_page'>";
		if ($current_page != 1)
		{
			$previous_page = $current_page - 1;
			echo "<a href='javascript:display_dhtml_page_nav($num_results, $num_per_page, $previous_page)'>&lt;&lt;</a>";
		}
		else
			echo "&lt;&lt;";
		echo "</span> ";

		for ($page=1; $page<=$total_pages; $page++)
		{
			echo "<span id='nav_page_$page'>";
			if ($page == $current_page)
				echo $page;
			else
				echo "<a href='javascript:display_dhtml_page_nav($num_results, $num_per_page, $page)'>$page</a>";
			echo "</span> ";
		}

		// always show a ">>" (next page) link. Its content is changed with JS
		echo "<span id='nav_next_page'>";
		if ($current_page != $total_pages)
		{
			$next_page = $current_page + 1;
			echo "<a href='javascript:display_dhtml_page_nav($num_results, $num_per_page, $next_page)'>&gt;&gt;</a>";
		}
		else
			echo "<span id='nav_next_page'>&gt;&gt;</span>";
		echo "</span> ";

	}

	echo "</div><br />";
}


/**
 * Provides basic permission checking on files.
 *
 * Verifies the user has permission to view the current page. It is used by feeding the minimum
 * account type to view the page - "client", will let administrators and clients view it, but
 * "admin" will only let administrators. If the person doesn't have permission to view the page
 * they are booted out.
 *
 * Should be called on ALL Form Tools pages.
 *
 * @param string $account_type The account type - "admin" / "client"
 */
function ft_check_permission($account_type)
{
	global $g_root_url;
	$boot_out_user = false;

	// check user_id has been set
	if      (empty($_SESSION['ft']['login_user_id']))
		$boot_out_user = true;
	else if ($_SESSION['ft']['login_account_type'] == "client" && $account_type == "admin")
		$boot_out_user = true;

	if ($boot_out_user)
	{
		header("Location: $g_root_url/logout.php");
		exit;
	}
}


/**
 * Checks that the currently logged in client is permitted to manage a form.
 *
 * This prevents one user trying to view the contents of another form which they have not been assigned
 * to. If this is the case, they are logged out.
 *
 * @param integer $form_id The unique form ID.
 * @param integer $form_id The unique client ID.
 */
function ft_check_may_manage_form($form_id, $client_id)
{
	global $g_root_url;
	$boot_out_user = false;

	// check client_id is included
	if (empty($_SESSION['ft']['login_user_id']))
		$boot_out_user = true;
	else
	{
		$link = ft_db_connect();
		$form_info = ft_get_form($form_id);

		$found = false;
		for ($i=0; $i<count($form_info['user_info']); $i++)
		{
			if ($form_info['user_info'][$i]['user_id'] == $client_id)
				$found = true;
		}

		if (!$found)
			$boot_out_user = true;

		ft_db_disconnect($link);
	}


	if ($boot_out_user)
	{
		header("Location: $g_root_url/logout.php");
		exit;
	}
}


/**
 * Return a date string from a MySQL datetime according based on an offset and a display format.
 * As of version 1.5.0, this function is language localized. The following php date() flags are
 * translated:
 * 			D    - Mon through Sun
 *      l    - Sunday through Saturday
 *      F    - January through December
 *      M    - Jan through Dec
 *      a    - am or pm
 *      A    - AM or PM
 *
 * Note that some flags (S for "st","rd", "nd" etc. and T for timezone, EST, MDT etc) are NOT
 * translated. This is. Also, this function only uses the standard Gregorian calendar. Nothing
 * fancy! My Unicode 5 book in on route, so I'll look into that in a later version. ;-)
 *
 * @param integer $offset the number of hours offset from GMT (- or +)
 * @param string $datetime the mysql datetime to format
 * @param string $format the date format to use (PHP's date() function).
 * @return string the date/time as a fully localized string
 */
function ft_get_date($offset, $datetime, $format)
{
	global $LANG;

	$year = substr($datetime, 0, 4);
	$mon  = substr($datetime, 5, 2);
	$day  = substr($datetime, 8, 2);
	$hour = substr($datetime, 11, 2);
	$min  = substr($datetime, 14, 2);
	$sec  = substr($datetime, 17, 2);

	$timestamp = mktime($hour + $offset, $min, $sec, $mon, $day, $year);

	// if this is an English language (British, US English, English Canadian, etc), just
	// use the standard date() functionality (this is faster)
	$date_str = "";
	if ($LANG["special_language"] == "English")
		$date_str = date($format, $timestamp);
	else
	{
		// here's how this works. We replace the special chars in the date formatting
		// string with a single "@" character - which has no special meaning for either date()
		// or in regular expressions - and keep track of the order in which they appear. Then,
		// we call date() to convert all other characters and then replace the @'s with their
		// translated versions.
		$special_chars = array("D", "l", "F", "M", "a", "A"); // M: short month, F: long month
		$char_map = array();
		$new_format = "";
		for ($char_ind=0; $char_ind<strlen($format); $char_ind++)
		{
			if (in_array($format[$char_ind], $special_chars))
			{
				$char_map[] = $format[$char_ind];
				$format[$char_ind] = "@";
			}
			$new_format .= $format[$char_ind];
		}
		$date_str = date($new_format, $timestamp);

		// now replace the @'s with their translated equivalents
		$eng_strings = date(join(",", $char_map), $timestamp);
		$eng_string_arr = split(",", $eng_strings);
		for ($char_ind=0; $char_ind<count($char_map); $char_ind++)
		{
			$eng_string = $eng_string_arr[$char_ind];

			switch($char_ind)
			{
				case "F":
					$translated_str = $LANG["date_month_short_$eng_string"];
					break;
				case "M":
					$translated_str = $LANG["date_month_$eng_string"];
					break;
				default:
					$translated_str = $LANG["date_$eng_string"];
					break;
			}
			$date_str = preg_replace("/@/", $translated_str, $date_str, 1);
		}
	}

	return $date_str;
}


/**
 * Returns a date in Y-m-d H:i:s format, generally used for inserting into a MySQL
 * datetime field.
 *
 * @param string $timestamp an optional Unix timestamp to convert to a datetime
 * @return string the current datetime in string format
 * */
function ft_get_current_datetime($timestamp = "")
{
	$datetime = "";
	if (!empty($timestamp))
		$datetime = date("Y-m-d H:i:s", $timestamp);
	else
		$datetime = date("Y-m-d H:i:s");

	return $datetime;
}


/**
 * Examines a folder to check (a) it exists and (b) it has correct permissions.
 *
 * @param string $folder The full path to the folder
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_check_upload_folder($folder)
{
	global $LANG;

	// first, check server's temporary file upload folder
	$upload_tmp_dir = ini_get("upload_tmp_dir");

	if (!empty($upload_tmp_dir))
	{
		if (!is_dir($upload_tmp_dir))
		{
			$replacement_info = array("upload_folder" => $upload_tmp_dir);
			$message = ft_replace_placeholders($LANG["validation_invalid_upload_folder"], $replacement_info);
			return array(false, );
		}

		if (!is_writable($upload_tmp_dir))
			return array(false, $LANG["validation_upload_folder_not_writable"]);
	}

	// now check the folder specified by
	if (!is_dir($folder))
		return array(false, $LANG["validation_invalid_folder"]);

	if (!is_writable($folder))
		return array(false, $LANG["validation_folder_not_writable"]);

	return array(true, $LANG["notify_folder_correct_permissions"]);
}


/**
 * Checks to see if a filename is already taken or not, and if so returns a new, "free" filename.
 *
 * Examines a folder for a particular filename. If it finds it, it renames the filename to
 * x_$filename, where x is an integer starting at 1. Then it checks to see if that file exists,
 * also. Repeats until it finds a filename that isn't in use, and returns that value.
 *
 * @param string $folder The folder to examine.
 * @param string $filename The name of the filename to check for
 * @return string The name of a free filename.
 */
function ft_check_duplicate_filename($folder, $filename)
{
	// check the supplied dir is a valid, readable directory
	if (!is_dir($folder) && is_readable($folder))
		return;

	// the filename string to return
	$return_filename = $filename;

	// store all the filenames in the folder into an array
	$filenames = array();
	if ($handle = opendir($folder))
	{
		while (false !== ($file = readdir($handle)))
			$filenames[] = $file;
	}

	// if a file with the same name exists in the directory, find the next free filename of the form:
	// x_$filename, where x is a number starting with 1
	if (in_array($filename, $filenames))
	{
		// if it already starts with x_, strip off the prefix.
		if (preg_match("/^\d+_/", $filename))
			$filename = preg_replace("/^\d+_/", "", $filename);

		// now find the next available filename
		$next_num = 1;
		$return_filename = $next_num . "_" . $filename;
		while (in_array($return_filename, $filenames))
		{
			$return_filename = $next_num . "_" . $filename;
			$next_num++;
		}
	}

	// return the appropriate filename
	return $return_filename;
}


/**
 * Confirms that a folder dir and a URL purportedly linking to that folder do, in fact, match.
 *
 * If the URL does point to the folder it returns true; otherwise returns false. The function works
 * by creating a temporary file in $folder, then try and scrape it via file(). If it exists, the
 * folder is a match for URL and it returns true.
 *
 * Assumption: the "allow_url_fopen" setting in php.ini is set to "1" (Checks for this). If it's
 * not set it always returns false.
 *
 * @param string $folder A folder on this server.
 * @param string $url The URL that claims to point to <b>$folder</b>
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_check_folder_url_match($folder, $url)
{
	global $g_debug, $g_default_error_reporting, $LANG;

	$folder = rtrim(trim($folder), "/\\");
	$url    = rtrim(trim($url), "/\\");

	list($success, $message) = ft_check_upload_folder($folder);
	if (!$success)
		return array(false, $LANG["validation_folder_invalid_permissions"]);

	if (ini_get("allow_url_fopen") != "1")
		return array(false, $LANG["notify_allow_url_fopen_not_set"]);

	// create the temp file
	$test_file = "ft_" . date("U") . ".tmp";
	$fh = fopen("$folder/$test_file", "w");
	fwrite($fh, "Folder-URL match test");
	fclose($fh);

	// now try and read the file. We activate error reporting for the duration of this test so we
	// can examine any error messages that occur to provide some pointers for the user
	error_reporting(2047);
	ob_start();
	$result = file("$url/$test_file");
	$errors = ob_get_clean();
	error_reporting($g_default_error_reporting);

	// delete temp file
	@unlink("$folder/$test_file");


	// if $errors is empty, that means there was a match
	if (is_array($result) && $result[0] == "Folder-URL match test")
	{
		return array(true, $LANG["notify_folder_url_match"]);
	}
	else
	{
		$debug = "";
		if ($g_debug)
			$debug = "<br />$errors";

		// let's take a look at the warning.  [Assumption: error messages in English]
		//   "404 Not Found" - Not a match
		if (preg_match("/404 Not Found/", $errors))
			return array(false, $LANG["notify_folder_url_no_match"] . " $debug");

		//   "Authorization Required"    - PHP isn't allowed to look at that URL (URL protected by a .htaccess probably)
		else if (preg_match("/Authorization Required/", $errors))
			return array(false, $LANG["notify_folder_url_no_access"] . " $debug");

		return array(false, $LANG["notify_folder_url_unknown_error"]);
	}
}


/**
 * Helper function, used on all POST data to properly escape user-inputted values; takes into
 * consideration whether magic quotes is enabled or not.
 *
 * Not recursive; only works for one layer of arrays.
 *
 * @param array $hash The $_GET or _$POST hash.
 * @return array The "clean" (escaped) hash.
 */
function ft_clean_hash($hash)
{
	$clean_hash = $hash;

	if (!get_magic_quotes_gpc())
	{
		while (list($key, $value) = each($hash))
		{
			if (!is_array($value))
				$clean_hash[$key] = addslashes($value);
			else
			{
				$clean_array = array();
				foreach ($value as $val)
					$clean_array[] = addslashes($val);
				$clean_hash[$key] = $clean_array;
			}
		}
	}

	return $clean_hash;
}


/**
 * This invaluable little function is used for storing and overwriting the contents of a single
 * form field in sessions based on a sequence of priorities.
 *
 * It assumes that a variable value can be found in GET, POST or SESSIONS (or all three). What this
 * function does is return the value stored in the most important variable (GET first, POST second,
 * SESSIONS third), and update sessions at the same time. This is extremely helpful in situations
 * where you need to store information in sessions, but possibly overwrite it depending on what
 * the user selected. Also, a third parameter is included as a way to set a default value.
 *
 * @param string $field_name the field name.
 * @param string $session_name the session key for this field name.
 * @param string $default_value the default value for the field.
 * @return string The field content.
 */
function ft_load_field($field_name, $session_name, $default_value = "")
{
	$field = $default_value;

	if (isset($_GET[$field_name]))
	{
		$field = $_GET[$field_name];
		$_SESSION['ft'][$session_name] = $field;
	}
	else if (isset($_POST[$field_name]))
	{
		$field = $_POST[$field_name];
		$_SESSION['ft'][$session_name] = $field;
	}
	else if (isset($_SESSION['ft'][$session_name]))
		$field = $_SESSION['ft'][$session_name];

	return $field;
}


/**
 * Generic field validation function.
 *
 * For documentation and examples on this function, visit:
 * http://www.benjaminkeen.com/software/php_validation/
 *
 * @param string $fields A hash of field values.
 * @param string $session_name The validation rules.
 */
function validate_fields($fields, $rules)
{
	$errors = array();

	// loop through rules
	for ($i=0; $i<count($rules); $i++)
	{
		// split row into component parts
		$row = split(",", $rules[$i]);

		// while the row begins with "if:..." test the condition. If true, strip the if:..., part and
		// continue evaluating the rest of the line. Keep repeating this while the line begins with an
		// if-condition. If it fails any of the conditions, don't bother validating the rest of the line
		$satisfies_if_conditions = true;
		while (preg_match("/^if:/", $row[0]))
		{
			$condition = preg_replace("/^if:/", "", $row[0]);

			// check if it's a = or != test
			$comparison = "equal";
			$parts = array();
			if (preg_match("/!=/", $condition))
			{
				$parts = split("!=", $condition);
				$comparison = "not_equal";
			}
			else
				$parts = split("=", $condition);

			$field_to_check = $parts[0];
			$value_to_check = $parts[1];

			// if the VALUE is NOT the same, we don't need to validate this field. Return.
			if ($comparison == "equal" && $fields[$field_to_check] != $value_to_check)
			{
				$satisfies_if_conditions = false;
				break;
			}
			else if ($comparison == "not_equal" && $fields[$field_to_check] == $value_to_check)
			{
				$satisfies_if_conditions = false;
				break;
			}
			else
				array_shift($row);    // remove this if-condition from line, and continue validating line
		}

		if (!$satisfies_if_conditions)
			continue;


		$requirement = $row[0];
		$field_name  = $row[1];

		// depending on the validation test, store the incoming strings for use later...
		if (count($row) == 6)        // valid_date
		{
			$field_name2   = $row[2];
			$field_name3   = $row[3];
			$date_flag     = $row[4];
			$error_message = $row[5];
		}
		else if (count($row) == 5)     // reg_exp (WITH flags like g, i, m)
		{
			$fieldName2   = $row[2];
			$fieldName3   = $row[3];
			$errorMessage = $row[4];
		}
		else if (count($row) == 4)     // same_as, custom_alpha, reg_exp (without flags like g, i, m)
		{
			$field_name2   = $row[2];
			$error_message = $row[3];
		}
		else
			$error_message = $row[2];    // everything else!


		// if the requirement is "length=...", rename requirement to "length" for switch statement
		if (preg_match("/^length/", $requirement))
		{
			$length_requirements = $requirement;
			$requirement         = "length";
		}

		// if the requirement is "range=...", rename requirement to "range" for switch statement
		if (preg_match("/^range/", $requirement))
		{
			$range_requirements = $requirement;
			$requirement        = "range";
		}


		// now, validate whatever is required of the field
		switch ($requirement)
		{
			case "required":
				if (!isset($fields[$field_name]) || $fields[$field_name] == "")
					$errors[] = $error_message;
				break;

			case "digits_only":
				if (isset($fields[$field_name]) && preg_match("/\D/", $fields[$field_name]))
					$errors[] = $error_message;
				break;

			case "letters_only":
				if (isset($fields[$field_name]) && preg_match("/[^a-zA-Z]/", $fields[$field_name]))
					$errors[] = $error_message;
				break;

			// doesn't fail if field is empty
			case "valid_email":
				$regexp="/^[a-z0-9]+([_\\.-][a-z0-9]+)*@([a-z0-9]+([\.-][a-z0-9]+)*)+\\.[a-z]{2,}$/i";
				if (isset($fields[$field_name]) && !empty($fields[$field_name]) && !preg_match($regexp, $fields[$field_name]))
					$errors[] = $error_message;
				break;

			case "length":
				$comparison_rule = "";
				$rule_string     = "";

				if      (preg_match("/length=/", $length_requirements))
				{
					$comparison_rule = "equal";
					$rule_string = preg_replace("/length=/", "", $length_requirements);
				}
				else if (preg_match("/length>=/", $length_requirements))
				{
					$comparison_rule = "greater_than_or_equal";
					$rule_string = preg_replace("/length>=/", "", $length_requirements);
				}
				else if (preg_match("/length<=/", $length_requirements))
				{
					$comparison_rule = "less_than_or_equal";
					$rule_string = preg_replace("/length<=/", "", $length_requirements);
				}
				else if (preg_match("/length>/", $length_requirements))
				{
					$comparison_rule = "greater_than";
					$rule_string = preg_replace("/length>/", "", $length_requirements);
				}
				else if (preg_match("/length</", $length_requirements))
				{
					$comparison_rule = "less_than";
					$rule_string = preg_replace("/length</", "", $length_requirements);
				}

				switch ($comparison_rule)
				{
					case "greater_than_or_equal":
						if (!(strlen($fields[$field_name]) >= $rule_string))
							$errors[] = $error_message;
						break;
					case "less_than_or_equal":
						if (!(strlen($fields[$field_name]) <= $rule_string))
							$errors[] = $error_message;
						break;
					case "greater_than":
						if (!(strlen($fields[$field_name]) > $rule_string))
							$errors[] = $error_message;
						break;
					case "less_than":
						if (!(strlen($fields[$field_name]) < $rule_string))
							$errors[] = $error_message;
						break;
					case "equal":
						// if the user supplied two length fields, make sure the field is within that range
						if (preg_match("/-/", $rule_string))
						{
							list($start, $end) = split("-", $rule_string);
							if (strlen($fields[$field_name]) < $start || strlen($fields[$field_name]) > $end)
								$errors[] = $error_message;
						}
						// otherwise, check it's EXACTLY the size the user specified
						else
						{
							if (strlen($fields[$field_name]) != $rule_string)
								$errors[] = $error_message;
						}
						break;
				}
				break;

			case "range":
				$comparison_rule = "";
				$rule_string     = "";

				if      (preg_match("/range=/", $range_requirements))
				{
					$comparison_rule = "equal";
					$rule_string = preg_replace("/range=/", "", $range_requirements);
				}
				else if (preg_match("/range>=/", $range_requirements))
				{
					$comparison_rule = "greater_than_or_equal";
					$rule_string = preg_replace("/range>=/", "", $range_requirements);
				}
				else if (preg_match("/range<=/", $range_requirements))
				{
					$comparison_rule = "less_than_or_equal";
					$rule_string = preg_replace("/range<=/", "", $range_requirements);
				}
				else if (preg_match("/range>/", $range_requirements))
				{
					$comparison_rule = "greater_than";
					$rule_string = preg_replace("/range>/", "", $range_requirements);
				}
				else if (preg_match("/range</", $range_requirements))
				{
					$comparison_rule = "less_than";
					$rule_string = preg_replace("/range</", "", $range_requirements);
				}

				switch ($comparison_rule)
				{
					case "greater_than":
						if (!($fields[$field_name] > $rule_string))
							$errors[] = $error_message;
						break;
					case "less_than":
						if (!($fields[$field_name] < $rule_string))
							$errors[] = $error_message;
						break;
					case "greater_than_or_equal":
						if (!($fields[$field_name] >= $rule_string))
							$errors[] = $error_message;
						break;
					case "less_than_or_equal":
						if (!($fields[$field_name] <= $rule_string))
							$errors[] = $error_message;
						break;
					case "equal":
						list($start, $end) = split("-", $rule_string);

						if (($fields[$field_name] < $start) || ($fields[$field_name] > $end))
							$errors[] = $error_message;
						break;
				}
				break;

			case "same_as":
				if ($fields[$field_name] != $fields[$field_name2])
					$errors[] = $error_message;
				break;

			case "valid_date":
				// this is written for future extensibility of isValidDate function to allow
				// checking for dates BEFORE today, AFTER today, IS today and ANY day.
				$is_later_date = false;
				if    ($date_flag == "later_date")
					$is_later_date = true;
				else if ($date_flag == "any_date")
					$is_later_date = false;

				if (!is_valid_date($fields[$field_name], $fields[$field_name2], $fields[$field_name3], $is_later_date))
					$errors[] = $error_message;
				break;

			case "is_alpha":
				if (preg_match('/[^A-Za-z0-9]/', $fields[$field_name]))
					$errors[] = $error_message;
				break;

			case "custom_alpha":
				$chars = array();
				$chars["L"] = "[A-Z]";
				$chars["V"] = "[AEIOU]";
				$chars["l"] = "[a-z]";
				$chars["v"] = "[aeiou]";
				$chars["D"] = "[a-zA-Z]";
				$chars["F"] = "[aeiouAEIOU]";
				$chars["C"] = "[BCDFGHJKLMNPQRSTVWXYZ]";
				$chars["x"] = "[0-9]";
				$chars["c"] = "[bcdfghjklmnpqrstvwxyz]";
				$chars["X"] = "[1-9]";
				$chars["E"] = "[bcdfghjklmnpqrstvwxyzBCDFGHJKLMNPQRSTVWXYZ]";

				$reg_exp_str = "";
				for ($j=0; $j<strlen($field_name2); $j++)
				{
					if (array_key_exists($field_name2[$j], $chars))
						$reg_exp_str .= $chars[$field_name2[$j]];
					else
						$reg_exp_str .= $field_name2[$j];
				}

				if (!empty($fields[$field_name]) && !preg_match("/$reg_exp_str/", $fields[$field_name]))
					$errors[] = $error_message;
				break;

			case "reg_exp":
				$reg_exp_str = $field_name2;

				// rather crumby, but...
				if (count($row) == 5)
					$reg_exp = "/" . $reg_exp_str . "/" . $row[3];
				else
					$reg_exp = "/" . $reg_exp_str . "/";

				if (!empty($fields[$field_name]) && !preg_match($reg_exp, $fields[$field_name]))
					$errors[] = $error_message;
				break;

			default:
				die("Unknown requirement flag in validateFields(): $requirement");
				break;
		}
	}

	return $errors;
}


/**
 * A helper function for the validate_fields function.
 *
 * For documentation and examples on this function, visit:
 * http://www.benjaminkeen.com/software/php_validation/
 *
 * @param string $fields A hash of field values.
 * @param string $session_name The validation rules.
 */
function is_valid_date($month, $day, $year, $is_later_date)
{
	// depending on the year, calculate the number of days in the month
	if ($year % 4 == 0)      // LEAP YEAR
		$days_in_month = array(31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
	else
		$days_in_month = array(31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);

	// first, check the incoming month and year are valid.
	if (!$month || !$day || !$year) return false;
	if (1 > $month || $month > 12)  return false;
	if ($year < 0)                  return false;
	if (1 > $day || $day > $days_in_month[$month-1]) return false;


	// if required, verify the incoming date is LATER than the current date.
	if ($is_later_date)
	{
		// get current date
		$today = date("U");
		$date = mktime(0, 0, 0, $month, $day, $year);
		if ($date < $today)
			return false;
	}

	return true;
}


/**
 * This function - called on the login page - checks that the /install folder has been removed.
 *
 * If the folder still exists, it redirects to error.php with a unique GET flag to display an
 * appropriate error message.
 */
function ft_verify_setup()
{
	global $LANG;

	if (is_dir('install'))
	{
		$error = "
			<div class='heading'>{$LANG["text_ran_installation_script_q"]}</div>
			<p>
				{$LANG["word_no_q"]} <a href='install/index.php'>{$LANG["phrase_click_here_get_started"]}</a>
			</p>
			<p>
				{$LANG["text_intro_page_delete_install_folder"]}
			</p>";

		ft_handle_error($error, "", "notify");
	}
}


/**
 * This function was added in 1.4.7 to handle serious errors instead of the former die()
 * messages hardcoded throughout. This function stores the error in sessions and redirects to
 * error.php, which decides how the error is displayed based on the error_type ("notify": for softer
 * errors like the install folder hasn't been deleted; "error" for more serious problems) and on
 * whether or not the global $g_debug option is enabled. If it is, the error.php page displays
 * the nitty-gritty errors returned by the server / database.
 *
 * @param string $error_message the user-friendly version of the error.
 * @param string $debug_details the error message returned by the server / database.
 * @param string $error_type either "error" or "notify"
 */
function ft_handle_error($error_message, $debug_details, $error_type = "error")
{
	global $g_root_url;

	if (!empty($g_root_url))
		$g_root_url = "$g_root_url/";

	$_SESSION["ft"]["last_error"]       = $error_message;
	$_SESSION["ft"]["last_error_debug"] = $debug_details;
	$_SESSION["ft"]["last_error_type"]  = $error_type;

	header("Location: {$g_root_url}error.php");
	exit;
}


// --------------------------------------------------------------------------------------------


if (!function_exists("mb_strtoupper"))
{
	/**
	 * A fallback function for servers that don't include the mbstring PHP extension. Note: 
	 * this function is NOT multi-byte; it can't be emulated without the extension. However,
	 * this will at least allow the user to use Form Tools without errors.
	 *
	 * @param string $str
	 * @return string the uppercased string
	 */
	function mb_strtoupper($str)
	{
		return strtoupper($str);
	}
}

if (!function_exists("mb_strlen"))
{
	/**
	 * A fallback function for servers that don't include the mbstring PHP extension. Note: 
	 * this function is NOT multi-byte; it can't be emulated without the extension. However,
	 * this will at least allow the user to use Form Tools without errors.
	 *
	 * @param string $str
	 * @return string the length of the string
	 */
	function mb_strlen($str)
	{
		return strlen($str);
	}
}

if (!function_exists("mb_substr"))
{
	/**
	 * A fallback function for servers that don't include the mbstring PHP extension. Note: 
	 * this function is NOT multi-byte; it can't be emulated without the extension. However,
	 * this will at least allow the user to use Form Tools without errors.
	 *
	 * @param string $str
	 * @return string the length of the string
	 */
	function mb_substr($str, $start, $length)
	{
		return substr($str, $start, $length);
	}
}
