<?php

/**
 * This file contains all functions relating to managing forms.
 *
 * @copyright Benjamin Keen 2007
 * @author Benjamin Keen <ben.keen@gmail.com>
 * @package 1-5-1
 * @subpackage Forms
 */


// -------------------------------------------------------------------------------------------------


/**
 * This function sets up the main form values in preparation for a test submission by the actual
 * form. It is called from step 1 of the form creation page for NEW forms.
 *
 * @param array $infohash This parameter should be a hash (e.g. $_POST or $_GET) containing the
 *             various fields from the step 1 add form page.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 *               [2]: new form ID (success only)
 */
function ft_setup_form($infohash)
{
	global $g_table_prefix, $g_debug, $LANG;

	$success = true;
	$message = "";

	// check required $infohash fields
	$rules = array();
	$rules[] = "required,selected_user_ids,{$LANG["validation_specify_client"]}";
	$rules[] = "required,form_name,{$LANG["validation_no_form_name"]}";
	$rules[] = "required,form_url,{$LANG["validation_no_form_url"]}";
	$rules[] = "required,form_type,{$LANG["validation_no_form_type"]}";
	$rules[] = "if:form_type=direct,required,redirect_url,{$LANG["validation_no_redirect_url"]}";
	$errors = validate_fields($infohash, $rules);

	// if there are errors, piece together an error message string and return it
	if (!empty($errors))
	{
		$success = false;
		array_walk($errors, create_function('&$el','$el = "&bull;&nbsp; " . $el;'));
		$message = join("<br />", $errors);
		return array ($success, $message, "");
	}

	// extract values
	$user_ids     = $infohash['selected_user_ids'];
	$form_type    = $infohash['form_type'];
	$form_name    = trim($infohash['form_name']);
	$form_url     = trim($infohash['form_url']);
	$redirect_url = trim($infohash['redirect_url']);

	$finalized_submissions = isset($infohash['finalized_submissions']) ? $infohash['finalized_submissions'] : "";
	if (!$finalized_submissions)
		$finalized_submissions = "no";

	$now = ft_get_current_datetime();

	// all checks out, so add the new form
	$settings = ft_get_settings();
	$num_submissions_per_page = $settings['default_num_submissions_per_page'];

	$query = "
		 INSERT INTO {$g_table_prefix}forms
						 (form_type, date_created, is_active, is_complete, form_name, form_url, redirect_url,
							finalized_submissions, auto_email_admin, num_submissions_per_page)
		 VALUES ('$form_type', '$now', 'no', 'no', '$form_name', '$form_url', '$redirect_url',
						 '$finalized_submissions', 'no', '$num_submissions_per_page')
					 ";

	$link = ft_db_connect();
	$result = mysql_query($query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());

	$new_form_id = mysql_insert_id();

	// now store which clients are assigned to this form [remove any old ones first]
	foreach ($user_ids as $user_id)
	{
		$query = mysql_query("
			 INSERT INTO {$g_table_prefix}client_forms (user_id, form_id)
			 VALUES  ($user_id, $new_form_id)
												 ");
	}

	return array($success, $message, $new_form_id);
}


/**
 * This function updates the main form values in preparation for a test submission by the actual
 * form. It is called from step 1 of the form creation page when UPDATING an existing, incomplete
 * form.
 *
 * @param array $infohash This parameter should be a hash (e.g. $_POST or $_GET) containing the
 *             various fields from the step 2 add form page.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_set_form_main_settings($infohash)
{
	global $g_table_prefix, $LANG;

	$success = true;
	$message = "";

	// check required infohash fields
	$rules = array();
	$rules[] = "required,selected_user_ids,{$LANG["validation_specify_client"]}";
	$rules[] = "required,form_name,{$LANG["validation_no_form_name"]}";
	$rules[] = "required,form_url,{$LANG["validation_no_form_url"]}";
	$rules[] = "required,form_type,{$LANG["validation_no_form_type"]}";
	$rules[] = "if:form_type=direct,required,redirect_url,{$LANG["validation_no_redirect_url"]}";
	$errors = validate_fields($infohash, $rules);

	if (!empty($errors))
	{
		$success = false;
		array_walk($errors, create_function('&$el','$el = "&bull;&nbsp; " . $el;'));
		$message = join("<br />", $errors);
		return array ($success, $message, "");
	}

	// extract values
	$user_ids     = $infohash['selected_user_ids'];
	$form_id      = $infohash['form_id'];
	$form_type    = $infohash['form_type'];
	$form_name    = trim($infohash['form_name']);
	$form_url     = trim($infohash['form_url']);
	$redirect_url = trim($infohash['redirect_url']);

	$finalized_submissions = $infohash['finalized_submissions'];
	if (!$finalized_submissions)
		$finalized_submissions = "no";

	// all checks out, so add the new form
	$settings = ft_get_settings();
	$num_submissions_per_page  = $settings['default_num_submissions_per_page'];

	$link = ft_db_connect();
	$query = mysql_query("
		 UPDATE {$g_table_prefix}forms
		 SET    form_type = '$form_type',
						is_active = 'no',
						is_complete = 'no',
						form_name = '$form_name',
						form_url = '$form_url',
						redirect_url = '$redirect_url',
						finalized_submissions = '$finalized_submissions',
						num_submissions_per_page = '$num_submissions_per_page'
		 WHERE  form_id = '$form_id'
					 ")
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());

	$query = mysql_query("
		DELETE FROM {$g_table_prefix}client_forms
		WHERE form_id = $form_id
											 ");

	foreach ($user_ids as $user_id)
	{
		$query = mysql_query("
			 INSERT INTO {$g_table_prefix}client_forms (user_id, form_id)
			 VALUES  ($user_id, $form_id)
												 ");
	}

	return array($success, $message);
}


/**
 * This function is called on Step 3 of the Add Form process and updates the
 * order, display, display name, field type and "pass on" value.
 *
 * @param array $infohash This parameter should be a hash (e.g. $_POST or $_GET) containing the
 *             various fields from the step 3 add form page.
 */
function ft_set_form_database_settings($infohash, $form_id)
{
	global $g_table_prefix, $g_debug;

	$field_info = array();

	// loop through $infohash and for each field_X_order values, store the various values
	while (list($key, $val) = each($infohash))
	{
		// find the field id
		preg_match("/^field_(\d+)$/", $key, $match);

		if (!empty($match[1]))
		{
			$field_id = $match[1];

			// field type (size)
			$field_size = "";
			if (isset($infohash["field_{$field_id}_type"]))
				$field_size = $infohash["field_{$field_id}_type"];

			// is this field to be included on redirect or not?
			$include_on_redirect = "no";
			if (isset($infohash["field_{$field_id}_include_on_redirect"]) && !empty($infohash["field_{$field_id}_include_on_redirect"]))
				$include_on_redirect = "yes";

			$field_info[] = array($field_id, $field_size, $include_on_redirect);
		}
	}
	reset($infohash);

	$link = ft_db_connect();
	foreach ($field_info as $field)
	{
		$query = "
			UPDATE {$g_table_prefix}form_templates
			SET    include_on_redirect = '{$field[2]}',
						 field_size = '{$field[1]}'
			WHERE  field_id = {$field[0]}
						 ";
	 mysql_query($query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());
	}

	ft_db_disconnect($link);
}


/**
 * This function is called on Step 4 of the Add Form process and updates the
 * display name, etc.
 *
 * @param array $infohash This parameter should be a hash (e.g. $_POST or $_GET) containing the
 *             various fields from the step 4 add form page.
 */
function ft_set_form_display_settings($infohash)
{
	global $g_table_prefix, $g_debug;

	$field_info = array();
	$form_id = $infohash['form_id'];

	// loop through $infohash and for each field_X_order values, store the various values
	while (list($key, $val) = each($infohash))
	{
		// find the field id
		preg_match("/^field_(\d+)$/", $key, $match);

		if (!empty($match[1]))
		{
			$field_id = $match[1];

			// display or not
			$field_display = "no";
			if (isset($infohash["field_{$field_id}_display"]) && !empty($infohash["field_{$field_id}_display"]))
				$field_display = "yes";

			// display name
			$field_display_name = "";
			if (isset($infohash["field_{$field_id}_display_name"]))
				$field_display_name = $infohash["field_{$field_id}_display_name"];

			$field_info[] = array($field_id, $field_display, $field_display_name);
		}
	}
	reset($infohash);

	$link = ft_db_connect();
	foreach ($field_info as $field)
	{
		$query = "
			UPDATE {$g_table_prefix}form_templates
			SET    admin_display = '{$field[1]}',
						 field_title = '{$field[2]}'
			WHERE  field_id = {$field[0]}
						 ";
		mysql_query($query)
			or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());
	}

	ft_db_disconnect($link);
}


/**
 * This function "finalizes" the form, i.e. marks it as set up and ready to go.
 *
 * This is where the excitement happens. This function is called when the user  has completed step
 * 4 of the Add Form process, after the user is satisfied that the data that is stored is correct.
 * This function does the following:
 * <ul>
 * <li>Adds a new record to the <b>form_admin_fields</b> table listing which of the database fields are
 * to be visible in the admin interface panel for this form.</li>
 * <li>Creates a new form table with the column information specified in infohash.</li>
 * </ul>
 *
 * @param array $infohash This parameter should be a hash (e.g. $_POST or $_GET) containing the
 *             various fields from the Step 4 Add Form page.
 */
function ft_finalize_form($form_id)
{
	global $g_table_prefix;

	$link = ft_db_connect();
	$form_info = ft_get_form_template($form_id);
	ft_db_disconnect($link);

	$query = "
		CREATE TABLE {$g_table_prefix}form_$form_id (
			submission_id MEDIUMINT UNSIGNED NOT NULL AUTO_INCREMENT,
			PRIMARY KEY(submission_id),\n";

	while ($info = mysql_fetch_assoc($form_info))
	{
		// don't add system fields (submission ID, date & IP address)
		if ($info['field_type'] == "system")
			continue;

		$query .= "{$info['col_name']} ";
		switch ($info['field_size'])
		{
			case "tiny":
				$query .= "VARCHAR(5),\n";
				break;
			case "small":
				$query .= "VARCHAR(20),\n";
				break;
			case "medium":
				$query .= "VARCHAR(255),\n";
				break;
			case "large":
				$query .= "BLOB,\n";
				break;
			case "very_large":
				$query .= "MEDIUMBLOB,\n";
				break;
		}
	}

	$query .= "submission_date DATETIME,
						 ip_address VARCHAR(15),
						 is_finalized ENUM('yes','no') default 'yes')
						 TYPE=INNODB";
	$handle = ft_db_connect();
	mysql_query($query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());

	$now = ft_get_current_datetime();

	// now the form is complete. Update it as is_complete and enabled
	$query = "
			UPDATE {$g_table_prefix}forms
			SET    is_initialized = 'yes',
						 is_complete = 'yes',
						 is_active = 'yes',
						 date_created = '$now'
			WHERE  form_id = $form_id
					 ";
	mysql_query($query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());

	ft_db_disconnect($handle);
}


/**
 * "Uninitializes" a form, letting the user to resend the test submission.
 *
 * @param integer $form_id The unique form ID
 */
function ft_uninitialize_form($form_id)
{
	global $g_table_prefix;

	$handle = ft_db_connect();

	mysql_query("
			UPDATE  {$g_table_prefix}forms
			SET     is_initialized = 'no'
			WHERE   form_id = $form_id
							");

	ft_db_disconnect($handle);
}


/**
 * Called by administrators; updates the content stored on the "Main" tab in the Edit Form pages.
 *
 * @param integer $infohash A hash containing the contents of the Edit Form Main tab.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_update_form_main_tab($infohash, $form_id)
{
	global $g_table_prefix, $LANG;

	$infohash = ft_clean_hash($infohash);

	$success = true;
	$message = $LANG["notify_form_updated"];

	// check required POST fields
	$rules = array();
	$rules[] = "required,form_name,{$LANG["validation_no_form_name"]}";
	$rules[] = "required,form_url,{$LANG["validation_no_form_url"]}";
	$rules[] = "required,form_type,{$LANG["validation_no_form_type"]}";
	$rules[] = "if:form_type=direct,required,redirect_url,{$LANG["validation_no_redirect_url"]}";
	$errors = validate_fields($infohash, $rules);

	if (!empty($errors))
	{
		$success = false;
		array_walk($errors, create_function('&$el','$el = "&bull;&nbsp; " . $el;'));
		$message = join("<br />", $errors);
		return array ($success, $message, "");
	}

	// extract values
	$user_ids     = $infohash['selected_user_ids'];
	$form_type    = $infohash['form_type'];
	$form_name    = $infohash['form_name'];
	$form_url     = $infohash['form_url'];
	$redirect_url = $infohash['redirect_url'];
	$finalized_submissions = isset($infohash['finalized_submissions']) ? $infohash['finalized_submissions'] : "yes";
	$printer_friendly_format = $infohash['printer_friendly_format'];
	$hide_printer_friendly_empty_fields = $infohash['hide_empty_fields'];
	$auto_delete_submission_files = $infohash['auto_delete_submission_files'];
	$submission_strip_tags = $infohash["submission_strip_tags"];

	$is_active ="";
	if (!empty($infohash['active']))
		$is_active = "is_active = '{$infohash['active']}',";

	$link = ft_db_connect();

	$query = "
		UPDATE {$g_table_prefix}forms
		SET    $is_active
					 form_type = '$form_type',
					 form_url = '$form_url',
					 form_name = '$form_name',
					 redirect_url = '$redirect_url',
					 finalized_submissions = '$finalized_submissions',
					 printer_friendly_format = '$printer_friendly_format',
					 hide_printer_friendly_empty_fields = '$hide_printer_friendly_empty_fields',
					 auto_delete_submission_files ='$auto_delete_submission_files',
					 submission_strip_tags = '$submission_strip_tags'
		WHERE  form_id = $form_id
					 ";

	$result = mysql_query($query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());

	$query = mysql_query("
		DELETE FROM {$g_table_prefix}client_forms
		WHERE form_id = $form_id
											 ");

	foreach ($user_ids as $user_id)
	{
		$query = mysql_query("
			 INSERT INTO {$g_table_prefix}client_forms (user_id, form_id)
			 VALUES  ($user_id, $form_id)
												 ");
	}

	ft_db_disconnect($link);

	// do an additional check here to see if the file exists

	return array($success, $message);
}


/**
 * Called by administrators; updates the content stored on the "Display" tab in the Edit Form
 * pages.
 *
 * @param array $infohash a hash containing the contents of the Edit Form Display tab
 * @param integer $form_id the unique form ID
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_update_form_display_tab($infohash, $form_id)
{
	global $g_table_prefix, $g_root_url, $g_root_dir, $g_debug, $LANG;

	$infohash = ft_clean_hash($infohash);

	$success = true;
	$message = "";

	// check required POST fields
	$rules = array();
	$rules[] = "required,num_submissions_per_page,{$LANG["validation_no_num_submissions_per_page"]}";
	$rules[] = "digits_only,num_submissions_per_page,{$LANG["validation_invalid_num_submissions_per_page_field"]}";
	$errors = validate_fields($infohash, $rules);

	if (!empty($errors))
	{
		$success = false;
		array_walk($errors, create_function('&$el','$el = "&bull;&nbsp; " . $el;'));
		$message = join("<br />", $errors);
		return array ($success, $message);
	}

	$num_submissions_per_page = $infohash['num_submissions_per_page'];
	$default_sort_field       = $infohash['default_sort_field'];
	$default_sort_field_order = $infohash['default_sort_field_order'];
	$show_ip_address          = $infohash['show_ip_address'];

	$link = ft_db_connect();

	// update form values
	$result = mysql_query("
		UPDATE {$g_table_prefix}forms
		SET    num_submissions_per_page = '$num_submissions_per_page',
					 default_sort_field = '$default_sort_field',
					 default_sort_field_order = '$default_sort_field_order',
					 show_ip_address = '$show_ip_address'
		WHERE  form_id = '$form_id'
				 ");

	if (!$result)
	{
		$success = false;
		$message = $LANG["notify_form_field_not_updated"];
		if ($g_debug) $message .= $LANG["word_error_c"] . " " . mysql_error();
		return array($success, $message);
	}


	// next, update the display fields
	$field_info = array();

	// loop through $infohash and for each field_X_order values, store the various values
	while (list($key, $val) = each($infohash))
	{
		// find the field id
		preg_match("/^field_(\d+)$/", $key, $match);

		if (!empty($match[1]))
		{
			$field_id = $match[1];

			// is this a display field or not?
			if (isset($infohash["field_{$field_id}_display"]))
				$display = "yes";
			else
				$display = "no";

			// sortable?
			if (isset($infohash["field_{$field_id}_sortable"]))
				$sortable = "yes";
			else
				$sortable = "no";

			// extract the display name
			if (isset($infohash["field_{$field_id}_display_name"]))
				$display_name = $infohash["field_{$field_id}_display_name"];
			else
				$display_name = "";

			// extract the field type
			if (isset($infohash["field_{$field_id}_type"]))
				$field_type = $infohash["field_{$field_id}_type"];
			else
				$field_type = "";

			$field_info[] = array($field_id, $display, $sortable, $display_name, $field_type);
		}
	}
	reset($infohash);

	// update the values
	foreach ($field_info as $field)
	{
		$query = "
			UPDATE {$g_table_prefix}form_templates
			SET    admin_display = '{$field[1]}',
						 is_sortable   = '{$field[2]}',
						 field_title   = '{$field[3]}',
						 field_type    = '{$field[4]}'
			WHERE  field_id = {$field[0]}
								";
		$link = ft_db_connect();
		mysql_query($query)
			or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());
		ft_db_disconnect($link);

		// additional check: if this field is being marked as a file upload field and the file_upload_url
		// and file_upload_dir values are set to blank, give them a default value
		if ($field[4] == "file")
		{
			$field_info = ft_get_form_field($field[0]);

			if (empty($field_info['file_upload_url']) && empty($field_info['file_upload_dir']))
			{
				$file_upload_url = "$g_root_url/files";
				$file_upload_dir = "$g_root_dir/files";

				$query = "
					UPDATE {$g_table_prefix}form_templates
					SET    file_upload_url = '$file_upload_url',
								 file_upload_dir = '$file_upload_dir'
					WHERE  field_id = {$field[0]}
										";
				$link = ft_db_connect();
				mysql_query($query)
					or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());
				ft_db_disconnect($link);
			}
		}

	}

	ft_db_disconnect($link);

	return array(true, $LANG["notify_display_fields_updated"]);
}


/**
 * Called by administrators; updates the content stored on the "Email" (basic) tab in the Edit
 * Form pages.
 *
 * @param integer $infohash A hash containing the contents of the Edit Form Email (basic) tab.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_update_form_email_basic_tab($infohash)
{
	global $g_table_prefix, $LANG;

	$success = true;
	$message = $LANG["notify_form_updated"];

	// clean the POST fields
	$infohash = ft_clean_hash($infohash);

	// extract values
	$form_id      = $infohash['form_id'];
	$email_format = $infohash['email_format'];

	$auto_email_admin = isset($infohash['auto_email_admin']) ? "yes" : "no";
	$auto_email_user  = isset($infohash['auto_email_user']) ? "yes" : "no";
	$user_email_field = $infohash['user_email_field'];
	$user_first_name_field = $infohash['user_first_name_field'];
	$user_last_name_field  = $infohash['user_last_name_field'];

	$include_admin_email = isset($infohash['include_admin_email']) ? $infohash['include_admin_email'] : "no";
	$admin_email_from = $infohash['admin_email_from'];
	$admin_email_from_custom = $infohash['admin_email_from_custom'];
	$admin_email_reply_to = $infohash['admin_email_reply_to'];
	$admin_email_reply_to_custom = $infohash['admin_email_reply_to_custom'];
	$user_email_from = $infohash['user_email_from'];
	$user_email_from_custom = $infohash['user_email_from_custom'];
	$user_email_reply_to = $infohash['user_email_reply_to'];
	$user_email_reply_to_custom = $infohash['user_email_reply_to_custom'];

	$link = ft_db_connect();

	$query = "
		UPDATE {$g_table_prefix}forms
		SET    auto_email_admin = '$auto_email_admin',
					 auto_email_user  = '$auto_email_user',
					 user_email_field = '$user_email_field',
					 user_first_name_field = '$user_first_name_field',
					 user_last_name_field = '$user_last_name_field',
					 admin_email_from = '$admin_email_from',
					 admin_email_from_custom = '$admin_email_from_custom',
					 admin_email_reply_to = '$admin_email_reply_to',
					 admin_email_reply_to_custom = '$admin_email_reply_to_custom',
					 user_email_from = '$user_email_from',
					 user_email_from_custom = '$user_email_from_custom',
					 user_email_reply_to = '$user_email_reply_to',
					 user_email_reply_to_custom = '$user_email_reply_to_custom',
					 include_admin_email = '$include_admin_email',
					 email_format = '$email_format'
		WHERE  form_id = $form_id
					 ";

	$result = mysql_query($query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());


	ft_db_disconnect($link);

	return array($success, $message);
}


/**
 * Called by administrators; updates the content stored on the "Email" (advanced) tab in the Edit
 * Form pages.
 *
 * @param integer $infohash A hash containing the contents of the Edit Form Email (advanced) tab.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_update_form_email_advanced_tab($infohash)
{
	global $g_table_prefix, $LANG;

	$success = true;
	$message = $LANG["notify_form_updated"];

	// clean the POST fields
	$infohash = ft_clean_hash($infohash);

	// extract values
	$form_id          = $infohash['form_id'];
	$auto_email_admin = isset($infohash['auto_email_admin']) ? "yes" : "no";
	$auto_email_user  = isset($infohash['auto_email_user']) ? "yes" : "no";
	$user_email_field = $infohash['user_email_field'];
	$user_first_name_field = $infohash['user_first_name_field'];
	$user_last_name_field  = $infohash['user_last_name_field'];

	$include_admin_email = isset($infohash['include_admin_email']) ? $infohash['include_admin_email'] : "no";
	$admin_email_subject = $infohash['admin_email_subject'];
	$admin_email_from = $infohash['admin_email_from'];
	$admin_email_from_custom = $infohash['admin_email_from_custom'];
	$admin_email_reply_to = $infohash['admin_email_reply_to'];
	$admin_email_reply_to_custom = $infohash['admin_email_reply_to_custom'];
	$user_email_subject = $infohash['user_email_subject'];
	$user_email_from = $infohash['user_email_from'];
	$user_email_from_custom = $infohash['user_email_from_custom'];
	$user_email_reply_to = $infohash['user_email_reply_to'];
	$user_email_reply_to_custom = $infohash['user_email_reply_to_custom'];

	$email_format = $infohash['email_format'];

	$admin_text_email_template = $infohash['admin_text_email_template'];
	$admin_html_email_template = $infohash['admin_html_email_template'];
	$user_text_email_template  = $infohash['user_text_email_template'];
	$user_html_email_template  = $infohash['user_html_email_template'];

	$link = ft_db_connect();

	$query = "
		UPDATE {$g_table_prefix}forms
		SET    auto_email_admin = '$auto_email_admin',
					 auto_email_user  = '$auto_email_user',
					 user_email_field = '$user_email_field',
					 user_first_name_field = '$user_first_name_field',
					 user_last_name_field = '$user_last_name_field',
					 admin_email_subject = '$admin_email_subject',
					 admin_email_from = '$admin_email_from',
					 admin_email_from_custom = '$admin_email_from_custom',
					 admin_email_reply_to = '$admin_email_reply_to',
					 admin_email_reply_to_custom = '$admin_email_reply_to_custom',
					 user_email_subject = '$user_email_subject',
					 user_email_from = '$user_email_from',
					 user_email_from_custom = '$user_email_from_custom',
					 user_email_reply_to = '$user_email_reply_to',
					 user_email_reply_to_custom = '$user_email_reply_to_custom',
					 include_admin_email = '$include_admin_email',
					 email_format = '$email_format',
					 admin_text_email_template = '$admin_text_email_template',
					 admin_html_email_template = '$admin_html_email_template',
					 user_text_email_template  = '$user_text_email_template',
					 user_html_email_template  = '$user_html_email_template'
		WHERE  form_id = $form_id
					 ";

	$result = mysql_query($query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());


	ft_db_disconnect($link);

	return array($success, $message);
}


/**
 * Called by administrators; updates the content stored on the Advanced tab in the Edit
 * Form pages.
 *
 * stores / updates the following info about a form template:
 *           - include on redirect<br/>
 *           - form field<br/>
 *           - field type<br/>
 *           - database column
 * @param integer $infohash A hash containing the contents of the Edit Form Advanced tab.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_update_form_advanced_tab($infohash)
{
	global $g_debug, $g_table_prefix, $LANG;

	$field_info = array();
	$form_id = $infohash['form_id'];


	// loop through $infohash and for each field_X_order values, store the various values
	while (list($key, $val) = each($infohash))
	{
		// find the field id (field type is always required)
		preg_match("/^field_(\d+)_type$/", $key, $match);

		if (!empty($match[1]))
		{
			$field_id = $match[1];

			// is this field to be included on redirect or not?
			if (isset($infohash["field_{$field_id}_include_on_redirect"]) && !empty($infohash["field_{$field_id}_include_on_redirect"]))
				$include_on_redirect = "yes";
			else
				$include_on_redirect = "no";

			// form field name
			if (isset($infohash["field_{$field_id}_name"]))
				$form_field_name = $infohash["field_{$field_id}_name"];
			else
				$form_field_name = "";

			// field type (size)
			if (isset($infohash["field_{$field_id}_type"]))
				$type = $infohash["field_{$field_id}_type"];
			else
				$type = "";

			// field data type (string/number)
			if (isset($infohash["field_{$field_id}_data_type"]))
				$data_type = $infohash["field_{$field_id}_data_type"];
			else
				$data_type = "";

			// form field name
			if (isset($infohash["col_{$field_id}_name"]))
				$col_name = $infohash["col_{$field_id}_name"];
			else
				$col_name = "";

			$field_info[] = array($field_id, $include_on_redirect, $form_field_name, $type, $data_type, $col_name);
		}
	}
	reset($infohash);

	// if the form is complete, update the actual form table
	$form_info = ft_get_form($form_id);
	if ($form_info['is_complete'] == "yes")
	{
		// update each db column in turn
		foreach ($field_info as $field)
		{
			// ignore system fields
			if ($field[3] == "system")
				continue;

			$old_field_info = ft_get_form_field($field[0]);

			// if any physical aspect of the form (column name, field type) needs to be changed, change it
			if (($old_field_info['col_name'] != $field[5]) || ($old_field_info['field_size'] != $field[3]))
			{
				$new_field_size = "";
				switch ($field[3])
				{
					case "tiny":       $new_field_size = "VARCHAR(5)";   break;
					case "small":      $new_field_size = "VARCHAR(20)";  break;
					case "medium":     $new_field_size = "VARCHAR(255)"; break;
					case "large":      $new_field_size = "BLOB";         break;
					case "very_large": $new_field_size = "MEDIUMBLOB";   break;
					default:           $new_field_size = "VARCHAR(255)"; break;
				}

				list ($is_success, $err_message) = _ft_alter_table_column("{$g_table_prefix}form_{$form_id}", $old_field_info['col_name'], $field[5], $new_field_size);

				// if the alter message didn't work, return with an error message
				if (!$is_success)
				{
					$success = false;
					$message = $LANG["validation_db_not_updated_invalid_input"];
					if ($g_debug) $message .= " \"$err_message\"";
					return array($success, $message);
				}

				// update this filter with the new column name
				if ($old_field_info['col_name'] != $field[5])
					ft_update_filter_sql($field[0], $field[5]);
			}
		}
	}

	// lastly, update the form template table values
	foreach ($field_info as $field)
	{
		if ($field[3] == "system")
		{
			$query = "
				UPDATE {$g_table_prefix}form_templates
				SET    include_on_redirect = '{$field[1]}'
				WHERE  field_id = {$field[0]}
									";
		}
		else
		{
			$query = "
				UPDATE {$g_table_prefix}form_templates
				SET    include_on_redirect = '{$field[1]}',
							 field_name = '{$field[2]}',
							 field_size = '{$field[3]}',
							 data_type  = '{$field[4]}',
							 col_name   = '{$field[5]}'
				WHERE  field_id = {$field[0]}
									";
		}

		$link = ft_db_connect();
		mysql_query($query)
			or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());

		ft_db_disconnect($link);
	}

	return array(true, $LANG["notify_fields_updated"]);
}


/**
 * Adds/updates all options for a given field.
 *
 * @param integer $infohash A hash containing the contents of the Edit Form Advanced tab.
 * @param integer $field_id The unique field ID.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_update_field_options($infohash, $field_id)
{
	global $g_table_prefix, $g_debug, $LANG;

	$link = ft_db_connect();

	// remove old field options
	$query = mysql_query("
		DELETE FROM {$g_table_prefix}field_options
		WHERE  field_id = $field_id
					 ");

	$num_options = $infohash['num_options'];
	$option_orientation = $infohash['option_orientation'];

	for ($row=1; $row<=$num_options; $row++)
	{
		$option_value = $infohash['value_' . $row];
		$option_name  = $infohash['text_' . $row];

		if (!empty($option_name))
		{
			mysql_query("
				INSERT INTO {$g_table_prefix}field_options (field_id, option_value, option_name, option_order)
				VALUES ('$field_id', '$option_value', '$option_name', $row)
									");
		}
	}

	if (!empty($option_orientation))
	{
		mysql_query("
			UPDATE {$g_table_prefix}form_templates
			SET    option_orientation = '{$infohash['option_orientation']}'
			WHERE  field_id = '$field_id'
								");
	}

	ft_db_disconnect($link);

	return array(true, $LANG["notify_form_field_options_updated"]);
}


/**
 * Adds/updates all options for a file.
 *
 * If the upload folder is invalid, it returns an error.
 *
 * @param integer $infohash A hash containing the contents of the Edit Form Advanced tab.
 * @param integer $field_id The unique field ID.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_update_field_file_settings($infohash, $field_id)
{
	global $g_table_prefix, $g_debug, $LANG;

	$success = true;
	$message = $LANG["notify_file_upload_settings_updated"];

	// extract and clean form values
	$form_id = $infohash['form_id'];
	$file_upload_dir = rtrim(trim($infohash['file_upload_dir']), "/\\");
	$file_upload_url = rtrim(trim($infohash['file_upload_url']), "/\\");
	$file_upload_max_size = $infohash['file_upload_max_size'];
	$file_upload_types = $infohash['file_upload_types'];

	$original_file_upload_dir = $infohash['original_file_upload_dir'];
	$wants_to_move_files = (isset($infohash['move_files']) && $infohash['move_files'] == "yes") ? true : false;

	$link = ft_db_connect();
	$query = "
		UPDATE {$g_table_prefix}form_templates
		SET    file_upload_dir = '$file_upload_dir',
					 file_upload_url = '$file_upload_url',
					 file_upload_max_size = '$file_upload_max_size',
					 file_upload_types = '$file_upload_types'
		WHERE  field_id = $field_id
					 ";

	$result = mysql_query($query);
	ft_db_disconnect($link);

	if (!$result)
	{
		$success = false;
		$message = $LANG["notify_file_upload_settings_not_updated"];
		if ($g_debug) $message .= __FUNCTION__ . ", failed query: " . mysql_error();
		return array($success, $message);
	}


	$link = ft_db_connect();

	// check the folder was valid
	list($valid_folder, $folder_message) = ft_check_upload_folder($file_upload_dir);
	if (!$valid_folder)
		return array($valid_folder, $folder_message);

	// if necessary, move all files to the new upload folder
	if (($original_file_upload_dir != $file_upload_dir) && $wants_to_move_files)
	{
		// get all the filenames associated with this field
		$filename_hash = ft_get_uploaded_filenames($form_id, $field_id);
		$filenames = array_values($filename_hash);

		if (!empty($filenames))
		{
			$problem_files = array();

			$field_info = get_form_field($field_id);
			$database_column_name = $field_info['col_name'];

			foreach ($filename_hash as $sub_id => $file)
			{
				$submission_id = $filename_hash[$sub_id];

				// check that a file with this name doesn't already exist in the new folder. If it does,
				// find a unique filename, move it, and update the database
				$new_filename = $file;
				if (file_exists("$file_upload_dir/$file"))
					$new_filename = check_duplicate_filename($file_upload_dir, $file);

				// now move the file. Record any problems that occur
				if (!rename("$original_file_upload_dir/$file", "$file_upload_dir/$new_filename"))
					$problem_files[] = $file;
				else
				{
					// if the filename has changed, update the database record
					if ($file != $new_filename)
					{
						$query = "
							UPDATE {$g_table_prefix}form_{$form_id}
							SET    $database_column_name = '$new_filename'
							WHERE  submission_id = $submission_id
										 ";
						$result = mysql_query($query);
					}
				}
			}

			if (empty($problem_files))
				return array(true, $LANG["notify_file_upload_settings_updated_files_moved"]);
			else
			{
				$message = $LANG["notify_file_upload_settings_updated_files_not_moved"] . "<br /><br />";
				array_walk($problem_files, create_function('&$el', '$el = "&bull;&nbsp; " . $el;'));
				$message .= join("<br />", $problem_files);
				$message .= "<br />" . $LANG["text_reason_files_not_moved"];
				return array(false, $message);
			}
		}
	}

	ft_db_disconnect($link);

	return array($success, $message);
}


/**
 * Adds new form field(s) for storage in the database. This function was updated in 1.4.6 to allow
 * for adding multiple fields in one go. This function ignores all fields that don't include a form
 * field name.
 *
 * Note: this function could use some real improvements to the data validation and error handling.
 * Currently it mostly relies on the JS validation in the form page. This isn't such a huge sin, however,
 * since the Add Fields page is written in entirely all javascript - so it's exceedingly unlikely that
 * this function will receive invalid data. Still, it should be re-examined when I get around to
 * standardizing the error message handling.
 *
 * @param integer $infohash A hash containing the contents of the Edit Form Advanced -> Add Fields page.
 * @param integer $form_id The unique form ID.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_add_form_fields($infohash, $form_id)
{
	global $g_debug, $g_table_prefix, $LANG;

	$success = true;
	$message = "";

	// grab the "global" values
	$auto_generate_col_names = isset($infohash["auto_generate_col_names"]) ? true : false;
	$num_rows                = isset($infohash["num_fields"]) ? $infohash["num_fields"] : 0;

	// if for some reason there are no rows, just return
	if ($num_rows == 0)
		return;

	// find out how many fields there are already created for this form. This is used for the field order
	$form_template = ft_get_form_template($form_id);
	$form_info = ft_get_form($form_id);
	$num_fields = mysql_num_rows($form_template);
	$order = $num_fields + 1;

	// if we're auto-generating the column names, get a list of unique strings (of length $num_fields)
	// in preparation for use
	$unique_col_names = array();
	if ($auto_generate_col_names)
	{
		$existing_col_names = array();
		while ($field_info = mysql_fetch_assoc($form_template))
			$existing_col_names[] = $field_info["col_name"];

		// since the database column names can be named virtually ANYTHING, we're simply going to
		// generate as many column names as are needed of the form col_X - where X is a number starting
		// from 1. I really can't think of another "nice" way to name them...
		$curr_num = 1;
		while (count($unique_col_names) < $num_rows)
		{
			 if (!in_array("col_$curr_num", $existing_col_names))
				 $unique_col_names[] = "col_$curr_num";

			 $curr_num++;
		}
	}


	// loop through $infohash and, if the data is valid, add each form field
	for ($i=1; $i<=$num_rows; $i++)
	{
		// ignore any blank / deleted fields
		if (!isset($infohash["field_name_$i"]) || empty($infohash["field_name_$i"]))
			continue;

		// extract values for use
		$include_on_redirect = isset($infohash["include_on_redirect_$i"]) ? "yes" : "no";
		$field_name    = $infohash["field_name_$i"];
		$field_title   = $infohash["field_title_$i"];
		$field_size    = $infohash["field_size_$i"];
		$data_type     = $infohash["data_type_$i"];

		// figure out the column name
		$col_name = "";
		if ($auto_generate_col_names)
		{
			// grab the next free unique column name
			$col_name = array_shift($unique_col_names);
		}
		else
		{
			// this should never happen, but check for it anyway.
			if (!isset($infohash["col_name_$i"]) || empty($infohash["col_name_$i"]))
				continue;

			$col_name = $infohash["col_name_$i"];
		}

		// add the new field to form_templates
		$link = ft_db_connect();
		$query = "INSERT INTO {$g_table_prefix}form_templates (form_id, field_name, field_size, field_type,
				data_type, field_title, col_name, list_order, admin_display, include_on_redirect)
			VALUES ($form_id, '$field_name', '$field_size', 'other',
				'$data_type', '$field_title', '$col_name', $order, 'no', '$include_on_redirect')";

		$result = mysql_query($query)
			or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());

		ft_db_disconnect($link);


		// if the form already exists, add the field
		if ($form_info['is_complete'] == "yes")
		{
			$new_field_size = "";

			switch ($field_size)
			{
				case "tiny":       $new_field_size = "VARCHAR(5)";   break;
				case "small":      $new_field_size = "VARCHAR(20)";  break;
				case "medium":     $new_field_size = "VARCHAR(255)"; break;
				case "large":      $new_field_size = "BLOB";         break;
				case "very_large": $new_field_size = "MEDIUMBLOB";   break;
				default:           $new_field_size = "VARCHAR(255)"; break;
			}

			list ($is_success, $err_message) = _ft_add_table_column("{$g_table_prefix}form_{$form_id}", $col_name, $new_field_size);

			// if the alter message didn't work, return with an error message
			if (!$is_success)
			{
				$success = false;

				$replacement_info = array("fieldname" => $field_name);
				$message = ft_replace_placeholders($LANG["notify_form_field_not_added"], $replacement_info);
				if ($g_debug) $message .= " \"$err_message\"";
				return array($success, $message);
			}
		}
	}

	return array($success, $message);
}


/**
 * Deletes unwanted form fields.
 *
 * Called in two instances:
 * 1. by administrator during form building process.<br/>
 * 2. by administrator during form editing process.<br/>
 *
 * If the field being removed is a FILE field, it checks to see if the auto_delete_submission_files
 * setting for this form is set to "yes". If it is, it removes ALL files that were associated with
 * this field. Otherwise, it leaves all files intact.
 *
 * @param integer $infohash A hash containing the contents of the Edit Form Advanced tab.
 * @return array Returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function ft_delete_form_fields($infohash, $form_id)
{
	global $g_table_prefix, $LANG;

	// return values
	$success = true;
	$message = "";

	// find out if the form exists
	$form_info = ft_get_form($form_id);
	$form_table_exists = ($form_info['is_complete'] == 'yes') ? true : false;

	$remove_field_ids = array();
	$files_to_remove = array();
	while (list($key, $value) = each($infohash))
	{
		if (preg_match("/^field_(\d+)_remove$/", $key, $match))
		{
			$field_id = $match[1];
			$remove_field_ids[] = $field_id;

			$old_field_info = ft_get_form_field($field_id);
			$file_upload_dir = $old_field_info['file_upload_dir'];

			// now get ALL files that have been uploaded through this field, so we can remove them if need be
			$filename_hash = ft_get_uploaded_filenames($form_id, $field_id);
			$field_filenames = array_values($filename_hash);

			if (!empty($field_filenames))
			{
				for ($i=0; $i<count($field_filenames); $i++)
					$files_to_remove[] = "$file_upload_dir/{$field_filenames[$i]}";
			}

			$link = ft_db_connect();
			mysql_query("
				DELETE FROM {$g_table_prefix}form_templates
				WHERE  field_id = $field_id
									");

			// delete any filters that are set up on this form field
			mysql_query("
				DELETE FROM {$g_table_prefix}client_filters
				WHERE  field_id = $field_id
									");

			// if the form already exists, then remove the column
			if ($form_table_exists)
			{
				$drop_column = $old_field_info['col_name'];
				mysql_query("ALTER TABLE {$g_table_prefix}form_$form_id DROP $drop_column");

				// if the current sort order was based on this field, re-set it to the submission_date
				if ($drop_column == $form_info['default_sort_field'])
				{
					mysql_query("
						UPDATE {$g_table_prefix}forms
						SET     default_sort_field = 'submission_date'
											");
				}
			}

			ft_db_disconnect($link);
		}
	}

	// if required, remove all associated files
	if ($form_info['auto_delete_submission_files'] == "yes")
	{
		// delete them all
		if (!empty($files_to_remove))
		{
			foreach ($files_to_remove as $file)
				@unlink($file);
		}
	}

	// build return message
	if (count($remove_field_ids) > 1)
		$message = $LANG["notify_form_fields_removed"];
	else
		$message = $LANG["notify_form_field_removed"];

	return array($success, $message);
}


/**
 * Reorders template fields and updates the corresponding column name field.
 *
 * Called by administrator in Add Form Step 3 page and on Advanced tab when editing. For the special
 * system fields (submission ID, submission Date and IP address), we don't want to override the
 * default DB table column names. To prevent this, we let this function know which fields are system
 * fields by passing hidden values:<br/>
 *
 * <input type="hidden" name="field_X_system" value="1" />
 *
 * @param integer $infohash A hash containing the contents of the Edit Form Advanced tab.
 * @param boolean $set_default_form_field_names A hash containing the contents of the Edit Form Advanced tab.
 */
function ft_reorder_form_fields($infohash, $form_id, $set_default_form_field_names = false)
{
	global $g_table_prefix;

	$link = ft_db_connect();
	$new_order = array();

	// loop through $infohash and for each field_X_order values, log the new order
	while (list($key, $val) = each($infohash))
	{
		// find the field id
		preg_match("/^field_(\d+)_order$/", $key, $match);

		if (!empty($match[1]))
		{
			$field_id = $match[1];

			// update the $account_order
			$new_order[$field_id] = $val;
		}
	}
	asort($new_order);
	reset($infohash);

	// now loop through the correct_order array and update
	$i = 1;
	while (list($key, $value) = each($new_order))
	{
		$col_name_qry = "";
		if ($set_default_form_field_names)
		{
			if (!isset($infohash["field_{$key}_system"]))
				$col_name_qry = ", col_name = 'col_$i' ";
		}

		mysql_query("
			UPDATE {$g_table_prefix}form_templates
			SET    list_order = $i
						 $col_name_qry
			WHERE  field_id = $key AND
						 form_id = $form_id
								");
			$i++;
	}

	ft_db_disconnect($link);
}


/**
 * Completely removes a form from the database.
 *
 * Includes an optional parameter to remove all files that were uploaded through file fields in
 * the form; defaulted to FALSE.
 *
 * @param integer $form_id the unique form ID
 * @param boolean $remove_associated_files A boolean indicating whether or not all files that were
 *              uploaded via file fields in this form should be removed as well.
 */
function ft_delete_form($form_id, $remove_associated_files = false)
{
	global $g_table_prefix;


	// if required, generate an array
	if ($remove_associated_files)
	{
		// get the names and URLs of all uploaded files
		$form_template = ft_get_form_template($form_id);
		$file_field_hash = array(); // field_id => upload folder path
		while ($field = mysql_fetch_assoc($form_template))
		{
			if ($field['field_type'] == "file")
				$file_field_hash[$field['field_id']] = $field['file_upload_dir'];
		}

		// now determine all files + paths and remove them
		if (!empty($file_field_hash))
		{
			foreach ($file_field_hash as $field_id => $upload_dir)
			{
				$uploaded_files = ft_get_uploaded_filenames($form_id, $field_id);

				if (!empty($uploaded_files))
				{
					foreach ($uploaded_files as $file)
						@unlink("$upload_dir/$file");
				}
			}
		}
	}

	$link = ft_db_connect();

	// remove the table
	$query = "DROP TABLE IF EXISTS {$g_table_prefix}form_$form_id";
	mysql_query($query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());



	// get a list of all field IDs in this form
	$query = "
		SELECT field_id, field_type
		FROM   {$g_table_prefix}form_templates
		WHERE  form_id = $form_id
						";
	$field_id_query = mysql_query($query);

	// remove any reference to the form in form_templates
	$query = "
		DELETE FROM {$g_table_prefix}form_templates
		WHERE  form_id = $form_id
					 ";
	mysql_query($query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());

	// remove any reference to the form in forms table
	$query = mysql_query("
		DELETE FROM {$g_table_prefix}forms
		WHERE  form_id = $form_id
					 ");

	// remove any filters
	$query = mysql_query("
		DELETE FROM {$g_table_prefix}client_filters
		WHERE  form_id = $form_id
					 ");

	// remove any fields in field_options table
	$field_ids = array();
	while ($field_info = mysql_fetch_array($field_id_query))
	{
		if ($field_info['field_type'] == "select" || $field_info['field_type'] == "multi-select" ||
				$field_info['field_type'] == "radio-buttons" || $field_info['field_type'] == "checkboxes")
			$field_ids[] = $field_info[0];
	}
	if (!empty($field_ids))
	{
		foreach ($field_ids as $field_id)
		{
			$query = "
				DELETE FROM {$g_table_prefix}form_templates
				WHERE  field_id = $field_id
							 ";

			// delete any filters associated with this field
			mysql_query("
				DELETE FROM {$g_table_prefix}client_filters
				WHERE  field_id = $field_id
									");

			mysql_query($query);
		}
	}

	ft_db_disconnect($link);
}


/**
 * Called by client accounts, allowing them to update the num_submissions_per_page and auto email
 * settings.
 *
 * @param array $infohash A hash containing the various form values to update.
 */
function ft_client_update_form_settings($infohash)
{
	global $g_table_prefix, $LANG;

	$success = true;
	$message = $LANG["notify_form_settings_updated"];

	// validate $infohash fields
	$rules = array();
	$rules[] = "required,form_id,{$LANG["validation_no_form_id"]}";
	$rules[] = "required,is_active,{$LANG["validation_is_form_active"]}";
	$rules[] = "required,num_submissions_per_page,{$LANG["validation_no_num_submissions_per_page"]}";
	$rules[] = "digits_only,num_submissions_per_page,{$LANG["validation_invalid_num_submissions_per_page"]}";
	$errors = validate_fields($infohash, $rules);


	$link = ft_db_connect();
	$query = "
			UPDATE {$g_table_prefix}forms
			SET    is_active = '{$infohash['is_active']}',
						 auto_email_admin = '{$infohash['auto_email_admin']}',
						 auto_email_user = '{$infohash['auto_email_user']}',
						 num_submissions_per_page = '{$infohash['num_submissions_per_page']}',
						 printer_friendly_format = '{$infohash['printer_friendly_format']}',
						 hide_printer_friendly_empty_fields = '{$infohash['hide_empty_fields']}'
			WHERE  form_id = '{$infohash['form_id']}'
						";

	$result = mysql_query($query)
		or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());

	ft_db_disconnect($link);

	if (!$result)
	{
		$success = false;
		$message = $LANG["notify_form_not_updated_notify_admin"];
		return array($success, $message);
	}

	return array($success, $message);
}


/**
 * Retrieves all information about single form; all associated client information is stored in the
 * user_info key, as an array. Client filters are stored in the [user_info][filters] array.
 *
 * @param integer $form_id the unique form ID
 * @return array A hash of form information - and any corresponding information about client accounts
 * accounts that are associated with the form.
 */
function ft_get_form($form_id)
{
	global $g_table_prefix;

	$link = ft_db_connect();
	$query = "SELECT * FROM {$g_table_prefix}forms WHERE form_id = $form_id";
	$form_query = mysql_query($query);

	$user_query = mysql_query("
		SELECT *
		FROM   {$g_table_prefix}client_forms cf, {$g_table_prefix}user_accounts ua
		WHERE  cf.form_id = $form_id
		AND    cf.user_id = ua.user_id
					 ");

	$user_info = array();
	while ($result = mysql_fetch_assoc($user_query))
	{
		$user_id = $result["user_id"];

		$query = "SELECT * FROM {$g_table_prefix}client_filters WHERE user_id = $user_id";
		$filter_query = mysql_query($query)
			or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>: <i>$query</i>", mysql_error());

		$filters = array();
		while ($filter_info = mysql_fetch_assoc($filter_query))
			$filters[] = $filter_info;

		$result["filters"] = $filters;
		$user_info[] = $result;
	}

	$form_info = mysql_fetch_assoc($form_query);
	$form_info['user_info'] = $user_info;

	ft_db_disconnect($link);

	return $form_info;
}


/**
 * Retrieves all template information about a form.
 *
 * Note: this function returns a RESOURCE to a query, rather than a standard data structure, since
 * the form template is stored in multiple rows. This sucks, and will be deprecated: no more
 * MySQL resources will be returned in future versions.
 *
 * @param integer $form_id the unique form ID
 * @return resource The MySQL resource for the SELECT query.
 */
function ft_get_form_template($form_id)
{
	global $g_table_prefix;

	$link = ft_db_connect();

	$query = mysql_query("
		SELECT *
		FROM   {$g_table_prefix}form_templates
		WHERE  form_id = '$form_id'
		ORDER BY list_order
					 ");

	ft_db_disconnect($link);

	return $query;
}


/**
 * Retrieves all information about a specific form template field.
 *
 * @param integer $field_id the unique field ID.
 * @return array A hash of information about this field.
 */
function ft_get_form_field($field_id)
{
	global $g_table_prefix;

	$link = ft_db_connect();

	$query = mysql_query("
		SELECT *
		FROM   {$g_table_prefix}form_templates
		WHERE  field_id = '$field_id'
					 ");

	$info = mysql_fetch_assoc($query);

	ft_db_disconnect($link);

	return $info;
}


/**
 * A very specific getter function to retrieve everything about a form field from
 * the based on the database column name. This is used in the ft_search_submissions
 * function.
 *
 * @param integer $form_id
 * @param string $col_name
 */
function ft_get_form_field_by_colname($form_id, $col_name)
{
	global $g_table_prefix;

	$link = ft_db_connect();
	$query = mysql_query("
		SELECT *
		FROM   {$g_table_prefix}form_templates
		WHERE  form_id = $form_id AND
					 col_name = '$col_name'
		LIMIT 1
			");
	ft_db_disconnect($link);

	$infohash = array();
	while ($row = mysql_fetch_assoc($query))
		$infohash = $row;

	return $infohash;
}


/**
 * Returns all the column names for a particular form.
 *
 * @param integer $form_id the unique form ID.
 * @return array A hash of form: [DB column name] => [column display name]. If the database
 * column doesn't have a display name (like with submission_id) the value is set to the same as
 * the key.
 */
function ft_get_form_column_names($form_id)
{
	global $g_table_prefix;

	$link = ft_db_connect();
	$result = mysql_query("
		SELECT *
		FROM   {$g_table_prefix}form_$form_id
			 ");

	// generate hash of [col name] => [column display name]
	$custom_col_names = array();
	$template_info = ft_get_form_template($form_id);
	while ($row = mysql_fetch_assoc($template_info))
		$custom_col_names[$row['col_name']] = $row['field_title'];


	// now build a hash of ALL db column info
	$db_col_info = array();

	$col = 0;
	while ($col < mysql_num_fields($result))
	{
		// get meta information about this field
		$meta = mysql_fetch_field($result, $col);

		// if this column name is defined in $custom_col_name, use it's hash value. Otherwise, just set
		// it's value to the column name (like with submission_date, ip_address).
		if (array_key_exists($meta->name, $custom_col_names))
			$db_col_info[$meta->name] = $custom_col_names[$meta->name];
		else
			$db_col_info[$meta->name] = $meta->name;
		$col++;
	}

	ft_db_disconnect($link);

	return $db_col_info;
}


/**
 * Returns all the field options for a particular multi-select field.
 *
 * @param integer $form_id the unique field ID.
 * @return resource The MySQL resource of the SELECT query (to be deprecated)
 */
function ft_get_field_options($field_id)
{
	global $g_table_prefix;

	$link = ft_db_connect();

	$result = mysql_query("
		SELECT *
		FROM   {$g_table_prefix}field_options
		WHERE  field_id = $field_id
		ORDER BY option_order
			 ");

	ft_db_disconnect($link);

	return $result;
}


/**
 * Returns all files associated with a particular form field.
 *
 * @param integer $form_id the unique form ID.
 * @param integer $field_id the unique field ID.
 * @return resource The MySQL resource of the SELECT query (to be deprecated)
 */
function ft_get_uploaded_filenames($form_id, $field_id)
{
	global $g_table_prefix;

	// get the column name for this field
	$field_info = ft_get_form_field($field_id);
	$col_name   = $field_info['col_name'];

	// if col_name is empty, the field doesn't exist - so the user is probably just setting up the form
	// just return an empty array.
	if (empty($col_name))
		return array();

	$query = "
		SELECT submission_id, $col_name
		FROM   {$g_table_prefix}form_{$form_id}
		WHERE  $col_name != ''
					 ";

	$link = ft_db_connect();
	$result = mysql_query($query);
	ft_db_disconnect($link);

	$filename_hash = array();
	while ($record = mysql_fetch_array($result))
	{
		if (!empty($record[1]))
			$filename_hash[$record[0]] = $record[1];
	}

	return $filename_hash;
}


/**
 * Calculates and returns a hash of value->label pairs for the search form dropdown list.
 *
 * It examines sessions to find the FIRST submission for this form. This information is stored in
 * $_SESSION["ft"]["form_X_first_submission_date"] and is a MySQL datetime.
 *
 * As of 1.5.0, this function now returns localized strings for the months, depending on the users
 * language.
 *
 * @param integer $form_id the unique form ID.
 * @return array Based on the info in sessions, it returns an array of search months, most recent
 *             first.
 *
 *             For example, if the current date was Jan 2007 and the form was created in Nov 2006,
 *             it returns:<br/>
 *             "1_2007" => "January 2007",<br/>
 *             "2_2007" => "December 2006",<br/>
 *             "3_2007" => "November 2006"<br/>
 *             If the form creation date was earlier, it would return more months - UP TO 12 months,
 *             no more.
 *
 *             If there are no results, it returns an empty array.
 */
function ft_get_search_months($form_id)
{
	$first_submission_date = $_SESSION["ft"]["form_{$form_id}_first_submission_date"];

	// if there is no submission date to base this on, just return. This would happen with
	// forms that don't have any submissions
	if (empty($first_submission_date))
		return array();

	$date_info = split(" ", $first_submission_date);
	list($year, $month, $day) = split("-", $date_info[0]);
	$unix_date_created = mktime(0, 0, 0, $month, 0, $year);
	$first_submission_start_month_unixtime = mktime(0, 0, 0, $month, 0, $year);
	$current_unixtime = date("U");
	$seconds_in_month = 30 * 24 * 60 * 60;

	$count = 0;

	$search_months = array();
	while ($count < 12 && $current_unixtime > $first_submission_start_month_unixtime)
	{
		// get the localized date info for current_unixtime
		$date_info = ft_get_date(0, ft_get_current_datetime($current_unixtime), "n F Y");
		list($month, $label_month, $year) = split(" ", $date_info);
		$search_months["{$month}_$year"] = "$label_month $year";

		$current_unixtime -= $seconds_in_month;
		$count++;
	}

	return $search_months;
}


// ---------------------------------------- helpers -----------------------------------------------


/**
 * Helper function to change the name and type of an existing MySQL table.
 *
 * @param string $table The name of the table to alter.
 * @param string $old_col_name The old column name.
 * @param string $new_col_name The new column name.
 * @param string $col_type The new column data type.
 * @return array Array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function _ft_alter_table_column($table, $old_col_name, $new_col_name, $col_type)
{
	global $g_table_prefix;

	$success = true;
	$message = "";

	$link = ft_db_connect();
	$result = mysql_query("
		ALTER TABLE $table
		CHANGE      $old_col_name $new_col_name $col_type
					 ");

	if (!$result)
	{
		$success = false;
		$message = mysql_error();
	}

	ft_db_disconnect($link);

	return array($success, $message);
}


/**
 * Helper function to add a new data column the end of a table.
 *
 * @param string $table The name of the table to alter.
 * @param string $col_name The new column name.
 * @param string $col_type The new column data type.
 * @return array Array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 */
function _ft_add_table_column($table, $col_name, $col_type)
{
	$success = true;
	$message = "";

	$link = ft_db_connect();
	$result = mysql_query("
		ALTER TABLE $table
		ADD         $col_name $col_type
					 ");

	if (!$result)
	{
		$success = false;
		$message = mysql_error();
	}

	ft_db_disconnect($link);

	return array($success, $message);
}


/**
 * Helper function which stores certain information about a form - or all forms - in sessions for
 * the current user.
 *
 * It stores:<br />
 * $_SESSION["ft"]["form_X_num_submissions"] = the number of submissions in a form<br/>
 * $_SESSION["ft"]["form_X_first_submission_date"] = the first submission date (mySQL datetime)
 *
 * @param integer $form_id A specific form ID. If blank, stores all form values in sessions.
 */
function _ft_load_form_values($form_id = "")
{
	global $g_table_prefix;

	$where_clause = "";
	if (!empty($form_id))
		$where_clause = "AND form_id = $form_id";

	$link = ft_db_connect();
	$query = mysql_query("
		SELECT form_id
		FROM   {$g_table_prefix}forms
		WHERE  is_complete = 'yes'
		$where_clause
					 ");

	// loop through all forms, extract the submission count and first submission date
	while ($form_info = mysql_fetch_assoc($query))
	{
		$form_id = $form_info["form_id"];

		// if there's a filter set up for this form, take it into account
		$filter_clause = "";
		if (isset($_SESSION["ft"]["form_{$form_id}_filters"]))
			$filter_clause = "AND " . join(" AND ", $_SESSION["ft"]["form_{$form_id}_filters"]);

		$count_query = mysql_query("
			SELECT count(*)
			FROM   {$g_table_prefix}form_$form_id
			WHERE  is_finalized = 'yes'
			$filter_clause
				")
				or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>", mysql_error());

		$info = mysql_fetch_row($count_query);
		$_SESSION["ft"]["form_{$form_id}_num_submissions"] = $info[0];

		$first_date_query = mysql_query("
			SELECT submission_date
			FROM   {$g_table_prefix}form_$form_id
			WHERE  is_finalized = 'yes'
			$filter_clause
			ORDER BY submission_date ASC
			LIMIT 1
				")
				or ft_handle_error("Failed query in <b>" . __FUNCTION__ . "</b>", mysql_error());
		$info = mysql_fetch_row($first_date_query);
		$_SESSION["ft"]["form_{$form_id}_first_submission_date"] = $info[0];
	}

	ft_db_disconnect($link);
}

