<?php

/**
 * This file defines all functions related to the Form Tools emailing options.
 *
 * @copyright Benjamin Keen 2007
 * @author Benjamin Keen <ben.keen@gmail.com>
 * @package 1-5-1
 * @subpackage Emails
 */


// -------------------------------------------------------------------------------------------------


/**
 * Sends a test email to administrators as they are building their email templates - or just to
 * confirm the emails are working properly.
 *
 * @param array $infohash This parameter should be a hash (e.g. $_POST or $_GET) containing the
 *             various fields from the test email form.
 */
function ft_send_test_email($infohash)
{
	global $g_table_prefix, $LANG;

	$success = true;
	$message = $LANG["notify_your_email_sent"];

	// store the test email settings in sessions so the user doesn't have to re-enter them when
	// coming to and from the email page (not worth storing in DB)
	$_SESSION['email_template'] = $infohash['email_template'];
	$_SESSION['test_email_recipient'] = $infohash['test_email_recipient'];
	$_SESSION['test_email_data_source'] = $infohash['test_email_data_source'];
	$_SESSION['test_email_submission_id'] = $infohash['test_email_submission_id'];

	// retrieve the test email settings
	$form_id = $infohash['form_id'];
	$email_template = $infohash['email_template'];
	$test_email_recipient = $infohash['test_email_recipient'];
	$test_email_data_source = $infohash['test_email_data_source'];
	$test_email_submission_id = $infohash['test_email_submission_id'];


	$link = ft_db_connect();

	// get the submission ID and verify it's valid
	switch ($test_email_data_source)
	{
		case "random_submission":
			$result = mysql_query("SELECT submission_id FROM {$g_table_prefix}form_$form_id ORDER BY rand() LIMIT 1");
			$row = mysql_fetch_row($result);
			$submission_id = $row[0];
			break;

		case "submission_id":
			$result = mysql_query("SELECT count(*) FROM {$g_table_prefix}form_$form_id WHERE submission_id=$test_email_submission_id");
			$row = mysql_fetch_row($result);
			if ($row[0] != 1)
				return array(false, $LANG["notify_submission_id_not_found"]);
			else
				$submission_id = $test_email_submission_id;
			break;
	}
	ft_db_disconnect($link);

	// convert the email template string into a format understood by the email functions
	$str_converter = array(
		"admin"      => "both",
		"admin_text" => "text",
		"admin_HTML" => "html",
		"user"       => "both",
		"user_text"  => "text",
		"user_HTML"  => "html"
			 );

	// send the appropriate email
	if      ($email_template == "admin" || $email_template == "admin_text" || $email_template == "admin_HTML")
		$success = ft_send_email("admin", $form_id, $submission_id, $test_email_recipient, $str_converter[$email_template]);
	else if ($email_template == "user" || $email_template == "user_text" || $email_template == "user_HTML")
		$success = ft_send_email("user", $form_id, $submission_id, $test_email_recipient, $str_converter[$email_template]);

	if (!$success)
		$message = $LANG["notify_test_email_not_sent"];

	return array($success, $message);
}


/**
 * Sends an email to an administrator (or multiple administrators) or to the person who just entered
 * a form - for a given form submission. This function decides whether to use email templates or not
 * depending on their Program Settings choice.
 *
 * @param string $email_type A string: "admin" / "user"
 * @param integer $form_id The unique form ID
 * @param integer $submission_id The unique submission ID
 * @param string $override_recipient The email address of the recipient of this email. If empty, the
 *             function uses the content of the admin_email_recipients field. But if this parameter
 *             is included (e.g. with email tests for administrators), it overrides that value and
 *             sends the email to the address specified here instead.
 * @param string $email_template The email template with which to send the email. As with the
 *             recipient parameter, it uses the value from the email_format field by default, unless
 *             it is explicitly overridden here. Valid values: "both" (both HTML and text), "text"
 *             (text only), html (HTML only).
 */
function ft_send_email($email_type, $form_id, $submission_id, $override_recipient = "", $override_email_template = "")
{
	$form_info = ft_get_form($form_id);
	$settings  = ft_get_settings();
	$submission_info = ft_get_submission($form_id, $submission_id);

	// if not specified, use whatever email format is in the DB
	$email_template = strtolower($override_email_template);
	if (empty($override_email_template))
		$email_template = strtolower($form_info["email_format"]);

	// get the end-of-line character for this system
	$eol = _ft_get_email_eol_char();

	list($submission_first_name, $submission_last_name, $submission_email) = _ft_get_submission_email_headers($form_info, $submission_info);
	$from     = _ft_get_submission_from_field($form_info, $submission_first_name, $submission_last_name, $submission_email, $email_type, $settings);
	$reply_to = _ft_get_submission_reply_to_field($form_info, $submission_first_name, $submission_last_name, $submission_email, $email_type, $settings);

	// retrieve the placeholders and their substitutes
	$common_placeholders = _ft_get_placeholder_hash($form_id, $submission_id);

	// build the subject
	$original_subject = $form_info["{$email_type}_email_subject"];
	$subject          = _ft_parse_template($original_subject, $common_placeholders);

	// construct the email headers
	$headers = "MIME-Version: 1.0$eol"
					 . "From: $from$eol"
					 . "Reply-to: $reply_to$eol";
	$message = "";


	// this stores the pieces of the email(s) to be sent
	$email_info = array();

	// this keeps track of what clients can manage this submission (for account submission filters)
	$clients_may_manage_submission = array();


	// test email, or email sent to person who submitted the form (the "user")
	if (!empty($override_recipient) || $email_type == "user")
	{
		$recipient = (!empty($override_recipient)) ? $override_recipient : $submission_email;

		// BASIC emails
		if ($settings['emailing_method'] == "basic")
		{
			for ($i=0; $i<count($submission_info); $i++)
				$message .= $submission_info[$i]['field_title'] . ": " . stripslashes($submission_info[$i]['content']) . "\n";
		}

		// ADVANCED emails (email templates)
		else
		{
			switch ($email_template)
			{
				case "both":
					$text_content = _ft_parse_template($form_info["{$email_type}_text_email_template"], $common_placeholders);
					$HTML_content = _ft_parse_template($form_info["{$email_type}_html_email_template"], $common_placeholders, true);
					$headers .= _ft_get_multipart_message($HTML_content, $text_content, $eol);
					break;

				case "text":
					$message = _ft_parse_template($form_info["{$email_type}_text_email_template"], $common_placeholders);
					$headers .= "Content-type: text/plain;charset=UTF-8";
					break;

				case "html":
					$message = _ft_parse_template($form_info["{$email_type}_html_email_template"], $common_placeholders, true);
					$headers .= "Content-type: text/html; charset=UTF-8";
					break;
			}
		}

		// provided the recipient email wasn't empty, store the email info for emailing
		if (!empty($recipient))
			$email_info[] = array($recipient, $subject, $message, $headers);
	}


	// client emails
	else if ($email_type == "admin")
	{
		$user_info = $form_info['user_info'];

		// check this submission hasn't been filtered out for any of the clients
		$user_filters = array();
		foreach ($user_info as $curr_user_info)
		{
			$user_id = $curr_user_info["user_id"];
			$filters = array();

			// only add the filters that are assigned to this form
			foreach ($curr_user_info["filters"] as $filter)
			{
				if ($filter["form_id"] == $form_id)
					$filters[] = $filter["filter_sql"];
			}

			$user_filters[$user_id] = $filters;
		}
		$clients_may_manage_submission = _ft_check_clients_may_manage_submission($user_filters, $form_id, $submission_id);


		// BASIC emails
		if ($settings['emailing_method'] == "basic")
		{
			for ($i=0; $i<count($submission_info); $i++)
				$message .= $submission_info[$i]['field_title'] . ": " . stripslashes($submission_info[$i]['content']) . "\n";

			// generate an email for each user assigned to this form
			for ($i=0; $i<count($user_info); $i++)
			{
				if ($clients_may_manage_submission[$user_info[$i]["user_id"]])
					$email_info[] = array($user_info[$i]['email'], $subject, $message, $headers);
			}
		}

		// ADVANCED emails (email templates)
		else
		{
			// loop through each client assigned to this form, and generate an email for them
			for ($i=0; $i<count($user_info); $i++)
			{
				$client_placeholders = _ft_get_placeholder_hash($form_id, $submission_id, $user_info[$i]);

				switch ($email_template)
				{
					case "both":
						$text_content = _ft_parse_template($form_info["{$email_type}_text_email_template"], $client_placeholders);
						$HTML_content = _ft_parse_template($form_info["{$email_type}_html_email_template"], $client_placeholders, true);
						$headers .= _ft_get_multipart_message($HTML_content, $text_content, $eol);
						break;

					case "text":
						$message = _ft_parse_template($form_info["{$email_type}_text_email_template"], $client_placeholders);
						$headers .= "Content-type: text/plain; charset=UTF-8";
						break;

					case "html":
						$message = _ft_parse_template($form_info["{$email_type}_html_email_template"], $client_placeholders, true);
						$headers .= "Content-type: text/html; charset=UTF-8";
						break;
				}

				// only send an email to this client if they have an actual email address specified and
				if (!empty($user_info[$i]['email']) && $clients_may_manage_submission[$user_info[$i]["user_id"]])
					$email_info[] = array($user_info[$i]['email'], $subject, $message, $headers);

				// if required, send an additional email to the site administrator (but NEVER when just testing)
				if (empty($override_recipient) && $email_type == "admin" && $form_info['include_admin_email'] == "yes")
					$email_info[] = array($settings['administrator_email'], $subject, $message, $headers);
			}
		}
	}

	// send the email(s)
	$success = true;
	for ($i=0; $i<count($email_info); $i++)
	{
		$recipient = $email_info[$i][0];
		$subject   = $email_info[$i][1];
		$message   = $email_info[$i][2];
		$headers   = $email_info[$i][3];
		if (!mail($recipient, $subject, $message, $headers))
			$success = false;
	}

	return $success;
}


/**
 * Called on the "have a question?" client page. Sends an email to the administrator containing
 * whatever question this client has. Note: the field labels ("Name:", "Question:") are in whatever
 * language that the particular client is in.
 *
 * @param array the form contents
 * @return array returns array with indexes:<br/>
 *               [0]: true/false (success / failure)<br/>
 *               [1]: message string<br/>
 *  */
function ft_send_question_form($infohash)
{
	global $LANG;

	$settings = ft_get_settings();

	$recipient  = $settings['administrator_email'];
	$subject    = stripslashes($_POST['subject']);
	$from       = "{$LANG["word_from_c"]} " . stripslashes($_POST['email']);
	$reply_to   = "{$LANG["word_reply_to_c"]} " . stripslashes($_POST['email']);
	$headers    = "$from\r\n"
							. "$reply_to\r\n";

	$name       = stripslashes($_POST['name']);
	$question   = stripslashes($_POST['question']);

	$body = "$from\n"
				. "{$LANG["word_name_c"]} $name\n"
				. "{$LANG["word_question_c"]} $question\n";

	if (mail($recipient, $subject, $body, $headers))
	{
		$success = true;
		$message = $LANG['text_question_sent'];
	}
	else
	{
		$success = false;
		$message = $LANG['text_question_not_sent'];
	}

	return array($success, $message);
}


//------------------------------------ helper functions -------------------------------------------


/**
 * Pieces together the main email content from the text and HTML content, generated separately by
 * {@link http://www.formtools.org/developerdoc/1-4-6/Emails/_code---emails.php.html#function_parse_template
 * _parse_template}.
 *
 * @param string $HTML_content The HTML section of the email.
 * @param string $text_content The text section of the email.
 * @param string $eol_content The end-of-line character for this system.
 */
function _ft_get_multipart_message($HTML_content, $text_content, $eol)
{
	$boundary = md5(time());

	$content = "Content-Type: multipart/alternative; boundary = $boundary$eol"
					 . "\n--$boundary$eol"
					 . "Content-type: text/plain; charset=UTF-8$eol"
					 . $text_content
					 . "\n--$boundary$eol"
					 . "Content-type: text/html; charset=UTF-8$eol"
					 . $HTML_content;

	return $content;
}


/**
 * Generates the email content for a particular template - either HTML or text.
 *
 * This function takes an individual template (i.e. the text the administrator enters into one of
 * the email template fields, like "text" and "user") and a hash of placeholders names and
 * replacement values. It makes the necessary replacements, as well as executes any PHP found in
 * the original template (e.g. for the "omit empty fields" option). It returns the end result as a
 * string value.
 *
 * @param string $template The template for this form.
 * @param string $placeholder_hash The placeholder hash.
 * @param boolean $convert_nl_to_br For HTML formatted emails; converts newlines in textareas to
 *     <br />'s.
 * @return string the email content for a template.
 */
function _ft_parse_template($template, $placeholder_hash, $convert_nl_to_br = false)
{
	$patterns     = array_keys($placeholder_hash);
	$replacements = array_values($placeholder_hash);

	// if required, convert newlines to <br />'s
	if ($convert_nl_to_br)
	{
		$new_replacements = array();
		foreach ($replacements as $replacement)
			$new_replacements[] = nl2br($replacement);
		$replacements = $new_replacements;
	}
	$clean_replacements = array();

	// prevent dollar signs being executed as PHP variables [hack...]
	foreach ($replacements as $replacement)
		$clean_replacements[] = preg_replace("/[$]/", '%DOLLARSIGN%', $replacement);
	$content = preg_replace($patterns, $clean_replacements, $template);


	// $content now contains the content of the email, but any PHP in it has yet to be executed.
	// To execute it, we load it into a temporary file on the server, then eval() the result.
	// We use output buffering to store the raw HTML/text outputted by the PHP in the template
	// into a PHP variable to return.
	if (preg_match("/<\?php/", $content))
	{
		ob_start();
		$temp_file = tempnam("/tmp", "tempfile");
		$handle = fopen($temp_file, "w");
		fwrite($handle, $content);
		fclose($handle);
		$code_array = file($temp_file);
		$code = join("", $code_array);
		$code = str_replace('<'.'?php', '<'.'?', $code);
		$code = '?'.'>' . trim($code) . '<'.'?';
		eval($code);
		unlink($temp_file);
		$content = ob_get_clean();
	}

	$content = preg_replace("/%DOLLARSIGN%/", '$', $content);

	return $content;
}


/**
 * Generates the email content for a particular template - either HTML or text.
 *
 * @param integer $form_id The unique form ID.
 * @param integer $submission_id The unique submission ID.
 * @param array $user_info a hash of information about the user this email is being
 *             sent to, containing: user_id, css and logo.
 * @return array A hash of placeholders and their replacement values (e.g. $arr["%FORMURL%"] => 17)
 */
function _ft_get_placeholder_hash($form_id, $submission_id, $user_info = "")
{
	global $g_root_url;

	$placeholder_hash = array();

	$settings = ft_get_settings($form_id);
	$form_info = ft_get_form($form_id);
	$template_info = ft_get_form_template($form_id);
	$submission_info = ft_get_submission($form_id, $submission_id);


	// add the %ANSWER%, %FILEURL% and %FILENAME% placeholders
	$count = 1;
	$submission_date = "";
	while ($info = mysql_fetch_assoc($template_info))
	{
		$column_name = $info['col_name'];
		$field_type  = $info['field_type'];

		// now loop through the submission hash and store the content for each %ANSWER%
		foreach ($submission_info as $row)
		{
			if ($row['col_name'] == $column_name)
			{
				if ($field_type == "file")
				{
					$placeholder_hash["/%FILENAME{$count}%/"] = $row["content"];
					$placeholder_hash["/%FILEURL{$count}%/"] = "{$row['file_upload_url']}/{$row['content']}";
				}
				else
					$placeholder_hash["/%ANSWER{$count}%/"] = $row["content"];

				if ($row['col_name'] == "submission_date")
				{
					$placeholder_hash["/%ANSWER{$count}%/"] = ft_get_date($settings['timezone_offset'], $row["content"], $settings['date_format']);
					$submission_date = $row['content'];
				}
				break;
			}
		}

		$count++;
	}
	mysql_data_seek($template_info, 0);


	$count = 1;
	while ($info = mysql_fetch_assoc($template_info))
	{
		$placeholder_hash["/%QUESTION{$count}%/"] = $info["field_title"];
		$count++;
	}

	$placeholder_hash['/%ADMINEMAIL%/'] = $settings['administrator_email'];
	$placeholder_hash['/%FORMNAME%/'] = $form_info['form_name'];
	$placeholder_hash['/%FORMURL%/'] = $form_info["form_url"];
	$placeholder_hash['/%SUBMISSIONDATE%/'] = ft_get_date($settings['timezone_offset'], $submission_date, $settings['date_format']);
	$placeholder_hash['/%SUBMISSIONID%/'] = $submission_id;

	if (!empty($user_info))
	{
		$placeholder_hash['/%LOGO%/'] = "$g_root_url/images/logos/{$user_info['logo']}";
		$placeholder_hash['/%CLIENTCSS%/'] = $user_info['css'];
		$placeholder_hash['/%LOGINURL%/'] = $g_root_url . "/index.php?id={$user_info['user_id']}";
	}
	else
		$placeholder_hash['/%LOGINURL%/'] = $g_root_url . "/index.php";

	return $placeholder_hash;
}


/**
 * Helper function to return the newline character for emails, appropriate for this user's system.
 *
 * @return string A string of the appropriate end-of-line character.
 */
function _ft_get_email_eol_char()
{
	$eol = "\n";
	if (strtoupper(substr(PHP_OS, 0, 3) == 'WIN'))
		$eol = "\r\n";
	else if (strtoupper(substr(PHP_OS, 0, 3) == 'MAC'))
		$eol = "\r";

	return $eol;
}


/**
 * Determines the first name, last name and email address used the email headers based on the
 * form and submission info.
 *
 * @param array $form_info A hash containing all the form information.
 * @param array $submission_info A hash containing all the submission information.
 * @return array An array: [0] submission first name, [1]: submission last name, [2]: submission
 *             email address.
 */
function _ft_get_submission_email_headers($form_info, $submission_info)
{
	// retrieve the user's name and email address from the form submission
	$user_first_name_field = $form_info['user_first_name_field'];
	$user_last_name_field  = $form_info['user_last_name_field'];
	$user_email_field      = $form_info['user_email_field'];

	$submission_first_name = "";
	$submission_last_name  = "";
	$submission_email      = "";

	foreach ($submission_info as $row)
	{
		// first name
		if      (!empty($user_first_name_field))
		{
			if ($row['col_name'] == $user_first_name_field)
				$submission_first_name = trim($row['content']);
		}
		// last name
		if (!empty($user_last_name_field))
		{
			if ($row['col_name'] == $user_last_name_field)
				$submission_last_name = trim($row['content']);
		}
		// email
		if ($row['col_name'] == $user_email_field)
			$submission_email = trim($row['content']);
	}

	return array($submission_first_name, $submission_last_name, $submission_email);
}


/**
 * Determines the "From" field for a particular email, based on the form configuration.
 *
 * @param array $form_info A hash containing all the form information.
 * @param string $submission_first_name The first name of the person submitting the form (may be empty)
 * @param string $submission_last_name The last name of the person submitting the form (may be empty)
 * @param string $submission_email The email address of the person submitting the form (may be empty)
 * @param array $settings The program settings hash (from {@link http://www.formtools.org/developerdoc/1-4-6/General/_code---general.php.html#functionget_settings get_settings})
 * @return string The string "from" field, WITHOUT the "From: " prefix or eol character.
 */
function _ft_get_submission_from_field($form_info, $submission_first_name, $submission_last_name,
																		$submission_email, $email_type, $settings)
{
	$from = "";

	switch ($form_info["{$email_type}_email_from"])
	{
		case "admin_email":
			$from = "{$settings['program_name']} <{$settings['administrator_email']}>";
			break;

		case "users_name":
			if (!empty($submission_first_name))
				$from = $submission_first_name;

			if (!empty($submission_last_name))
				$from .= " $submission_last_name";

			if (!empty($from))
				$from .= " <$submission_email>";
			else
				$from = $submission_email;
			break;

		case "custom":
			$from = $form_info["{$email_type}_email_from_custom"];
			break;
	}

	return $from;
}


/**
 * Determines the "Reply-to" field for a particular email, based on the form configuration.
 *
 * @param array $form_info A hash containing all the form information.
 * @param string $submission_first_name The first name of the person submitting the form (may be empty)
 * @param string $submission_last_name The last name of the person submitting the form (may be empty)
 * @param string $submission_email The email address of the person submitting the form (may be empty)
 * @param array $settings The program settings hash (from {@link http://www.formtools.org/developerdoc/1-4-6/General/_code---general.php.html#functionget_settings get_settings})
 * @return string The string "reply-to" field, WITHOUT the "Reply-to: " prefix or eol character.
 */
function _ft_get_submission_reply_to_field($form_info, $submission_first_name, $submission_last_name,
																		$submission_email, $email_type, $settings)
{
	$reply_to = "";
	switch ($form_info["{$email_type}_email_reply_to"])
	{
		case "admin_email":
			$reply_to = "{$settings['program_name']} <{$settings['administrator_email']}>";
			break;

		case "users_name":
			if (!empty($submission_first_name))
				$reply_to = $submission_first_name;

			if (!empty($submission_last_name))
				$reply_to .= " $submission_last_name";

			if (!empty($reply_to))
				$reply_to .= " <" . $submission_email . ">";
			else
				$reply_to = $submission_email;
			break;

		case "custom":
			$reply_to = $form_info["{$email_type}_email_reply_to_custom"];
			break;
	}

	return $reply_to;
}

