<?php
class DynResize
{
	private $cfg = array();
	private $colors = array();
	private $size = array();
	private $ratio = array();
	private $style;
	private $type;
	private $img;
	
	public function __construct()
	{
		$this->loadStyles();
		$this->parseSourcePath();
		$this->parseConfig();
		$this->getSourceSize();
		$this->getOutputSize();
		if(DYNRESIZE_CACHE) $this->checkCache(); // Check whether the image has not been already generated. If it has, read the cached image.
		$this->loadSource();
		$this->loadOutput();
		$this->parseColors();
		$this->getResizedSize();
		$this->resize();
		$this->output();
	}
	
	private function checkCache()
	{
		if(!defined('DYNRESIZE_ERROR'))
		{
			$this->cfg['cache'] = $this->cfg['path'] . '.' . $this->size['output'][0] . 'x' . $this->size['output'][1] . '.jpg';
			
			if(file_exists($this->cfg['cache']))
			{
				header('content-type: image/jpeg');
				readfile($this->cfg['cache']);
				imagedestroy($this->img['source']);
				die();
			}
		}
	}
	
	private function parseSourcePath()
	{
		if(!defined('DYNRESIZE_ERROR'))
		{
			// Source path
			if(isset($_GET['path']))
			{
				preg_match('/^http:\/\/([^\/]+)\/(.*?)$/',$_GET['path'],$matches);
				if(isset($matches[1]) && $matches[1][0] != $_SERVER['SERVER_NAME'] && !DYNRESIZE_EXTERNAL) define('DYNRESIZE_ERROR',true);
				$this->cfg['path'] = DYNRESIZE_ROOT . $_GET['path'];
			}
			else define('DYNRESIZE_ERROR',true);
		}
	}
	
	private function parseConfig()
	{
		// Output style
		$this->cfg['style'] = DYNRESIZE_STYLE;
		
		// Output type
		if(isset($_GET['type'])) $this->cfg['type'] = $_GET['type'];
				
		// Error lines
		if(isset($_GET['lines'])) $this->cfg['lines'] = (boolean) $_GET['lines'];
		else $this->cfg['lines'] = DYNRESIZE_ERROR_LINES;
		
		// Error lines radius
		if(isset($_GET['radius'])) $this->cfg['radius'] = (int) $_GET['radius'];
		elseif(!defined('DYNRESIZE_ERROR_RADIUS')) $this->cfg['radius'] = 0;
		else $this->cfg['radius'] = DYNRESIZE_ERROR_RADIUS;
		
		// Output quality
		if(isset($_GET['quality'])) $this->cfg['quality'] = (int) $_GET['quality'];
		elseif(defined('DYNRESIZE_QUALITY')) $this->cfg['quality'] = DYNRESIZE_QUALITY;
		elseif(isset($this->type->quality[0])) $this->cfg['quality'] = (int) $this->type->quality[0];
		else $this->cfg['quality'] = 60;
	}
	
	private function parseColors()
	{
		$this->colors['bg'] = sscanf(DYNRESIZE_BGCOLOR,'#%2x%2x%2x');
		$this->colors['bg'] = imagecolorallocate($this->img['output'],$this->colors['bg'][0],$this->colors['bg'][1],$this->colors['bg'][2]);
		
		// Error colors
		$this->colors['error_bg'] = sscanf(DYNRESIZE_ERROR_BGCOLOR,'#%2x%2x%2x');
		$this->colors['error_bg'] = imagecolorallocate($this->img['output'],$this->colors['error_bg'][0],$this->colors['error_bg'][1],$this->colors['error_bg'][2]);
		$this->colors['error_lines'] = sscanf(DYNRESIZE_ERROR_LINECOLOR,'#%2x%2x%2x');
		$this->colors['error_lines'] = imagecolorallocate($this->img['output'],$this->colors['error_lines'][0],$this->colors['error_lines'][1],$this->colors['error_lines'][2]);
	}
	
	private function loadStyles()
	{
		if(DYNRESIZE_STYLE_SOURCE == 1)
		{
			if(file_exists(DYNRESIZE_ROOT . 'dynresize-styles.xml'))
			{
				$xml = simplexml_load_file(DYNRESIZE_ROOT . 'dynresize-styles.xml');
				
				$this->style = $xml->xpath('style[@codename="' . DYNRESIZE_STYLE . '"]');
				$this->style = $this->style[0];
				$this->type = $this->style->xpath('types/type[@codename="' . $_GET['type'] . '"]');
				$this->type = $this->type[0];
			}
		}
	}
	
	private function loadSource()
	{
		if(isset($_GET['path']))
		{
			if(!$this->loadImage('source',$this->cfg['path']) && !defined('DYNRESIZE_ERROR')) define('DYNRESIZE_ERROR',true);
		}
	}
	
	private function loadImage($codename,$path)
	{
		if(file_exists($path))
		{
			$ext = end(explode('.',$path));
			switch($ext)
			{
				case('jpg'):case('jpeg'): $this->img[$codename] = imagecreatefromjpeg($path); break;
				case('png'): $this->img[$codename] = imagecreatefrompng($path); break;
				case('gif'): $this->img[$codename] = imagecreatefromgif($path); break;
				default: if(!defined('DYNRESIZE_ERROR')) define('DYNRESIZE_ERROR',true); break;
			}
			
			$this->size[$codename] = getimagesize($path);
			return($this->img[$codename]);
		}
		else return(false);
	}
	
	private function loadOutput()
	{
		$this->img['output'] = imagecreatetruecolor($this->size['output'][0],$this->size['output'][1]);
	}
	
	private function getSourceSize()
	{
		if(!defined('DYNRESIZE_ERROR') && isset($_GET['path']))
		{
			$this->size['source'] = getimagesize($this->cfg['path']);
			$this->ratio['source'] = $this->size['source'][0] / $this->size['source'][1];
		}
	}
	
	private function getOutputSize()
	{
		switch(DYNRESIZE_MODE)
		{
			case(1):
				if($this->type != NULL)
				{
					if(!isset($this->type->width[0]) && !isset($this->type->height[0]) && !defined('DYNRESIZE_ERROR')) define('DYNRESIZE_ERROR',true);
					elseif(isset($this->type->width[0],$this->type->height[0]))
					{
						$this->size['output'][0] = (int) $this->type->width[0];
						$this->size['output'][1] = (int) $this->type->height[0];
					}
					elseif(isset($this->type->width[0]))
					{
						$this->size['output'][0] = (int) $this->type->width[0];
						$this->size['output'][1] = $this->getOutputHeight($this->size['output'][0]);
						if($this->size['output'][1] > $this->size['source'][1]) $this->size['output'][1] = $this->size['source'][1];
					}
					else
					{
						$this->size['output'][1] = (int) $this->type->height[0];
						$this->size['output'][0] = $this->getOutputHeight($this->size['output'][1]);
						if($this->size['output'][0] > $this->size['source'][0]) $this->size['output'][0] = $this->size['source'][0];
					}
				}
				else
				{
					if(!defined('DYNRESIZE_ERROR')) define('DYNRESIZE_ERROR',true);
					$this->size['output'][0] = 256;
					$this->size['output'][1] = 160;
				}
				break;
			case(2):
				$this->size['output'][0] = (int) $_GET['w'];
				$this->size['output'][1] = (int) $_GET['h'];
				break;
			case(3):
				$this->size['output'][0] = (int) $_GET['w'];
				$this->size['output'][1] = $this->getOutputHeight($this->size['output'][0]);
				if($this->size['output'][1] > $this->size['source'][1]) $this->size['output'][1] = $this->size['source'][1];
				break;
			case(4):
				$this->size['output'][1] = (int) $_GET['h'];
				$this->size['output'][0] = $this->getOutputWidth($this->size['output'][1]);
				if($this->size['output'][0] > $this->size['source'][0]) $this->size['output'][0] = $this->size['source'][0];
				break;
			default:
				if(!defined('DYNRESIZE_ERROR')) define('DYNRESIZE_ERROR',true);
				$this->size['output'][0] = 256;
				$this->size['output'][1] = 160;
				break;
		}
		
		$this->ratio['output'] = $this->size['output'][0] / $this->size['output'][1];
	}
	
	private function getOutputWidth($height)
	{
		return(floor(($this->size['source'][0] * $height) / $this->size['source'][1]));
	}
	private function getOutputHeight($width)
	{
		return(floor(($this->size['source'][1] * $width) / $this->size['source'][0]));
	}
	
	private function getResizedSize()
	{
		if(!defined('DYNRESIZE_ERROR'))
		{
			if($this->ratio['source'] / $this->ratio['output'] == 1)
			{
				$this->size['resized'] = array($this->size['output'][0],$this->size['output'][1]);
				$this->size['resized'][2] = array(0,0);
			}
			else
			{
				if(!DYNRESIZE_CROP) // There will be some stripes on edges.
				{
					if($this->ratio['source'] / $this->ratio['output'] < 1)
					{
						$this->size['resized'] = array($this->getOutputWidth($this->size['output'][1]),$this->size['output'][1]);
						$this->size['resized'][2] = array(($this->size['output'][0] - $this->size['resized'][0]) / 2,0);
					}
					else
					{
						$this->size['resized'] = array($this->size['output'][0],$this->getOutputHeight($this->size['output'][0]));
						$this->size['resized'][2] = array(0,($this->size['output'][1] - $this->size['resized'][1]) / 2);
					}
				}
				else
				{
					if($this->ratio['source'] / $this->ratio['output'] > 1)
					{
						$this->size['resized'] = array($this->getOutputWidth($this->size['output'][1]),$this->size['output'][1]);
						$this->size['resized'][2] = array(($this->getOutputHeight($this->size['output'][0]) - $this->size['output'][1]) / 2,0);
						
						// Check whether the source height is not smaller than the resized source height.
						if($this->size['source'][1] < $this->size['resized'][1])
						{
							$this->size['resized'] = array($this->size['source'][0],$this->size['source'][1]);
							$this->size['resized'][2] = array(($this->size['output'][0] - $this->size['source'][0]) / 2,($this->size['output'][1] - $this->size['source'][1]) / 2);
						}
					}
					else
					{
						$this->size['resized'] = array($this->size['output'][0],$this->getOutputHeight($this->size['output'][0]));
						$this->size['resized'][2] = array(0,($this->getOutputWidth($this->size['output'][1]) - $this->size['output'][0]) / 2);
						
						// Check whether the source width is not smaller than the resized source width.
						if($this->size['source'][0] < $this->size['resized'][0])
						{
							$this->size['resized'] = array($this->size['source'][0],$this->size['source'][1]);
							$this->size['resized'][2] = array(($this->size['output'][0] - $this->size['source'][0]) / 2,($this->size['output'][1] - $this->size['source'][1]) / 2);
						}
					}
				}
			}
		}
	}
	
	private function resize()
	{
		if(!defined('DYNRESIZE_ERROR'))
		{
			imagefill($this->img['output'],0,0,$this->colors['bg']);
			
			if($this->size['output'][0] > $this->size['source'][0] && $this->size['output'][1] > $this->size['source'][1])
			{
				imagecopy(
					$this->img['output'],$this->img['source'],
					floor(($this->size['output'][0] - $this->size['source'][0]) / 2),floor(($this->size['output'][1] - $this->size['source'][1]) / 2),
					0,0,
					$this->size['source'][0],$this->size['source'][1]
				);
			}
			else
			{
				imagecopyresampled(
					$this->img['output'],$this->img['source'],
					$this->size['resized'][2][0],$this->size['resized'][2][1],
					0,0,
					$this->size['resized'][0],$this->size['resized'][1],
					$this->size['source'][0],$this->size['source'][1]
				);
			}
		}
		else
		{
			imagefill($this->img['output'],0,0,$this->colors['error_bg']);
			if($this->cfg['lines'])
			{
				imageantialias($this->img['output'],true);
				imageline(
					$this->img['output'],
					$this->cfg['radius'],$this->cfg['radius'],
					$this->size['output'][0]-$this->cfg['radius'],$this->size['output'][1]-$this->cfg['radius'],
					$this->colors['error_lines']
				);
				imageline(
					$this->img['output'],
					$this->size['output'][0]-$this->cfg['radius'],$this->cfg['radius'],
					$this->cfg['radius'],$this->size['output'][1]-$this->cfg['radius'],
					$this->colors['error_lines']
				);
				imageantialias($this->img['output'],false);
			}
			
			if(defined('DYNRESIZE_ERROR_IMAGE') && file_exists(DYNRESIZE_ERROR_IMAGE))
			{
				if($this->loadImage('error',DYNRESIZE_ERROR_IMAGE))
				{
					imagecopy(
						$this->img['output'],$this->img['error'],
						floor(($this->size['output'][0] - $this->size['error'][0]) / 2),floor(($this->size['output'][1] - $this->size['error'][1]) / 2),
						0,0,
						$this->size['error'][0],$this->size['error'][1]
					);
				}
			}
		}
	}
	
	private function output()
	{
		if(!defined('DYNRESIZE_ERROR'))
		{
			header('content-type: image/jpeg');
			imagejpeg($this->img['output'],NULL,$this->cfg['quality']);
			if(DYNRESIZE_CACHE) imagejpeg($this->img['output'],$this->cfg['cache'],$this->cfg['quality']);
			imagedestroy($this->img['output']);
			imagedestroy($this->img['source']);
		}
		else
		{
			header('content-type: image/png');
			imagepng($this->img['output']);
			imagedestroy($this->img['output']);
			imagedestroy($this->img['error']);
		}
	}
}
?>